/******************************** CPPFile *****************************

* FileName [PredAbsRefiner.cpp]

* PackageName [main]

* Synopsis [Method definitions of PredAbsRefiner class.]

* SeeAlso [PredAbsRefiner.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <gmp.h>
#include <string>
#include <vector>
#include <set>
#include <map>
#include <list>
#include <stack>
using namespace std;

#include "BigInt.h"
#include "Util.h"
#include "Statistics.h"
#include "Node.h"
#include "Action.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "ContLoc.h"
#include "ImplState.h"
#include "LocalAbsLts.h"
#include "GlobalCEDag.h"
#include "ModelExtractor.h"
#include "ConcCEDag.h"
#include "Component.h"
#include "ProcManager.h"
#include "PredAbsRefiner.h"
#include "CEDagVerifier.h"
#include "LtsInfo.h"
using namespace magic;

/*********************************************************************/
//methods for class PredIter
/*********************************************************************/
PredIter::PredIter(const set<ContLoc*> &a,size_t s)
{
  all.insert(all.end(),a.begin(),a.end());
  for(size_t i = 0;i < s;++i) indices.push_back(i);
}

const PredIter &PredIter::operator = (const PredIter &rhs)
{
  all = rhs.all;
  indices = rhs.indices;
  return *this;
}

bool PredIter::End() const
{
  return (indices[0] >= all.size());
}

set<ContLoc*> PredIter::Next()
{
  size_t allSize = all.size();
  size_t idSize = indices.size();
  
  //construct the result
  set<ContLoc*> res;
  for(size_t i = 0;i < idSize;++i) {
    res.insert(all[indices[i]]);
  }

  //get the first index to be incremented
  int firstId = idSize - 1;
  for(;firstId >= 0;--firstId) {
    if(indices[firstId] < (allSize - idSize + firstId)) {
      break;
    }
  }

  //if this was the last valuation
  if(firstId == -1) {
    indices[0] = allSize;
  }
  //increment that id and reset the remaining
  else {
    size_t start = indices[firstId] + 1;
    for(size_t i = firstId;i < idSize;++i,++start) {
      indices[i] = start;
    }
  }

  //all done
  return res;
}

/*********************************************************************/
//static field definitions
/*********************************************************************/
const int PredAbsRefiner::REFINED_NEW_PREDS = 11630;
const int PredAbsRefiner::REFINED_NO_NEW_PREDS = 11640;

/*********************************************************************/
//refine the predicate abstraction by inferring new predicates
/*********************************************************************/
int PredAbsRefiner::Run()
{
  newPreds = false;
  //if we are optimizing predicates we need to recompute the optimal
  //predicates and seed the control locations
  if(Database::CEGAR_TYPE == Database::CEGAR_OPTIMIZE_PRED) {
    UpdateConstraintFile();
    SolveOptimization();	    
  }
  //if we should add useful predicates
  else if(Database::CEGAR_TYPE == Database::CEGAR_USEFUL_PRED) {
    UpdateSeedBranchesUseful();
  }
  //if we should add predicates that eliminate new CEs
  else if(Database::CEGAR_TYPE == Database::CEGAR_ELIMINATE_PRED) {
    UpdateSeedBranchesEliminate(true);
  }
  //if we should be greedy
  else if(Database::CEGAR_TYPE == Database::CEGAR_GREEDY_PRED) {
    UpdateSeedBranchesGreedy();
  }
  //this should not happen
  else assert(false);

  //clear the set of paths to be refined on
  procManager.dagToLocs.clear();

  return newPreds ? REFINED_NEW_PREDS : REFINED_NO_NEW_PREDS;
}

/*********************************************************************/
//update the seed branches based on the maps created during the
//validity checking of the counter examples
/*********************************************************************/
void PredAbsRefiner::UpdateSeedBranchesUseful()
{
  set<ContLoc*> newSeeds = procManager.seedBranches;
  int predsAdded = 0;
  for(map< ConcCEDag,set<ContLoc*> >::const_iterator i = procManager.dagToLocs.begin();(predsAdded <= Database::MAX_ADD_PRED) && (i != procManager.dagToLocs.end());++i) {
    if((Database::CONF_TYPE == Database::CONF_REACH) || (Database::CONF_TYPE == Database::CONF_LTL) ||
       (Database::CONF_TYPE == Database::CONF_DLOCK)) {
      i->first.Display(procManager);
    }
    for(set<ContLoc*>::const_iterator j = procManager.branchLocs.begin();(predsAdded <= Database::MAX_ADD_PRED) && (j != procManager.branchLocs.end());++j) {
      ContLoc *loc = (*j);
      bool locUseful = (!Database::PREDS_FROM_SPEC) || (loc->IsUseful());
      if(locUseful) {
	if(i->second.count(loc) != 0) {
	  if(newSeeds.insert(loc).second) {
	    newPreds = true;
	    ++predsAdded;
	  }
	}
      }
    }
  }
  for(set<ContLoc*>::const_iterator i = newSeeds.begin();i != newSeeds.end();++i) {
    Util::Message(2,"Seeding with %s\n",(*i)->ToString().c_str());
  }
  procManager.SetSeedBranches(newSeeds);
}

/*********************************************************************/
//update the seed branches based on the maps created during the
//validity checking of the counter examples. the argument indicates
//whether the set of new seeds should be displayed.
/*********************************************************************/
void PredAbsRefiner::UpdateSeedBranchesEliminate(bool message)
{
  set<ContLoc*> newSeeds = procManager.seedBranches;
  int oldSeedNum = newSeeds.size();
  int predsAdded = 0;
  bool someElim = false;
  for(map< ConcCEDag,set<ContLoc*> >::const_iterator i = procManager.dagToLocs.begin();(predsAdded <= Database::MAX_ADD_PRED) && (i != procManager.dagToLocs.end());++i) {
    if((Database::CONF_TYPE == Database::CONF_REACH) || (Database::CONF_TYPE == Database::CONF_LTL) ||
       (Database::CONF_TYPE == Database::CONF_DLOCK)) {
      i->first.Display(procManager);
    }
    //ignore this path if it can be eliminated by an empty set of
    //predicates
    if(CanEliminate(set<ContLoc*>(),i->first)) continue;
    //try subsets in increasing order of size
    size_t subsetNum = 0;
    bool flag = false;
    set< set <ContLoc*> > removed;
    for(size_t l = 0;(subsetNum < Database::SUBSET_CUTOFF) && 
	  (!flag || !Database::STOP_OPT_AFTER_ELIM || (l < Database::SMALLEST_SUBSET)) &&
	  (predsAdded <= Database::MAX_ADD_PRED) && (l < Database::LARGEST_SUBSET) && 
	  (l < i->second.size());++l) {
      //get the subsets of predicates that eliminate this path
      PredIter pIter(i->second,l+1);
      while((predsAdded <= Database::MAX_ADD_PRED) && (subsetNum < Database::SUBSET_CUTOFF) && (!pIter.End())) {
	set<ContLoc*> locSub = pIter.Next();
	if (!IsSuperSetOfAny(locSub,removed)) {
	  ++subsetNum;
	  if(CanEliminate(locSub,i->first)) {
	    newSeeds.insert(locSub.begin(),locSub.end());
	    removed.insert(locSub);
	    predsAdded = newSeeds.size() - oldSeedNum;
	    newPreds = (predsAdded > 0) ? true : newPreds;
	    flag = true;
	  }
	}
      }
    }    
    //check if this path eliminated by at least one predicate
    someElim = flag ? true : someElim;
  }
  if(!someElim) {
    Util::Error("ERROR: No CE eliminated by any predicate set ...\n"
		"HINT1: generate more counterexamples: use --ceDag %d\n"
		"HINT2: unwind loops further: use --predLoop %d\n"
		"HINT3: try more subsets: use --subsetCutoff %d\n",
		Database::MAX_GLOBAL_CE + 1,Database::MAX_PRED_INFER_LOOP + 1,
		Database::SUBSET_CUTOFF + 5);
  }
  if(message) {
    for(set<ContLoc*>::const_iterator i = newSeeds.begin();i != newSeeds.end();++i) {
      Util::Message(2,"Seeding with %s\n",(*i)->ToString().c_str());
    }
  }
  procManager.SetSeedBranches(newSeeds);
}

/*********************************************************************/
//update the set of predicates by greedily trying to minimize the
//number of predicates. i.e. first add a predicate. the try to remove
//as many predicates as possible.
/*********************************************************************/
void PredAbsRefiner::UpdateSeedBranchesGreedy()
{
  UpdateSeedBranchesEliminate(false);
  for(map< ConcCEDag,set<ContLoc*> >::const_iterator i = procManager.dagToLocs.begin();i != procManager.dagToLocs.end();++i) {    
    //check for duplicate paths
    if(!procManager.optDags.insert(i->first).second) {
      Util::Error ("ABORTING ... Trace already seen ...\n");
    }
  }
  bool reduced = false;
  set<ContLoc*> currSeeds = procManager.seedBranches;
  do {
    reduced = false;
    //create a random permutation of the seed branches
    vector<ContLoc*> perm;
    set<ContLoc*> remaining = currSeeds;
    while(!remaining.empty()) {
#ifdef WIN32
      int randInt = static_cast<int>((rand() * 1.0 / RAND_MAX) * remaining.size());
#else
      int randInt = static_cast<int>((random() * 1.0 / RAND_MAX) * remaining.size());
#endif //WIN32
      int count = 0;
      bool picked = false;
      for(set<ContLoc*>::const_iterator i = remaining.begin();i != remaining.end();++i,++count) {
	if(count == randInt) {
	  perm.push_back(*i);
	  remaining.erase(i);
	  picked = true;
	  break;
	}
      }
      assert(picked);
    }
    assert(perm.size() == currSeeds.size());
    //try to eliminate one at a time
    for(vector<ContLoc*>::const_iterator i = perm.begin();i != perm.end();++i) {
      set<ContLoc*> newSeeds = currSeeds;
      newSeeds.erase(*i);
      procManager.SetSeedBranches(newSeeds);
      ModelExtractor::Run(true);
      bool allEliminated = true;
      for(set<ConcCEDag>::const_iterator j = procManager.optDags.begin();j != procManager.optDags.end();++j) {
	j->Display(procManager);
	if(!CanEliminate(*j)) {
	  allEliminated = false;
	  break;
	} 
      }
      if(allEliminated) {
	reduced = true;
	currSeeds = newSeeds;
	break;
      }
    }
  } while(reduced);
  for(set<ContLoc*>::const_iterator i = currSeeds.begin();i != currSeeds.end();++i) {
    Util::Message(2,"Seeding with %s\n",(*i)->ToString().c_str());
  }
  Util::Message(2,"***************************************************\n");
  procManager.SetSeedBranches(currSeeds);
}

/*********************************************************************/
//checks if a set of locations is a superset of any constraints
/*********************************************************************/
bool PredAbsRefiner::IsSuperSetOfAny(const set<ContLoc*> &locSub,const set<set <ContLoc*> > &constraints)
{
  for(set<set <ContLoc*> >::const_iterator i = constraints.begin();i != constraints.end();++i) {
    bool subset = true;
    for (set <ContLoc*>::const_iterator j = i->begin();j != i->end ();++j) {
      if (locSub.count(*j) == 0) {
	subset = false;
	break;
      }
    }
    if (subset) {
      return true;
    }
  }
  return false;
}

/*********************************************************************/
//updates the constraint file to reflect values for the current path
/*********************************************************************/
void PredAbsRefiner::UpdateConstraintFile()
{
  Util::Message(4,"Calling UpdateConstraintFile ...\n");
  bool someElim = false;
  for(map< ConcCEDag,set<ContLoc*> >::const_iterator i = procManager.dagToLocs.begin();i != procManager.dagToLocs.end();++i) {
    if((Database::CONF_TYPE == Database::CONF_REACH) || (Database::CONF_TYPE == Database::CONF_LTL) ||
       (Database::CONF_TYPE == Database::CONF_DLOCK)) {
      i->first.Display(procManager);
    }    
    /*
    //check for duplicate paths
    if(!procManager.optDags.insert(i->first).second) {
      Util::Error ("ABORTING ... Trace already seen ...\n");
    }
    */
    //ignore this path if it can be eliminated by an empty set of
    //predicates
    if(CanEliminate(set<ContLoc*>(),i->first)) continue;
    //try subsets in increasing order of size
    size_t subsetNum = 0;
    size_t consNum = 0;    
    set< set<ContLoc*> > cnf,dnf;
    bool redundant = false;
    for(size_t l = 0;(!redundant) && (subsetNum < Database::SUBSET_CUTOFF) && 
	  (dnf.empty() || !Database::STOP_OPT_AFTER_ELIM || (l < Database::SMALLEST_SUBSET)) &&
	  (consNum < Database::CONSTRAINT_CUTOFF) && (l < Database::LARGEST_SUBSET) && 
	  (l < i->second.size());++l) {
      //get the subsets of predicates that eliminate this path
      Statistics::maxTriedCombSize = ((l+1) > Statistics::maxTriedCombSize) ? (l+1) : Statistics::maxTriedCombSize;
      PredIter pIter(i->second,l+1);
      while((!redundant) && (subsetNum < Database::SUBSET_CUTOFF) &&
	    (consNum < Database::CONSTRAINT_CUTOFF) && (!pIter.End())) {
	set<ContLoc*> locSub = pIter.Next();
	if (!IsSuperSetOfAny(locSub,dnf)) {
	  ++subsetNum;
	  if(CanEliminate(locSub,i->first)) {
	    Util::Message(4,"ELIMINATED ...\n");
	    dnf.insert(locSub);
	    UpdateCNF(cnf,locSub);
	    ++consNum;
	    ++Statistics::elimCombNum;
	    Statistics::maxElimCombSize = (locSub.size() > Statistics::maxElimCombSize) ? locSub.size() : Statistics::maxElimCombSize;
	    EliminateRedundant(cnf);
	    if(cnf.empty()) {
	      redundant = true;
	      break;
	    }
	  }
	}
      }
    }    
    //check if this path eliminated by at least one predicate
    someElim = dnf.empty() ? someElim : true;
    //create the constraints and add them to the file
    Statistics::maxElimCombNum = (consNum > Statistics::maxElimCombNum) ? consNum : Statistics::maxElimCombNum;
    //add the constraints to file
    UpdateConstraints(cnf);
  }
  if(!someElim) {
    Util::Error("ERROR: No CE eliminated by any predicate set ...\n"
		"HINT1: generate more counterexamples: use --ceDag %d\n"
		"HINT2: unwind loops further: use --predLoop %d\n"
		"HINT3: try more subsets: use --subsetCutoff %d\n",
		Database::MAX_GLOBAL_CE + 1,Database::MAX_PRED_INFER_LOOP + 1,
		Database::SUBSET_CUTOFF + 5);
  }
  WriteConstraints();
}

/*********************************************************************/
//given CNF formula old and conjunctive clause new return CNF for, for
//(old OR new)
/*********************************************************************/
void PredAbsRefiner::UpdateCNF(set< set<ContLoc*> > &cnf,const set<ContLoc*> &clause) const
{
  if(cnf.empty()) {
    for(set<ContLoc*>::const_iterator i = clause.begin();i != clause.end();++i) {
      set<ContLoc*> x;
      x.insert(*i);
      cnf.insert(x);
    }
  } else {
    set< set<ContLoc*> > temp = cnf;
    cnf.clear();
    for(set< set<ContLoc*> >::const_iterator i = temp.begin();i != temp.end();++i) {
      for(set<ContLoc*>::const_iterator j = clause.begin();j != clause.end();++j) {
	set<ContLoc*> x = *i;
	x.insert(*j);
	cnf.insert(x);
      }
    }
  }
}

/*********************************************************************/
//remove clauses that are implied by already existing clauses
/*********************************************************************/
void PredAbsRefiner::EliminateRedundant(set< set<ContLoc*> > &arg) const
{
  set< set<ContLoc*> > res;
  //return false;
  for(set< set<ContLoc*> >::const_iterator i = arg.begin();i != arg.end();++i) {
    bool implied = false;
    for(set< set<ContLoc*> >::const_iterator j = procManager.allCons.begin();j != procManager.allCons.end();++j) {
      set<ContLoc*> x = *i;
      x.insert(j->begin(),j->end());
      if(x == (*i)) {
	implied = true;
	break;
      }
    }
    if(!implied) res.insert(*i);
  }
  arg = res;
}

/*********************************************************************/
//splits a set of Exprs into singleton and non-singleton sets
/*********************************************************************/
pair< set< set<ContLoc*> >,set<ContLoc*> > PredAbsRefiner::SplitSet(const list< set<ContLoc*> > &clause)
{
  pair< set< set<ContLoc*> >,set<ContLoc*> > res;
  for(list< set<ContLoc*> >::const_iterator i = clause.begin ();
      i != clause.end ();++i) {
    if ((*i).size () > 1) res.first.insert(*i);
    else {
      ContLoc *j = *(i->begin());
      res.second.insert(j);
    }
  }
  return res;
}

/*********************************************************************/
// based on current optMap, solves the problem of finding smalled set
// of predicates that eliminates all spurious counterexamples.  sets
// predicates in control location accordingly.
/*********************************************************************/
void PredAbsRefiner::SolveOptimization () 
{
  Util::Message(4,"Calling SolveOptimization ...\n");
  if (!Database::CNF_NO_NEW_VARS) {
    assert(false);
    /*
    Util::Message(4,"New variables coming ...\n");
    for (map< ConcPath,set< set<ContLoc*> > >::const_iterator i =
	   Database::optMap.begin (); i != Database::optMap.end ();++i) {
      CNFInsert ((*i).second, varMap, mapVar, disjunctMap, mapDisjunct, varNo,
		 clauses);
    }
    */
  }
  else {
    Util::Message(4,"NO New variables coming ...\n");
  }
  //create the dimacs file
  FILE *pbscnf = fopen(procManager.dimacsFile.c_str(),"w");
  size_t varNo = procManager.varMap.size();
  fprintf(pbscnf,"p cnf %d %d\n",varNo,procManager.clauseNum);
  //Util::Message(1,"p cnf %d %d\n",varNo,procManager.clauseNum);
  Util::Message(4,"p cnf %d %d\n",varNo,procManager.clauseNum);
  fclose (pbscnf);
  string command = string("cat ") + procManager.constraintFile + " >> " + procManager.dimacsFile;
  system(command.c_str());
  //create the pb file
  FILE *pbspb = fopen(procManager.pbFile.c_str(),"w");
  fprintf(pbspb,"# PBType: SE\n");
  fprintf(pbspb,"# PBGoal: %d\n",varNo);
  fprintf(pbspb,"# PBObj : MIN\n");
  fprintf(pbspb,"# NumCoef: %d\n",varNo);
  for (size_t i = 1; i <= varNo; ++i) {
    fprintf(pbspb,"v%d c1\n", i);
  }
  fclose (pbspb);
  //check if environment variable MAGICROOT is defined
  char *envptr = getenv("MAGICROOT");
  if(envptr == NULL) Util::Error("MAGICROOT not set ... Aborting ...\n");
  //removed any old copies of the output file
  remove(procManager.outputFile.c_str());
  //run PBS
#ifdef WIN32
  command = string(envptr) + "\\pbs\\PBS.exe -f " + procManager.dimacsFile + " -D 1 -I -a > " + procManager.outputFile;
#else
  command = string(envptr) + "/pbs/PBS -f " + procManager.dimacsFile + " -D 1 -I -a > " + procManager.outputFile;
#endif //WIN32
  system(command.c_str());
  //read results and get variable assignment
  FILE* results = fopen(procManager.outputFile.c_str(),"r");
  char line[256];
  set <int> assigned;
  bool sat = true;
  while (!feof(results) && sat) {
    fgets(line, 256, results);
    Util::Message(2,line);
    if (strstr(line,"Variable Assignments Satisfying CNF Formula:") != NULL) {
      Util::Message(2,"Reading variable assignments.\n");
      assigned.clear ();
      for(size_t i = 0; i < varNo; ++i) {
	int v;
	fscanf (results,"%i",&v);
	if (v > 0) {
	  assigned.insert(v);
	  Util::Message(2,"%d ********** %s\n",v,procManager.mapVar[v]->ToString().c_str());
	}
      }
    }
    else if (strcmp(line,"Result is     : UNS\n") == 0) {
      Util::Message(2, "UNSATISFIABLE\n");
      sat = false;
    }
    else if (strcmp(line,"Result is     : SAT\n") == 0) {
      Util::Message(2, "Finished (SAT)\n");
      sat = false;
    }
  }
  fclose (results);
  //create new seed branches from variable assignment
  set<ContLoc*> optLocs;
  for (set <int>::const_iterator i = assigned.begin ();
       i != assigned.end ();++i) {
    map <int, ContLoc*>::const_iterator f = procManager.mapVar.find(*i);
    assert(f != procManager.mapVar.end ());
    Util::Message(2,"Seeding with %s\n",f->second->ToString().c_str());
    optLocs.insert(f->second);
  }
  //set seed branches
  procManager.SetSeedBranches(optLocs);
  newPreds = true;
}

/*********************************************************************/
//add a set of constraints to the global set and the file
/*********************************************************************/
void PredAbsRefiner::UpdateConstraints(const set< set<ContLoc*> > &clauses) const
{
  set< set<ContLoc*> > res = clauses;
  for(set< set<ContLoc*> >::const_iterator i = procManager.allCons.begin();i != procManager.allCons.end();++i) {
    bool implied = false;
    for(set< set<ContLoc*> >::const_iterator j = clauses.begin();j != clauses.end();++j) {
      set<ContLoc*> x = *i;
      x.insert(j->begin(),j->end());
      if(x == (*i)) {
	implied = true;
	break;
      }
    }
    if(!implied) res.insert(*i);
  }
  procManager.allCons = res;
}

/*********************************************************************/
//write the constraints to the file
/*********************************************************************/
void PredAbsRefiner::WriteConstraints() const
{  
  FILE *cfptr = fopen(procManager.constraintFile.c_str(),"w");
  for(set< set<ContLoc*> >::const_iterator i = procManager.allCons.begin();i != procManager.allCons.end();++i) {
    for(set<ContLoc*>::const_iterator j = i->begin();j != i->end();++j) {
      ContLoc *loc = *j;
      map<ContLoc*,int>::const_iterator f = procManager.varMap.find(loc);
      if (f != procManager.varMap.end ()) fprintf(cfptr,"%d ",f->second);
      else {
	size_t varNo = procManager.varMap.size() + 1;
	procManager.varMap[loc] = varNo;
	procManager.mapVar[varNo] = loc;
	fprintf(cfptr,"%d ",varNo);
      }
    }
    fprintf(cfptr,"0\n");
  }
  procManager.clauseNum = procManager.allCons.size();
  fclose(cfptr);
}

/*********************************************************************/
//update the predicates on the basis of a spurious counterexample and
//some seed branches
/*********************************************************************/
void PredAbsRefiner::UpdatePreds(const ConcCEDag &concDag,const set<ContLoc*> &seeds)
{
  //if new predicates are to be inferred just from the counterexample
  if(Database::PREDS_FROM_CE) {
    //initialise with necessary predicates
    vector<PredSet> dagToWP(concDag.stateLabel.size());
    for(size_t i = 0;i < concDag.stateLabel.size();++i) {
      ContLoc *loc = (ContLoc*)(static_cast<const CL1AbsState*>(concDag.stateLabel[i])->GetContLoc());
      dagToWP[i].Merge(procManager.necPreds[loc]);
    }
    //if checking simulation of ERROR state reachability
    if((Database::CONF_TYPE == Database::CONF_SIMUL) || (Database::CONF_TYPE == Database::CONF_REACH)) {
      vector<int> topo; concDag.ComputeTopo(topo);
      for(vector<int>::const_iterator i = topo.begin();i != topo.end();++i) {
	for(set<int>::const_iterator j = concDag.trans[*i].begin();j != concDag.trans[*i].end();++j) {	
	  dagToWP[*i].Merge(dagToWP[*j]);
	}
	map< int,pair<Action,Expr> > inlinedAssignMap;
	CEDagVerifier(procManager,false).UpdatePrePost(seeds,dagToWP[*i],*i,concDag,inlinedAssignMap,true);
      }
    }
    //if checking LTL formula
    else if(Database::CONF_TYPE == Database::CONF_LTL) {
      for(int i = 0;i < concDag.loopCount;++i) {
	for(list<size_t>::const_iterator j = concDag.loop.begin();j != concDag.loop.end();++j) {
	  for(set<int>::const_iterator k = concDag.trans[*j].begin();k != concDag.trans[*j].end();++k) {
	    dagToWP[*j].Merge(dagToWP[*k]);
	  }	
	  map< int,pair<Action,Expr> > inlinedAssignMap;
	  CEDagVerifier(procManager,false).UpdatePrePost(seeds,dagToWP[*j],*j,concDag,inlinedAssignMap,true);
	}
      }
      if(concDag.loopCount == -1) {
	for(list<size_t>::const_iterator j = concDag.handle.begin();j != concDag.handle.end();++j) {
	  for(set<int>::const_iterator k = concDag.trans[*j].begin();k != concDag.trans[*j].end();++k) {
	    dagToWP[*j].Merge(dagToWP[*k]);
	  }	
	  map< int,pair<Action,Expr> > inlinedAssignMap;
	  CEDagVerifier(procManager,false).UpdatePrePost(seeds,dagToWP[*j],*j,concDag,inlinedAssignMap,true);
	}
      }
    }
    //impossible
    else assert(false);
    //initialise with necessary predicates
    for(size_t i = 0;i < procManager.allLocs.size();++i) {
      ContLoc *loc = procManager.allLocs[i];
      loc->GetPreds().Merge(procManager.necPreds[loc]);
    }
    //update the predicates
    for(size_t i = 0;i < dagToWP.size();++i) {
      ContLoc *loc = (ContLoc*)(static_cast<const CL1AbsState*>(concDag.stateLabel[i])->GetContLoc());
      loc->GetPreds().Merge(dagToWP[i]);
    }
    newPreds = true;
  }
}

/*********************************************************************/
//returns a CNF-ized set of clauses for the Expr map for a particular
//trace, introducing new variables.  currently uses deeply inefficient
//maps
/*********************************************************************/
void PredAbsRefiner::CNFInsert(const set< set<ContLoc*> > &clause,
			    map <ContLoc*, int> &varMap,
			    map <int, ContLoc*> &mapVar,
			    map <set <ContLoc*> , int> &disjunctMap,
			    map <int, set <ContLoc*> > &mapDisjunct,
			    int &varNo,set <set <int> > &clauses)
{
  /*
  pair <set <set <ContLoc* > >, set <ContLoc*> > split = SplitSet (clause);
  const set <set <ContLoc*> > &multiVar = split.first;
  const set <ContLoc*> &singleVar = split.second;
  set <int> firstClause;
  for (set <ContLoc*>::const_iterator i = singleVar.begin ();
       i != singleVar.end (); ++i) {
    map <ContLoc*, int>::const_iterator f = varMap.find(*i);
    if (f != varMap.end ()) {
      firstClause.insert ((*f).second);
    }
    else {
      varMap[(*i)] = ++varNo;
      mapVar[varNo] = (*i);
      firstClause.insert(varNo);
    }
  }
  for (set <set <ContLoc*> >::const_iterator i = multiVar.begin ();
       i != multiVar.end (); ++i) {
    map <set <ContLoc*>, int>::const_iterator f = disjunctMap.find(*i);
    if (f != disjunctMap.end ()) {
      firstClause.insert ((*f).second);
    }
    else {
      disjunctMap[(*i)] = ++varNo;
      mapDisjunct[varNo] = (*i);
      firstClause.insert(varNo);
      set <int> allClause;
      allClause.insert (varNo);
      for (set <ContLoc*> :: const_iterator j = (*i).begin ();
	   j != (*i).end (); ++j) {
	set <int> thisClause;
	thisClause.insert (-varNo);
	map <ContLoc*, int>::const_iterator f = varMap.find(*j);
	if (f != varMap.end ()) {
	  thisClause.insert ((*f).second);
	  allClause.insert (-(*f).second);
	}
	else {
	  varMap[(*j)] = ++varNo;
	  mapVar[varNo] = (*j);
	  thisClause.insert(varNo);
	  allClause.insert(-varNo);
	}
	clauses.insert(thisClause);
      }
      clauses.insert(allClause);
    }
  }
  clauses.insert (firstClause);
  // DEBUGGING
  Util::Message(4,"Clause was:\n");
  for (set <set <ContLoc*> >::const_iterator i = clause.begin ();
       i != clause.end (); ++i) {
    Util::Message(4,"(");
    for (set <ContLoc*> :: const_iterator j = (*i).begin ();
	 j != (*i).end (); ++j) {
      Util::Message(4,"%i ",varMap[*j]);
    }
    Util::Message(4,") +");
  }
  */
}

/* template <class Key, class Value> static inline void keys(const
hash_map<Key,Value> &m,set<Key> &ks)
{
  typename hash_map<Key,Value>::const_iterator i;
  i = m.begin();

  for (i=m.begin();i!=m.end();++i) {
    ks.insert(i->first);
  }
}
*/

/*********************************************************************/
/*********************************************************************/
ConcCEDag PredAbsRefiner::Concretize(POPath path) 
{
  assert(false);
  return ConcCEDag();

  /*
  ConcCEDag cd;
  cd.PushBack (Tokenizable((path.first).first->GetContLoc ()));
  for (vector <pair <Action, POState> >::const_iterator i = 
	 (path.second).begin (); i != (path.second).end (); ++i) {
    cd.PushBack (Tokenizable((*i).first));
    cd.PushBack (Tokenizable((((*i).second).first)->GetContLoc ()));
  }
  Util::Message(1,"Pre-counterexample found:\n");
  if((Database::CONF_TYPE == Database::CONF_REACH) || (Database::CONF_TYPE == Database::CONF_LTL) ||
    (Database::CONF_TYPE == Database::CONF_DLOCK)) {
    cd.Display ();
  }
  return cd;
  */
}

/*********************************************************************/
/*********************************************************************/
bool PredAbsRefiner::AddsToCoverage (POPath path,map <POState, int> & coverMap) 
{
  if (!Database::COVERAGE)
    return true;
  bool coverageImproved = false;
  POState initial = path.first;
  if (coverMap.count(initial) == 0) {
    coverMap[initial] = 1;
    coverageImproved = true;
  } else {
    ++coverMap[initial];
  }
  for (vector <pair <Action, POState> >::const_iterator i = 
	 (path.second).begin (); i != (path.second).end (); ++i) {
    if (coverMap.count((*i).second) == 0) {
      coverMap[(*i).second] = 1;
      coverageImproved = true;
    } else {
      ++coverMap[(*i).second];
    }
  }
  if (!coverageImproved) {
    Util::Message(1, "Rejecting counterexample for coverage reasons.\n");
  }
  return coverageImproved;
}

/*********************************************************************/
/*********************************************************************/
vector<ConcCEDag> PredAbsRefiner::CFACounterexamplesBFS () 
{
  assert(false);
  return vector<ConcCEDag>();
  /*
  vector <ConcPath> CFAtraces;
  map <POState, int> coverMap;
  const LtsInfo &specInfo = Database::GetLtsInfo(Database::specLtsName);
  string initSpec = specInfo.GetInitState ();
  procManager.ComputeImplStates ();
  set<const ImplState*> initImpls = procManager.initLoc->GetImplStates(procManager.guard);
  set <pair <POState, set <POPath> > > currStates;
  map <POState, int > visitedMap;
  int depth = 0;
  bool done = false;
  for (set<const ImplState*>::const_iterator i = initImpls.begin ();i != initImpls.end (); ++i) {
    //ConcPath cp = ConcPath ();
    //cp.PushBack (Tokenizable((*i)->GetContLoc ()));
    //set <ConcPath> cps;
    //cps.insert(cp);
    vector < pair <Action, POState> > trail;
    set <POPath> cps;
    POState s = POState(*i,initSpec);
    cps.insert (POPath (s, trail));
    if (Database::PREOPTVISIT) {
      visitedMap[s] = 1;
    }
    pair <POState, set <POPath> > state = 
      pair <POState, set <POPath> > (s, cps);
    currStates.insert(state);
  }
  while (!done && (depth < Database::PREOPTMAXDEPTH) && 
	 !(currStates.empty ())) {
    ++depth;
    Util::Message(1,"Depth = %i, frontier = %i, CEs = %i\n", depth, 
		  currStates.size (), CFAtraces.size ());
    set<POState> frontier;
    map<POState, set <POPath> > frontierMap;
    set<pair <POState, set <POPath> > > nextStates;
    for (set<pair <POState, set <POPath> > >::const_iterator i = 
	   currStates.begin(); i != currStates.end (); ++i) {
      pair <POState, set <POPath> > state = *i;
      POState s = (state.first);
      const ImplState* istate = s.first;
      string sstate = s.second;
      set <POPath> cps = state.second;
      if (sstate != Database::ERROR_STATE) {
	list< pair<const ImplState*,Action> > succActions = 
	  istate->GetSuccsAndActions();
	for (list< pair<const ImplState*,Action> >::const_iterator j = 
	       succActions.begin (); j != succActions.end (); ++j) {
	  const ImplState* nextistate = (*j).first;
	  Action act = (*j).second;
	  set<string> nextsstates = specInfo.GetSuccsOnAction(sstate,act);
	  for (set<string>::const_iterator k = nextsstates.begin (); 
	       k != nextsstates.end (); ++k) {
	    POState s = POState(nextistate,*k);
	    set <POPath> cpsnext;
	    for (set<POPath>::const_iterator l = cps.begin (); 
		 l != cps.end (); ++l) {
	      POPath cpnext = POPath(*l);
	      (cpnext.second).push_back(pair <Action, POState> (act, s));
	      assert (cpnext.second.size () > (*l).second.size ());
	      cpsnext.insert(cpnext);
	    }
	      bool visitSkip = false;
	      if (Database::PREOPTVISIT) {
		if (visitedMap.count(s) == 0) {
		  visitedMap[s] = 1;
		} else {
		  if ((++visitedMap[s]) > Database::PREOPTVISITS)
		    visitSkip = true;
		}
	      }
	      if (!visitSkip) {
		set <POState>::const_iterator finds = frontier.find(s);
		if (finds == frontier.end ()) {
		  frontier.insert(s);
		  frontierMap[s] = cpsnext;
		} else {
		  frontierMap[s].insert(cpsnext.begin (), cpsnext.end ());
		}
	      }
	  }
	}
      } else {
	for (set<POPath>::const_iterator j = cps.begin (); j != cps.end (); ++j) {
	  if (AddsToCoverage(*j, coverMap)) {
	    CFAtraces.push_back(Concretize(*j));
	    if (CFAtraces.size () >= Database::PREOPTMAXCES) {
	      done = true;
	    }
	  }
	}
      }
    }
    for (set <POState>::const_iterator i = frontier.begin ();
	 i != frontier.end (); ++i) {
      nextStates.insert (pair <POState, set <POPath> > (*i, frontierMap[*i]));
    }
    currStates = nextStates;
    if ((CFAtraces.size () >= Database::PREOPTMINCES) && 
	(depth >= Database::PREOPTMINDEPTH)) {
      done = true;
    }
  }
  return CFAtraces;
  */
}

/*********************************************************************/
/*********************************************************************/
vector<ConcCEDag> PredAbsRefiner::CFACounterexamplesDFS () 
{
  assert(false);
  return vector<ConcCEDag>();
  /*
  vector <ConcPath> CFAtraces;
  stack < pair < list< pair<const ImplState*,Action> >::const_iterator,
    set <string>::const_iterator> > sstack;
  set<const ImplState*> initImpls = procManager.initLoc->GetImplStates(procManager.guard);
  procManager.ComputeImplStates ();
  return CFAtraces;
  */
}

/*********************************************************************/
/*********************************************************************/
vector <ConcCEDag> PredAbsRefiner::CFACounterexamples () 
{
  return CFACounterexamplesBFS ();
}

/*********************************************************************/
/*********************************************************************/
void PredAbsRefiner::PreOptUpdateConstraintFile(vector <ConcCEDag> counterexamples) 
{
  assert(false);
  /*
  //open file in append mode
  FILE *cfptr = fopen(procManager.constraintFile.c_str(),"a");
  size_t subsetNum = 0;
  bool flag = false;
  size_t consNum = 0;
  size_t numBranches = procManager.locToPreds.size ();
  set <ContLoc*> branches;
  for (map <ContLoc*,map<ContLoc*,PredSet> >::const_iterator i = procManager.locToPreds.begin ();
       i != procManager.locToPreds.end (); ++i) {
    branches.insert((*i).first);
  }
  map <ConcPath, vector < set<ContLoc*> > > constraints;
  map <ConcPath, bool > eliminated;
  for (vector <ConcPath>::const_iterator i = counterexamples.begin ();
       i != counterexamples.end (); ++i) {
    eliminated[*i] = false;
  }
  //Util::Message (1,"numBranches = %i", numBranches);
  for(size_t l = 0;(subsetNum < Database::SUBSET_CUTOFF) && (!flag || !Database::STOP_OPT_AFTER_ELIM || (l < Database::SMALLEST_SUBSET)) && (l < Database::LARGEST_SUBSET) && 
	(l < numBranches);++l) {
    PredIter pIter(branches,l+1);
    while((subsetNum < Database::SUBSET_CUTOFF) && 
	  (consNum < Database::CONSTRAINT_CUTOFF) && (!pIter.End())) {
      set<ContLoc*> locSub = pIter.Next();
      ++subsetNum;
      for (set<ContLoc*>::const_iterator i = locSub.begin();
	   i != locSub.end(); ++i) {
	Util::Message(1,"Pre-seeding with: %s\n",(*i)->ToString().c_str());
      }
      procManager.SetSeedBranches(locSub);
      ModelExtractor::Run(true);
      bool allEliminated = true;
      for (vector <ConcPath>::const_iterator i = counterexamples.begin ();
	   i != counterexamples.end (); ++i) {
	Util::Message(1, "calling CanEliminate\n");
	if(CanEliminate(*i)) {
	  Util::Message(1,"ELIMINATED ...\n");
	  constraints[*i].push_back(locSub);
	  eliminated[*i] = true;
	}
	if (!eliminated[*i])
	  allEliminated = false;
      }
      if (allEliminated) {
	Util::Message(1,"All counterexamples have been eliminated.\n");
	flag = true;
      }
    }
  }
  for (vector <ConcPath>::const_iterator i = counterexamples.begin ();
       i != counterexamples.end (); ++i) {
    PureCNFForm(constraints[*i],cfptr);
  }

  //close file
  fclose(cfptr);
  */
}

/*********************************************************************/
/*********************************************************************/
void PredAbsRefiner::PreOptimize ()
{
  assert(false);
  /*
  Util::Message(1,"Starting pre-optimization.\n");
  Util::Message(1,"Generating traces.\n");
  vector <ConcPath> CEs = CFACounterexamples ();
  Util::Message(1,"Finished generating %i traces.\n", CEs.size ());
  PreOptUpdateConstraintFile (CEs);
  SolveOptimization ();
  Util::Message(1,"Finished pre-optimization.\n");
  */
}

/*********************************************************************/
// compute whether the set of predicates given can eliminate the given
// path.  assumes predicates and implementation states already
// installed.
/*********************************************************************/
bool PredAbsRefiner::CanEliminate(const ConcCEDag &concDag)
{
  Util::Message(4,"Calling CanEliminate ...\n");
  //if checking for reachability or simulation or deadlock or LTL with finite CE
  if((Database::CONF_TYPE == Database::CONF_SIMUL) || (Database::CONF_TYPE == Database::CONF_REACH) ||
     (Database::CONF_TYPE == Database::CONF_DLOCK) || 
     ((Database::CONF_TYPE == Database::CONF_LTL) && concDag.NoLoop())) {
    //compute in reverse topological order
    vector<L1StSet> dagToImpls(concDag.stateLabel.size());
    procManager.GetL1AbsInit(dagToImpls[0]);
    for(size_t i = 1;i < concDag.stateLabel.size();++i) {
      const CL1AbsState *l1as = static_cast<const CL1AbsState*>(concDag.stateLabel[i]);
      const ContLoc *x = l1as->GetContLoc();
      set<L1AbsState*> y;
      for(set<const ImplState*>::const_iterator j = x->GetImplStates().begin();j != x->GetImplStates().end();++j) {
	if(((*j)->GetSpecStateId() == l1as->GetSpecStateId()) &&
	   ((*j)->GetSpecInfoId() == l1as->GetSpecInfoId())) {
	  y.insert(new CL1AbsState(*j));
	}
      }
      procManager.CreateL1StSet(y,dagToImpls[i]);
      for(set<L1AbsState*>::const_iterator j = y.begin();j != y.end();++j) delete *j;
    }
    vector<int> topo; concDag.ComputeTopo(topo);
    for(vector<int>::const_iterator i = topo.begin();i != topo.end();++i) {
      //intersect successors
      L1StSet is; procManager.GetCompleteL1StSet(is);
      for(set<int>::const_iterator j = concDag.trans[*i].begin();j != concDag.trans[*i].end();++j) {	
	procManager.IntersectL1StSets(is,dagToImpls[*j]);
	if(is.Empty()) return true;
      }
      //take pre-image if necessary
      if((Database::CONF_TYPE != Database::CONF_DLOCK) || (i != topo.begin())) {
	const Action &act = concDag.actLabel[*i];
	if(procManager.absLts.GetActs().count(act) != 0) {
	  is = procManager.GetL1AbsPredsOnAction(is,act);
	  if(is.Empty()) return true;
	}
      }
      //project on global ce dag
      procManager.IntersectL1StSets(dagToImpls[*i],is);
      if(dagToImpls[*i].Empty()) return true;
    }
    return false;
  }
  //if checking LTL formula
  else if(Database::CONF_TYPE == Database::CONF_LTL) {
    //initialize the simulation relation
    map< size_t,set<const ImplState*> > simRel;
    set<const ImplState*> initImpls = procManager.initLoc->GetImplStates(procManager.guard);
    for(set<const ImplState*>::const_iterator i = procManager.reachImpl.begin();i != procManager.reachImpl.end();++i) {
      const ContLoc *contLoc = (*i)->GetContLoc();
      const PredSet &preds = contLoc->GetPreds();
      const PredVal &pval = preds.PValToPredVal((*i)->GetPredVal());
      set<Expr> pcons; preds.ToExpr(pval,pcons);
      set<Expr> negCons; ExprManager::NegateExprSet(pcons,negCons);
      size_t currNode = 0;
      while(true) {
	const CL1AbsState *l1as = static_cast<const CL1AbsState*>(concDag.stateLabel[currNode]);
	if(((*i)->GetContLoc() == l1as->GetContLoc()) && 
	   ((*i)->GetSpecStateId() == l1as->GetSpecStateId()) &&
	   ((*i)->GetSpecInfoId() == l1as->GetSpecInfoId())) {
	  set<Expr> cons = static_cast<CL1AbsState*>(concDag.stateLabel[currNode])->GetCons();
	  if(!ExprManager::ProveImplies(cons,negCons)) {
	    if(((currNode == 0) && (initImpls.count(*i) != 0)) || (currNode != 0)) {
	      simRel[currNode].insert(*i);
	    }	    
	  }
	}
	size_t nextNode = *(concDag.trans[currNode].begin());
	if(nextNode <= currNode) break;
	currNode = nextNode;
      }
    }
    //compute the maximal simulation relation
    while(true) {
      bool changed = false;
      for(map< size_t,set<const ImplState*> >::iterator i = simRel.begin();i != simRel.end();++i) {
	//check if simulation relation does not exist
	if(i->second.empty()) return true;
	//update simulation relation
	size_t succ = *(concDag.trans[i->first].begin());
	set<const ImplState*> succSim = simRel[succ];
	const Action &act = concDag.actLabel[i->first];
	set<const ImplState*> currSim;
	if(procManager.absLts.GetActs().count(act) == 0) currSim = succSim;
	else {
	  for(set<const ImplState*>::const_iterator j = succSim.begin();j != succSim.end();++j) {
	    set<const ImplState*> x; (*j)->GetPredsOnAction(act,x);
	    currSim.insert(x.begin(),x.end());
	  }
	}
	set<const ImplState*> newCurrSim;
	for(set<const ImplState*>::const_iterator j = i->second.begin();j != i->second.end();++j) {
	  if(currSim.count(*j) == 0) changed = true;
	  else newCurrSim.insert(*j);
	}
	i->second = newCurrSim;
      }
      if(!changed) return false;
    }
    assert(false);
  }
  //impossible
  else assert(false);
}

/*********************************************************************/
// compute whether the set of predicates given can eliminate the given
// path. 
/*********************************************************************/
bool PredAbsRefiner::CanEliminate(const set<ContLoc*> &locSet,const ConcCEDag &concDag)
{
  //if new predicates are to be inferred just from the counterexample
  if(Database::PREDS_FROM_CE) {
    vector< set<Expr> > dagToWP(concDag.stateLabel.size());
    //if checking simulation of ERROR state reachability
    if((Database::CONF_TYPE == Database::CONF_SIMUL) || (Database::CONF_TYPE == Database::CONF_REACH)) {
      vector<int> topo; concDag.ComputeTopo(topo);
      for(vector<int>::const_iterator i = topo.begin();i != topo.end();++i) {
	for(set<int>::const_iterator j = concDag.trans[*i].begin();j != concDag.trans[*i].end();++j) {
	  dagToWP[*i].insert(dagToWP[*j].begin(),dagToWP[*j].end());
	}
	map< int,pair<Action,Expr> > inlinedAssignMap;
	set<ContLoc*> dummyExp;
	CEDagVerifier(procManager,false).UpdatePrePost(locSet,dummyExp,dagToWP[*i],*i,concDag,inlinedAssignMap,true,1);
	//set<Expr> cons = static_cast<const CL1AbsState*>(concDag.stateLabel[*i])->GetCons();
	//dagToWP[*i].insert(cons.begin(),cons.end());
      }
    }
    //if checking LTL formula
    else if(Database::CONF_TYPE == Database::CONF_LTL) {
      for(int i = 0;i < concDag.loopCount;++i) {
	for(list<size_t>::const_iterator j = concDag.loop.begin();j != concDag.loop.end();++j) {
	  for(set<int>::const_iterator k = concDag.trans[*j].begin();k != concDag.trans[*j].end();++k) {
	    dagToWP[*j].insert(dagToWP[*k].begin(),dagToWP[*k].end());
	  }
	  map< int,pair<Action,Expr> > inlinedAssignMap;
	  set<ContLoc*> dummyExp;
	  CEDagVerifier(procManager,false).UpdatePrePost(locSet,dummyExp,dagToWP[*j],*j,concDag,inlinedAssignMap,true,1);
	  set<Expr> cons = static_cast<const CL1AbsState*>(concDag.stateLabel[*j])->GetCons();
	  dagToWP[*j].insert(cons.begin(),cons.end());
	}
      }
      for(list<size_t>::const_iterator j = concDag.handle.begin();j != concDag.handle.end();++j) {
	for(set<int>::const_iterator k = concDag.trans[*j].begin();k != concDag.trans[*j].end();++k) {
	  dagToWP[*j].insert(dagToWP[*k].begin(),dagToWP[*k].end());
	}	
	map< int,pair<Action,Expr> > inlinedAssignMap;
	set<ContLoc*> dummyExp;
	CEDagVerifier(procManager,false).UpdatePrePost(locSet,dummyExp,dagToWP[*j],*j,concDag,inlinedAssignMap,true,1);
	set<Expr> cons = static_cast<const CL1AbsState*>(concDag.stateLabel[*j])->GetCons();
	dagToWP[*j].insert(cons.begin(),cons.end());
      }
    }
    //impossible
    else assert(false);
    dagToWP[0].insert(procManager.guard);
    return ExprManager::ProveImplies(dagToWP[0],set<Expr>());
  }
  //if new predicates are to be inferred using information other than
  //just the counterexample
  else {
    //create the predicate sets at each location
    procManager.SetSeedBranches(locSet);
    //create the implementation states at useful locations
    ModelExtractor::Run(true);
    return CanEliminate(concDag);
  }
}

/*********************************************************************/
//end of PredAbsRefiner.cpp
/*********************************************************************/
