/******************************** CPPFile *****************************

* FileName [ModelExtractor.cpp]

* PackageName [main]

* Synopsis [This file contains method definitions of the
* ModelExtractor class.]

* SeeAlso [ModelExtractor.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <gmp.h>
#include <string>
#include <list>
#include <set>
#include <map>
#include <vector>
using namespace std;

#include "BigInt.h"
#include "Timer.h"
#include "Util.h"
#include "Node.h"
#include "Action.h"
#include "ProcAbs.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "ContLoc.h"
#include "ProcInfo.h"
#include "CallGraph.h"
#include "AbsRefiner.h"
#include "LocalAbsLts.h"
#include "ConcCEDag.h"
#include "Component.h"
#include "ProcManager.h"
#include "GlobalAbsLts.h"
#include "ModelExtractor.h"
using namespace magic;

/*********************************************************************/
//extract finite state model from C program. return true on success
//and false otherwise.
/*********************************************************************/
bool ModelExtractor::Run(const bool predAbsOnly)
{
  Timer meTimer; meTimer.Start();
  //clear all caches of the expression manager
  ExprManager::ClearCaches();
  //if the implementation is a C program
  if(Database::IMPL_TYPE == Database::IMPL_C) {
    //compute the implementation states
    ComputeImplStates(predAbsOnly);
    Util::Message(2,"implementation states for control locations computed ...\n");    
    //compute the level-2 abstract transitions if required
    if(!predAbsOnly) {
      GlobalAbsLts::ComputeActInfo();
      for(size_t i = 0;i < Database::components.size();++i) {
	if((!Database::HANDLE_DATA_COMM) && (Database::lastRefComp != -1) && 
	   (static_cast<int>(i) != Database::lastRefComp)) continue;
	ProcManager *procManager = static_cast<ProcManager*>(Database::components[i]);
	procManager->ComputeAbsLtsTrans();
      }
      Util::Message(2,"action-guided transitions computed ...\n");
    }
  }
  //otherwise the implementation must be a PACC process. in this case
  //we do not have to do anything.
  else assert(Database::IMPL_TYPE == Database::IMPL_PACC);
  meTimer.Stop();
  Util::Message(2,"model extracted in %.1f milliseconds ...\n",meTimer.Read() * 1000);
  //success
  return true;
}

/*********************************************************************/
//compute the set of implementation states for each control location
/*********************************************************************/
void ModelExtractor::ComputeImplStates(const bool predAbsOnly)
{
  //check if some component has a new seed predicates
  int nsCompId = -1;
  for(size_t i = 0;i < Database::components.size();++i) {
    ProcManager *procManager = static_cast<ProcManager*>(Database::components[i]);
    if(procManager->GetNewSeeds()) {
      nsCompId = i;
      break;
    }
  }
  if(nsCompId == -1) return;  
  //if some component has new seeds
  vector< set<Action> > newChanActs(Database::components.size());
  vector< set<Action> > pendChanActs(Database::components.size());
  vector< set<Action> > doneChanActs(Database::components.size());
  for(size_t i = 0;i < Database::components.size();++i) {
    ProcManager *procManager = static_cast<ProcManager*>(Database::components[i]);
    if((!Database::HANDLE_DATA_COMM) && (!procManager->GetNewSeeds())) continue;
    procManager->InitializeImplStates(newChanActs[i]);
  }
  //compute the implementation states
  while(Database::HANDLE_DATA_COMM) {
    bool flag = false;
    for(size_t i = 0;i < Database::components.size();++i) {
      for(size_t j = 0;j < Database::components.size();++j) {
	if(i == j) continue;
	for(set<Action>::const_iterator k = newChanActs[i].begin();k != newChanActs[i].end();++k) {
	  if(doneChanActs[j].count(*k) == 0) {
	    pendChanActs[j].insert(*k);
	    flag = true;
	  }
	}
      }
    }
    if(!flag) break;
    //clear the new channel actions
    for(size_t i = 0;i < Database::components.size();++i) {
      newChanActs[i].clear();
    }
    //update the component implementation states on the basis of
    //pending channel actions. get the new channel actions back
    for(size_t i = 0;i < Database::components.size();++i) {
      if(!pendChanActs[i].empty()) {
	ProcManager *procManager = static_cast<ProcManager*>(Database::components[i]);
	procManager->UpdateImplStates(pendChanActs[i],newChanActs[i]);
      }
    }
    //update the done channel actions and clear the pending channel
    //actions
    for(size_t i = 0;i < Database::components.size();++i) {
      doneChanActs[i].insert(pendChanActs[i].begin(),pendChanActs[i].end());
    }   
  }
  //finish up
  for(size_t i = 0;i < Database::components.size();++i) {
    ProcManager *procManager = static_cast<ProcManager*>(Database::components[i]);
    if((!Database::HANDLE_DATA_COMM) && (!procManager->GetNewSeeds())) continue;
    if(Database::DRAW_INFERRED_PREDS) procManager->DrawInferredPreds();
    if(Database::DRAW_PRED_ABS_LTS) procManager->DrawPredAbsLts();
    if(!predAbsOnly) {
      if(Database::OUTPUT_MODELS) procManager->OutputModel();
      procManager->EliminateSilentTrans();
      procManager->CreateInitialAbstractStates();
      procManager->ComputeAbsLtsInit();
    }
    procManager->SetNewSeeds(false);
  }
}

/*********************************************************************/
//end of ModelExtractor.cpp
/*********************************************************************/
