/******************************** CPPFile *****************************

* FileName [LtsInfo.cpp]

* PackageName [main]

* Synopsis [Method definitions of LtsInfo class.]

* SeeAlso [LtsInfo.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <string>
#include <list>
#include <set>
#include <map>
#include <vector>
using namespace std;

#include "Util.h"
#include "Node.h"
#include "Action.h"
#include "ActionManager.h"
#include "LtsTrans.h"
#include "LtsInfo.h"
#include "Database.h"
using namespace magic;

/*********************************************************************/
//constructors and destructors
/*********************************************************************/
LtsInfo::LtsInfo()
{
  useful = false;
  index = 0;
}

LtsInfo::LtsInfo(const string &n,const set<string> &s,const set<Action> &a,const set<LtsTrans> &t)
{
  name = n;
  useful = false;
  states = s;
  actions = a;  
  trans = t;
  index = 0;
  //create the states to indices map
  int id = 0;
  for(set<string>::const_iterator i = states.begin();i != states.end();++i,++id) {
    statesToIds[*i] = id;
  }
  //create the successor and predecessor action map
  for(set<LtsTrans>::const_iterator i = trans.begin();i != trans.end();++i) {
    succMap[pair<string,Action>(i->GetInitState(),i->GetAction())].insert(i->GetFinalState());
    predMap[pair<string,Action>(i->GetFinalState(),i->GetAction())].insert(i->GetInitState());
  }
}

LtsInfo::LtsInfo(const LtsInfo &rhs) { *this = rhs; }

/*********************************************************************/
//operators
/*********************************************************************/
const LtsInfo &LtsInfo::operator = (const LtsInfo &rhs)
{
  name = rhs.name;
  useful = rhs.useful;
  states = rhs.states;
  statesToIds = rhs.statesToIds;
  actions = rhs.actions;
  trans = rhs.trans;
  index = rhs.index;
  succMap = rhs.succMap;
  predMap = rhs.predMap;
  return *this;
}

/*********************************************************************/
//compute the set of useful states
/*********************************************************************/
void LtsInfo::ComputeUseful()
{
  //if this is the info about the spec LTS then it is useful
  if(name == Database::specLtsName) {
    useful = true;
    return;
  }
  //check if this lts has any return action other than the default
  //action or assign actions
  for(set<Action>::iterator i = actions.begin();i != actions.end();++i) {
    if((!i->IsDefaultRet() && (i->IsReturn())) || 
       (i->IsAssign())) {
      useful = true;
      return;
    }
  }  
  //collect spec actions and channels associated with those actions
  const set<Action> &specActs = Database::GetSpecActions();
  set<string> specChans;
  for(set<Action>::const_iterator i = specActs.begin();i != specActs.end();++i) {
    if(i->IsBcast() || i->IsSend() || i->IsRecv()) specChans.insert(i->GetChannel());
  }
  //check if it has some spec action or some communication action that
  //has the same channel as a spec action or if the channel name is
  //the same as a spec action
  for(set<Action>::iterator i = actions.begin();i != actions.end();++i) {
    if(i->IsBcast() || i->IsSend() || i->IsRecv()) {
      if((specChans.count(i->GetChannel()) != 0) || (specActs.count(Action(i->GetChannel())) != 0)) {
	useful = true;
	return;
      }
    }
    //only consider basic non-epsilon actions
    if((!i->IsBasic()) || (i->IsSpecEpsilon()))continue;
    if(specActs.count(*i) != 0) {
      useful = true;
      return;
    }
  }
}

/*********************************************************************/
//return the expressions associated with assign actions
/*********************************************************************/
void LtsInfo::GetAssignActExprs(set< pair<Expr,Expr> > &res) const
{
  for(set<Action>::iterator i = actions.begin();i != actions.end();++i) {
    if(i->IsAssign()) {
      res.insert(pair<Expr,Expr>(i->GetLhsExpr(),i->GetRhsExpr()));
    }
  }
}

/*********************************************************************/
//return the expressions associated with return actions
/*********************************************************************/
void LtsInfo::GetRetActExprs(set<Expr> &res) const
{
  for(set<Action>::iterator i = actions.begin();i != actions.end();++i) {
    if(i->IsReturn() && (!i->GetRetExpr().IsEmptyExpr())) res.insert(i->GetRetExpr());
  }
}

/*********************************************************************/
//compute the set of actions associated with a return. the arguments
//are the expression for the return value, a list of arguments to the
//procedure, and a list of expressions describing the state in which
//the return is executed.
/*********************************************************************/
void LtsInfo::GetReturnActions(const Expr &expr,const list<string> &params,
			       const set<Expr> &context,set<Action> &res) const
{
  set<Action> returns; ComputeReturnActions(returns);
  for(set<Action>::const_iterator i = returns.begin();i != returns.end();++i) {
    if(i->ReturnActionPossible(expr,params,context)) res.insert(*i);
  }
}

/*********************************************************************/
//compute the set of actions associated with a return. the arguments
//are the expression for the return value, a list of arguments to the
//procedure, and a list of expressions describing the state in which
//the return is executed.
/*********************************************************************/
void LtsInfo::GetAssignActions(const Expr &lhs,const Expr &rhs,
			       const list<string> &params,
			       const set<Expr> &context,set<Action> &res) const
{
  set<Action> assigns; ComputeAssignActions(assigns);
  for(set<Action>::const_iterator i = assigns.begin();i != assigns.end();++i) {
    if(i->AssignActionPossible(lhs,rhs,params,context)) res.insert(*i);
  }
}

/*********************************************************************/
//compute the set of return actions in this LTS
/*********************************************************************/
void LtsInfo::ComputeReturnActions(set<Action> &res) const
{
  for(set<Action>::const_iterator i = actions.begin();i != actions.end();++i) {
    if(i->IsReturn()) res.insert(*i);
  }
}

/*********************************************************************/
//compute the set of assign actions in this LTS
/*********************************************************************/
void LtsInfo::ComputeAssignActions(set<Action> &res) const
{
  for(set<Action>::const_iterator i = actions.begin();i != actions.end();++i) {
    if(i->IsAssign()) res.insert(*i);
  }
}

/*********************************************************************/
//given a state of the lts return the list of its outgoing transitions
/*********************************************************************/
void LtsInfo::GetOutTrans(const string &source,list<LtsTrans> &res) const
{
  for(set<LtsTrans>::const_iterator i = trans.begin();i != trans.end();++i) {
    if(i->GetInitState() == source) res.push_back(*i);
  }
}

/*********************************************************************/
//given a state X and an action Y return the set of states Z such that
//there is transition from X to Z with action Y.
/*********************************************************************/
void LtsInfo::GetSuccsOnAction(const string &source,const Action &action,set<string> &res) const
{
  map< pair<string,Action>,set<string> >::const_iterator i = succMap.find(pair<string,Action>(source,action));
  if(i == succMap.end()) res = set<string>();
  else res = i->second;
}

/*********************************************************************/
//given a state X and an action Y return the set of states Z such that
//there is transition from Z to X with action Y.
/*********************************************************************/
void LtsInfo::GetPredsOnAction(const string &source,const Action &action,set<string> &res) const
{
  map< pair<string,Action>,set<string> >::const_iterator i = predMap.find(pair<string,Action>(source,action));
  if(i == predMap.end()) res = set<string>();
  else res = i->second;
}

/*********************************************************************/
//given a state X and an action Y return the set of states Z such that
//there is transition from X to Z with action Y.
/*********************************************************************/
void LtsInfo::GetSuccsOnActionSpec(const string &source,const Action &action,set<string> &res) const
{
  //if this is a send or broadcast action then change it into a basic
  //action based on the channel name
  if(action.IsBcast() || action.IsSend() || action.IsRecv()) {
    Action newAct(action.GetChannel());
    if(actions.count(newAct) == 0) {
      GetSuccsOnAction(source,ActionManager::GetSpecEpsilonAction(),res);
      if(((Database::CONF_TYPE == Database::CONF_REACH) && (!Util::IsTempVar(Database::specLtsName))) || 
	 (Database::CONF_TYPE == Database::CONF_SIMUL)) {
	if((source != Database::STOP_STATE) && (source != Database::ERROR_STATE)) {
	  res.insert(source);
	}
      }
    } else GetSuccsOnAction(source,newAct,res);
  } else if(action.IsBasic() || action.IsAssign() || action.IsReturn()) {
    //if the action is not a default return or invalid pointer
    //dereference or assertion failure and does not belong to the set of
    //actions of this LTS then turn it into an epsilon action
    if((!action.IsDefaultRet()) && (!action.IsInvalidPtrDeref()) && 
       (!action.IsAssertionFailure()) && (actions.count(action) == 0)) {
      GetSuccsOnAction(source,ActionManager::GetSpecEpsilonAction(),res);
      //consider epsilon self-loops
      if(((Database::CONF_TYPE == Database::CONF_REACH) && (!Util::IsTempVar(Database::specLtsName))) || 
	 (Database::CONF_TYPE == Database::CONF_SIMUL)) {
	if((source != Database::STOP_STATE) && (source != Database::ERROR_STATE)) {
	  res.insert(source);
	}
      }
    } else {
      GetSuccsOnAction(source,action,res);
      //consider epsilon self-loops
      if(((Database::CONF_TYPE == Database::CONF_REACH) && (!Util::IsTempVar(Database::specLtsName))) || 
	 (Database::CONF_TYPE == Database::CONF_SIMUL)) {
	if((source != Database::STOP_STATE) && (source != Database::ERROR_STATE) && 
	   (action == ActionManager::GetSpecEpsilonAction())) {
	  res.insert(source);
	}
      }
    }
  } else assert(action.IsComplement());
}

void LtsInfo::GetSuccsOnActionSpec(const int source,const Action &action,set<string> &res) const
{
  GetSuccsOnActionSpec(GetIdState(source),action,res);
}

/*********************************************************************/
//return the id of a state
/*********************************************************************/
int LtsInfo::GetStateId(const string &s) const
{
  map<string,int>::const_iterator i = statesToIds.find(s);
  assert(i != statesToIds.end());
  return i->second;
}

/*********************************************************************/
//given an id return the corresponding state
/*********************************************************************/
string LtsInfo::GetIdState(const int id) const
{
  for(map<string,int>::const_iterator i = statesToIds.begin();i != statesToIds.end();++i) {
    if(i->second == id) return i->first;
  }
  assert(false);
  return "";
}

/*********************************************************************/
//draw the LTS as a dot file
/*********************************************************************/
void LtsInfo::Draw() const
{
  FILE *out = Database::dotFilePtr;
  fprintf(out,"  //FSP process %s\n",name.c_str());
  fprintf(out,"  subgraph cluster_FSP_%s {\n",name.c_str());  
  fprintf(out,"    color = white;\n");
  fprintf(out,"    %s_%s [peripheries = 2];\n",name.c_str(),name.c_str()); 
  for(set<string>::const_iterator i = states.begin();i != states.end();++i) {
    fprintf(out,"    %s_%s [label = \"%s\"];\n",name.c_str(),i->c_str(),i->c_str()); 
  }
  for(set<LtsTrans>::const_iterator i = trans.begin();i != trans.end();++i) {
    fprintf(out,"    %s_%s -> %s_%s [label = \"%s\"];\n",name.c_str(),i->GetInitState().c_str(),name.c_str(),i->GetFinalState().c_str(),i->GetAction().ToString().c_str());
  }
  fprintf(out,"    label = \"FSP process: %s\";\n",name.c_str());
  fprintf(out,"    labelloc = \"b\";\n");
  fprintf(out,"  }\n\n");
}

/*********************************************************************/
//end of LtsInfo.cpp
/*********************************************************************/
