/******************************** CPPFile *****************************

* FileName [LtsAbsRefiner.cpp]

* PackageName [main]

* Synopsis [Method definitions of LtsAbsRefiner class.]

* SeeAlso [LtsAbsRefiner.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <gmp.h>
#include <string>
#include <vector>
#include <set>
#include <map>
#include <list>
#include <bitset>
using namespace std;

#include "BigInt.h"
#include "Util.h"
#include "Node.h"
#include "Action.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "ContLoc.h"
#include "ImplState.h"
#include "LocalAbsLts.h"
#include "ConcCEDag.h"
#include "Component.h"
#include "ProcManager.h"
#include "LtsAbsRefiner.h"
using namespace magic;

/*********************************************************************/
//define static fields
/*********************************************************************/
const int LtsAbsRefiner::LTS_ABS_REFINED = 11700;
const int LtsAbsRefiner::LTS_ABS_NOT_REFINED = 11710;

/*********************************************************************/
//do it all
/*********************************************************************/
int LtsAbsRefiner::Run()
{
  //create the old map from absIds to implStates
  vector< set<const ImplState*> > oldIdToImpls(procManager.absLts.stateNum);
  for(set<const ImplState*>::const_iterator i = procManager.reachImpl.begin();i != procManager.reachImpl.end();++i) {
    oldIdToImpls[(*i)->GetAbsId()].insert(*i);
  }
  //refine
  LocalAbsLts &absLts = procManager.absLts;
  absLts.prevStateNum = absLts.stateNum;
  //ids of split states
  set<int> currIds; 
  for(map< int,set<const ImplState*> >::const_iterator i = procManager.absLtsRefine.begin();i != procManager.absLtsRefine.end();++i) {
    assert((!i->second.empty()) && (i->second.size() < oldIdToImpls[i->first].size()));
    currIds.insert(i->first);
    currIds.insert(absLts.stateNum);
    for(set<const ImplState*>::const_iterator j = i->second.begin();j != i->second.end();++j) {
      ImplState *is = (ImplState*)(*j);
      is->SetAbsId(absLts.stateNum);
      set<Action> ref; is->GetRefusal(ref);
      absLts.refMap[absLts.stateNum].insert(ref.begin(),ref.end());
    }
    //update the proposition map
    (*(i->second.begin()))->GetPropositions(absLts.propMap[absLts.stateNum]);
    //update the refusal map of split state
    absLts.refMap[i->first].clear();
    for(set<const ImplState*>::const_iterator j = oldIdToImpls[i->first].begin();j != oldIdToImpls[i->first].end();++j) {
      if((*j)->GetAbsId() == i->first) {
	set<Action> ref; (*j)->GetRefusal(ref);
	absLts.refMap[i->first].insert(ref.begin(),ref.end());
      }
    }
    ++absLts.stateNum;
  } 
  //compute new abstract initial states
  Util::Message(2,"%d abstract implementation states after lts refinement of component #%d ...\n",absLts.stateNum,procManager.GetIndex());
  absLts.prevInit = absLts.init;
  absLts.init.clear();
  set<const ImplState*> initImpls = procManager.initLoc->GetImplStates(procManager.guard);
  for(set<const ImplState*>::const_iterator i = initImpls.begin();i != initImpls.end();++i) {
    absLts.init.insert((*i)->GetAbsId());
  }
  //compute the new abstract successors and actions
  if(Database::PRECOMPUTE_IMPL_TRANS) {
    absLts.succsChanged = currIds;
    //ids of predecessors of split states
    set<int> predIds;
    for(set<int>::const_iterator i = currIds.begin();i != currIds.end();++i) {
      const map< Action,set<int> > &x = absLts.predMap[*i];
      for(map< Action,set<int> >::const_iterator j = x.begin();j != x.end();++j) {
	for(set<int>::const_iterator k = j->second.begin();k != j->second.end();++k) {
	  if(currIds.count(*k) == 0) {
	    predIds.insert(*k);
	    absLts.succsChanged.insert(*k);
	  }
	}	
      }
    }
    //ids of successors of split states
    set<int> succIds;
    for(set<int>::const_iterator i = currIds.begin();i != currIds.end();++i) {
      const map< Action,set<int> > &x = absLts.succMap[*i];
      for(map< Action,set<int> >::const_iterator j = x.begin();j != x.end();++j) {
	for(set<int>::const_iterator k = j->second.begin();k != j->second.end();++k) {
	  if(currIds.count(*k) == 0) {
	    succIds.insert(*k);
	  }
	}
      }
    }
    //ids of predecessors of successors of split states
    set<int> succPredIds;
    for(set<int>::const_iterator i = succIds.begin();i != succIds.end();++i) {
      const map< Action,set<int> > &x = absLts.predMap[*i];
      for(map< Action,set<int> >::const_iterator j = x.begin();j != x.end();++j) {
	for(set<int>::const_iterator k = j->second.begin();k != j->second.end();++k) {
	  if(currIds.count(*k) == 0) {
	    succPredIds.insert(*k);
	  }
	}
      }
    }

    //save the necessary transition relations from previous iteration
    absLts.prevSuccMap.clear();
    for(set<int>::const_iterator i = absLts.succsChanged.begin();i != absLts.succsChanged.end();++i) {
      absLts.prevSuccMap[*i] = absLts.succMap[*i];
    }

    //clear predecessors and successors that will be recomputed
    for(set<int>::const_iterator i = currIds.begin();i != currIds.end();++i) {
      absLts.succMap[*i].clear();
      absLts.predMap[*i].clear();
    }
    for(set<int>::const_iterator i = predIds.begin();i != predIds.end();++i) {
      absLts.succMap[*i].clear();
    }
    for(set<int>::const_iterator i = succIds.begin();i != succIds.end();++i) {
      absLts.predMap[*i].clear();
    }
    for(set<int>::const_iterator i = succPredIds.begin();i != succPredIds.end();++i) {
      absLts.succMap[*i].clear();
    }

    //map from abstract id to impl states
    map< int,set<const ImplState*> > idToImpls;
    for(set<const ImplState*>::const_iterator i = procManager.reachImpl.begin();i != procManager.reachImpl.end();++i) {
      idToImpls[(*i)->GetAbsId()].insert(*i);
    }

    //recompute successors of predecessor states
    for(set<int>::const_iterator i = predIds.begin();i != predIds.end();++i) {
      for(set<const ImplState*>::const_iterator j = idToImpls[*i].begin();j != idToImpls[*i].end();++j) {
	map< Action,set<const ImplState*> > sa;
	(*j)->GetSuccsAndActions(set<Action>(),set<Action>(),sa);
	for(map< Action,set<const ImplState*> >::const_iterator k = sa.begin();k != sa.end();++k) {
	  for(set<const ImplState*>::const_iterator l = k->second.begin();l != k->second.end();++l) {
	    absLts.succMap[(*j)->GetAbsId()][k->first].insert((*l)->GetAbsId());
	    absLts.predMap[(*l)->GetAbsId()][k->first].insert((*j)->GetAbsId());
	  }
	}
      }
    }
    //recompute successors of current states
    for(set<int>::const_iterator i = currIds.begin();i != currIds.end();++i) {
      for(set<const ImplState*>::const_iterator j = idToImpls[*i].begin();j != idToImpls[*i].end();++j) {
	map< Action,set<const ImplState*> > sa;
	(*j)->GetSuccsAndActions(set<Action>(),set<Action>(),sa);
	for(map< Action,set<const ImplState*> >::const_iterator k = sa.begin();k != sa.end();++k) {
	  for(set<const ImplState*>::const_iterator l = k->second.begin();l != k->second.end();++l) {
	    absLts.succMap[(*j)->GetAbsId()][k->first].insert((*l)->GetAbsId());
	    absLts.predMap[(*l)->GetAbsId()][k->first].insert((*j)->GetAbsId());
	  }
	}
      }
    }
    //recompute successors of predecessors of successor states
    for(set<int>::const_iterator i = succPredIds.begin();i != succPredIds.end();++i) {
      for(set<const ImplState*>::const_iterator j = idToImpls[*i].begin();j != idToImpls[*i].end();++j) {
	map< Action,set<const ImplState*> > sa;
	(*j)->GetSuccsAndActions(set<Action>(),set<Action>(),sa);
	for(map< Action,set<const ImplState*> >::const_iterator k = sa.begin();k != sa.end();++k) {
	  for(set<const ImplState*>::const_iterator l = k->second.begin();l != k->second.end();++l) {
	    absLts.succMap[(*j)->GetAbsId()][k->first].insert((*l)->GetAbsId());
	    absLts.predMap[(*l)->GetAbsId()][k->first].insert((*j)->GetAbsId());
	  }
	}
      }
    }
  }
  procManager.absLtsRefine.clear();
  //all done
  return (absLts.stateNum > absLts.prevStateNum) ? LTS_ABS_REFINED : LTS_ABS_NOT_REFINED;
}

/*********************************************************************/
//end of LtsAbsRefiner.cpp
/*********************************************************************/
