/******************************** CPPFile *****************************

* FileName [LtlChecker.cpp]

* PackageName [main]

* Synopsis [Method definitions of LtlChecker class.]

* SeeAlso [LtlChecker.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cassert>
#include <gmp.h>
#include <string>
#include <list>
#include <set>
#include <map>
#include <vector>
using namespace std;

#include "BigInt.h"
#include "Node.h"
#include "Action.h"
#include "Seaw.h"
#include "Database.h"
#include "GlobalAbsLts.h"
#include "Buchi.h"
#include "LtlChecker.h"
#include "SeawChecker.h"
using namespace magic;

/*********************************************************************/
//define static members
/*********************************************************************/
size_t LtlChecker::ssn;
set<BigInt> LtlChecker::visited1;
set<BigInt> LtlChecker::visited2;
set<BigInt> LtlChecker::stack;
list< list<BigInt> > LtlChecker::statePaths;
list< list<Action> > LtlChecker::actPaths;
set<BigInt> LtlChecker::ceStates;
bool LtlChecker::error;
bool LtlChecker::done;

/*********************************************************************/
//check if the spec LTL formula holds on the program. return true if
//it holds and false otherwise.
/*********************************************************************/
bool LtlChecker::CheckFormula()
{
  //get initial states
  set<BigInt> implInit; GlobalAbsLts::GetInitStates(implInit);
  set<int> specInit; Buchi::GetInitStates(specInit);
  //initialise data structures
  ssn = Buchi::GetStateNum();
  statePaths.clear();
  actPaths.clear();
  ceStates.clear();
  error = false;
  done = false;
  //do double DFS
  for(set<BigInt>::const_iterator i = implInit.begin();(!done) && (i != implInit.end());++i) {
    for(set<int>::const_iterator j = specInit.begin();(!done) && (j != specInit.end());++j) {
      if(PropCompatible(*i,*j)) {
	visited1.clear();
	visited2.clear();
	stack.clear();
	BigInt node = (*i) * ssn + (*j);
	list<BigInt> statePath;
	list<Action> actPath;
	if(ceStates.count(node) == 0) DFS1(node,statePath,actPath);
	else CreateShorterCE(node,statePath,actPath);
      }
    }
  }
  //cleanup and return
  visited1.clear();
  visited2.clear();
  stack.clear();
  ceStates.clear();
  return !error;
}

/*********************************************************************/
//check if the spec LTL formula holds on the given state of the
//program. return true if it holds and false otherwise.
/*********************************************************************/
bool LtlChecker::CheckFormula(const BigInt &implInit)
{
  assert(Database::CONF_TYPE == Database::CONF_SEAW);
  //get initial states
  set<int> specInit; Buchi::GetInitStates(specInit);
  //initialise data structures
  ssn = Buchi::GetStateNum();
  statePaths.clear();
  actPaths.clear();
  ceStates.clear();
  error = false;
  done = false;
  //do double DFS
  for(set<int>::const_iterator i = specInit.begin();(!done) && (i != specInit.end());++i) {
    if(PropCompatible(implInit,*i)) {
      visited1.clear();
      visited2.clear();
      stack.clear();
      BigInt node = implInit * ssn + (*i);
      list<BigInt> statePath;
      list<Action> actPath;
      if(ceStates.count(node) == 0) DFS1(node,statePath,actPath);
      else CreateShorterCE(node,statePath,actPath);
    }
  }
  //cleanup and return
  visited1.clear();
  visited2.clear();
  stack.clear();
  ceStates.clear();
  return !error;
}

/*********************************************************************/
//the first dfs of ltl model checking
/*********************************************************************/
void LtlChecker::DFS1(const BigInt &node,list<BigInt> &statePath,list<Action> &actPath)
{
  visited1.insert(node);
  stack.insert(node);
  statePath.push_back(node);
  BigInt implState,rem; BigInt::Div(implState,rem,node,ssn);
  int specState = rem.ToSL();
  map< Action,set<BigInt> > saa;  
  GlobalAbsLts::GetSuccsAndActions(implState,saa);
  for(map< Action,set<BigInt> >::const_iterator i = saa.begin();(!done) && (i != saa.end());++i) {
    if(ActCompatible(i->first,specState)) {
      set<int> specSuccs = Buchi::GetSuccs(specState);
      for(set<BigInt>::const_iterator j = i->second.begin();(!done) && (j != i->second.end());++j) {
	for(set<int>::const_iterator k = specSuccs.begin();(!done) && (k != specSuccs.end());++k) {
	  if(PropCompatible(*j,*k)) {
	    BigInt next = (*j) * ssn + (*k);
	    if(visited1.count(next) == 0) {
	      actPath.push_back(i->first);
	      DFS1(next,statePath,actPath);
	      actPath.pop_back();
	    } else if(ceStates.count(next) != 0) {
	      actPath.push_back(i->first);
	      CreateShorterCE(next,statePath,actPath);
	      actPath.pop_back();
	    }
	  }
	}
      }
    }
  }
  if((!done) && (Buchi::IsAccept(specState)) && (visited2.count(node) == 0)) {
    DFS2(node,statePath,actPath);
  }
  stack.erase(node);
  statePath.pop_back();
}

/*********************************************************************/
//the second dfs of ltl model checking
/*********************************************************************/
void LtlChecker::DFS2(const BigInt &node,list<BigInt> &statePath,list<Action> &actPath)
{
  visited2.insert(node);
  BigInt implState,rem; BigInt::Div(implState,rem,node,ssn);
  int specState = rem.ToSL();
  map< Action,set<BigInt> > saa;
  GlobalAbsLts::GetSuccsAndActions(implState,saa);
  //if the Buchi state is a sink state
  if(Buchi::IsSink(specState)) {
    if(statePaths.size() >= Database::MAX_GLOBAL_CE) {      
      list<BigInt> sp = statePath; sp.pop_back();
      list< list<BigInt> >::iterator si = statePaths.begin(),sm = statePaths.end();
      list< list<Action> >::iterator ai = actPaths.begin(),am = actPaths.end();
      size_t max = 0;
      for(;si != statePaths.end();++si,++ai) {
	if(max < si->size()) {
	  max = si->size();
	  sm = si; am = ai;
	}
      }
      if(sm != statePaths.end()) {
	if(max > sp.size()) {
	  statePaths.erase(sm); statePaths.push_back(sp);
	  actPaths.erase(am); actPaths.push_back(actPath);
	  ceStates.insert(sp.begin(),sp.end());
	}		    
      }
    } else {
      statePaths.push_back(statePath);
      statePaths.back().pop_back();
      actPaths.push_back(actPath);
      ceStates.insert(statePaths.back().begin(),statePaths.back().end());
    }
    error = true;
    done = (statePaths.size() >= Database::MAX_GLOBAL_CE) ? true : done;
    return;
  }
  //if the Buchi state is not a sink state
  for(map< Action,set<BigInt> >::const_iterator i = saa.begin();(!done) && (i != saa.end());++i) {
    if(ActCompatible(i->first,specState)) {
      set<int> specSuccs = Buchi::GetSuccs(specState);
      for(set<BigInt>::const_iterator j = i->second.begin();(!done) && (j != i->second.end());++j) {
	for(set<int>::const_iterator k = specSuccs.begin();(!done) && (k != specSuccs.end());++k) {
	  if(PropCompatible(*j,*k)) {
	    BigInt next = (*j) * ssn + (*k);
	    if(stack.count(next) != 0) {
	      if(statePaths.size() >= Database::MAX_GLOBAL_CE) {
		list<BigInt> sp = statePath; sp.push_back(next);
		list<Action> ap = actPath; ap.push_back(i->first);
		list< list<BigInt> >::iterator si = statePaths.begin(),sm = statePaths.end();
		list< list<Action> >::iterator ai = actPaths.begin(),am = actPaths.end();
		size_t max = 0;
		for(;si != statePaths.end();++si,++ai) {
		  if(max < si->size()) {
		    max = si->size();
		    sm = si; am = ai;
		  }
		}
		if(sm != statePaths.end()) {
		  if(max > sp.size()) {
		    statePaths.erase(sm); statePaths.push_back(sp);
		    actPaths.erase(am); actPaths.push_back(ap);
		    ceStates.insert(sp.begin(),sp.end());
		  }		    
		}
	      } else {
		statePaths.push_back(statePath);
		actPaths.push_back(actPath);
		statePaths.back().push_back(next);
		actPaths.back().push_back(i->first);
		ceStates.insert(statePaths.back().begin(),statePaths.back().end());
	      }
	      error = true;
	      done = (statePaths.size() >= Database::MAX_GLOBAL_CE) ? true : done;
	      return;
	    }
	    if((!done) && (visited2.count(next) == 0)) {
	      statePath.push_back(next);
	      actPath.push_back(i->first);
	      DFS2(next,statePath,actPath);
	      statePath.pop_back();
	      actPath.pop_back();
	    }
	  }
	}
      }
    }
  }
}

/*********************************************************************/
//try to create a shorter counterexample
/*********************************************************************/
void LtlChecker::CreateShorterCE(const BigInt &node,list<BigInt> &statePath,list<Action> &actPath)
{
  return;
  list< list<BigInt> > newStatePaths;
  list< list<Action> > newActPaths;
  list< list<BigInt> >::const_iterator sit1 = statePaths.begin();
  list< list<Action> >::const_iterator ait1 = actPaths.begin();
  for(;sit1 != statePaths.end();++sit1,++ait1) {
    list<BigInt> stoNode;
    list<Action> atoNode;
    bool flag = false;
    list<BigInt>::const_iterator sit2 = sit1->begin();
    list<Action>::const_iterator ait2 = ait1->begin();
    for(;sit2 != sit1->end();++sit2,++ait2) {
      if((*sit2) == node) {
	flag = true;
	break;
      } else if((*sit2) == sit1->back()) break;
      else {
	stoNode.push_back(*sit2);
	atoNode.push_back(*ait2);
      }
    }
    if(flag && (stoNode.size() > statePath.size())) {
      stoNode = statePath;
      atoNode = actPath;
      stoNode.push_back(node);
      ++sit2;
      stoNode.insert(stoNode.end(),sit2,sit1->end());
      atoNode.insert(atoNode.end(),ait2,ait1->end());
      assert((sit1->size() - stoNode.size()) == (ait1->size() - atoNode.size()));
      newStatePaths.push_back(stoNode);
      newActPaths.push_back(atoNode);
    } else {
      newStatePaths.push_back(*sit1);
      newActPaths.push_back(*ait1);
    }
  }
  statePaths = newStatePaths;
  actPaths = newActPaths;
  ceStates.clear();
  for(list< list<BigInt> >::const_iterator i = statePaths.begin();i != statePaths.end();++i) {
    ceStates.insert(i->begin(),i->end());
  }
}

/*********************************************************************/
//return true if an implementation state is propositionally compatible
//with a specification state. the two states are arguments.
/*********************************************************************/
bool LtlChecker::PropCompatible(const BigInt &implState,int specState)
{
  set<Expr> implProps;
  if(Database::CONF_TYPE == Database::CONF_SEAW) {
    const vector<BasicSeaw*> &args = SeawChecker::opSeaw.first->GetSubs();
    for(size_t i = 0;i < args.size();++i) {
      if(args[i]->GetType() != BasicSeaw::SEAW_ACTS) {
	if(SeawChecker::label[args[i]].count(implState) != 0) {
	  char buf[64]; snprintf(buf,64,"prop_%d",i + 1);
	  implProps.insert(ExprManager::GetIdExpr(buf));
	}
      }
    }
  } else {
    GlobalAbsLts::GetPropositions(implState,implProps);
  }
  const set<Expr> posProps = Buchi::GetPosProps(specState);
  const set<Expr> negProps = Buchi::GetNegProps(specState);
  for(set<Expr>::const_iterator i = posProps.begin();i != posProps.end();++i) {
    if(implProps.count(*i) == 0) return false;
  }
  for(set<Expr>::const_iterator i = negProps.begin();i != negProps.end();++i) {
    if(implProps.count(*i) != 0) return false;
  }  
  return true;
}

/*********************************************************************/
//return true if the specification state is compatible with the action
//i.e. either the action belongs to the label or its negation does not
/*********************************************************************/
bool LtlChecker::ActCompatible(const Action &act,int specState)
{  
  const set<Action> posActs = Buchi::GetPosActs(specState);
  const set<Action> negActs = Buchi::GetNegActs(specState);
  if(act.IsBcast() || act.IsSend() || act.IsRecv()) {
    Action newAct(act.GetChannel());
    if(posActs.empty()) {
      for(set<Action>::const_iterator i = negActs.begin();i != negActs.end();++i) {      
	if(newAct == (*i)) return false;
      }
      return true;
    } else {
      //assert(posActs.size() == 1);
      return ((posActs.size() == 1) && (newAct == *(posActs.begin())));
    }
  } else {
    if(posActs.empty()) {
      for(set<Action>::const_iterator i = negActs.begin();i != negActs.end();++i) {      
	if(act == (*i)) return false;
      }
      return true;
    } else {
      //assert(posActs.size() == 1);
      return ((posActs.size() == 1) && (act == *(posActs.begin())));
    }
  }
}

/*********************************************************************/
//end of LtlChecker.cpp
/*********************************************************************/
