/**************************** CPPHeaderFile ***************************

* FileName [LocalAbsLts.h]

* PackageName [main]

* Synopsis [Header file for LocalAbsLts class.]

* Description [This class encapsulates a local abstract LTS.]

* SeeAlso [LocalAbsLts.cpp]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University.  All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#ifndef __LOCALABSLTS_H__
#define __LOCALABSLTS_H__

namespace magic {

//other classes needed
class ProcManager;
typedef int L2AbsState;

class LocalAbsLts
{
 private:
  //the id of the proc manager corresponding to this lts
  size_t pmid;

  //the total number of abstract states in this iteration and the
  //previous iteration
  size_t stateNum,prevStateNum;

  //initial abstract states in this iteration and the previous iteration.
  set<L2AbsState> init,prevInit;

  //the map from states to propositions
  map< L2AbsState,set<Expr> > propMap;

  //abstract actions
  set<Action> acts;

  //the map from states to refusals
  map< L2AbsState,set<Action> > refMap;

  //the set of states whose successors might have changed due to
  //action guided refinement
  set<L2AbsState> succsChanged;

  //the abstract successor and predecessor transition relation
  map< L2AbsState,map< Action,set<L2AbsState> > > succMap;
  map< L2AbsState,map< Action,set<L2AbsState> > > predMap;

  //the abstract successor relation from the previous iteration
  map< L2AbsState,map< Action,set<L2AbsState> > > prevSuccMap;
  
 public:
  LocalAbsLts() {}
  LocalAbsLts(size_t p) { pmid = p; stateNum = 0; prevStateNum = 0; }
  LocalAbsLts(const LocalAbsLts &rhs) { *this = rhs; }

  const LocalAbsLts &operator = (const LocalAbsLts &rhs);

  size_t GetStateNum() const { return stateNum; }
  size_t GetPrevStateNum() const { return prevStateNum; }
  const set<int> &GetInit() const { return init; }
  const set<int> &GetPrevInit() const { return prevInit; }
  bool HasAct(const Action &act) const { return (acts.count(act) != 0); }
  const set<Action> &GetActs() const { return acts; }
  const set<int> &GetSuccsChanged() const { return succsChanged; }
  void GetPropositions(const L2AbsState &state,set<Expr> &res) const;  

  void GetActsOut(const L2AbsState &state,set<Action> &res) const;
  void GetActsIn(const L2AbsState &state,set<Action> &res) const;
  void GetSuccsOnAction(const L2AbsState &from,const Action &act,set<L2AbsState> &res);
  bool IsTrans(const L2AbsState &from,const Action &act,const L2AbsState &to);
  void GetPredsOnAction(const L2AbsState &from,const Action &act,set<L2AbsState> &res) const;
  void GetPrevSuccsOnAction(const L2AbsState &from,const Action &act,set<L2AbsState> &res) const;
  void GetRefusal(const L2AbsState &state,set<Action> &refusal) const;
  void Display();

  friend class ProcManager;
  friend class LtsAbsRefiner;
};

} //namespace magic

#endif //__LOCALABSLTS_H__

/*********************************************************************/
//end of LocalAbsLts.h
/*********************************************************************/
