/******************************** CPPFile *****************************

* FileName [LocalAbsLts.cpp]

* PackageName [main]

* Synopsis [Method definitions of LocalAbsLts class.]

* SeeAlso [LocalAbsLts.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cassert>
#include <gmp.h>
#include <string>
#include <list>
#include <vector>
#include <set>
#include <map>
using namespace std;

#include "BigInt.h"
#include "Util.h"
#include "Node.h"
#include "Action.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "ContLoc.h"
#include "ImplState.h"
#include "LocalAbsLts.h"
#include "ConcCEDag.h"
#include "Component.h"
#include "ProcManager.h"
using namespace magic;

/*********************************************************************/
//operators
/*********************************************************************/
const LocalAbsLts &LocalAbsLts::operator = (const LocalAbsLts &rhs)
{
  pmid = rhs.pmid;
  stateNum = rhs.stateNum;
  prevStateNum = rhs.prevStateNum;
  init = rhs.init;
  prevInit = rhs.prevInit;
  propMap = rhs.propMap;
  acts = rhs.acts;
  refMap = rhs.refMap;
  succsChanged = rhs.succsChanged;
  succMap = rhs.succMap;
  predMap = rhs.predMap;
  prevSuccMap = rhs.prevSuccMap;
  return *this;
}

/*********************************************************************/
//return the set of propositions true in the argument state
/*********************************************************************/
void LocalAbsLts::GetPropositions(const L2AbsState &state,set<Expr> &res) const
{
  map< int,set<Expr> >::const_iterator i = propMap.find(state);
  if(i == propMap.end()) res.clear();
  else res = i->second;
}

/*********************************************************************/
//given an abstract state return the set of actions enabled from it
/*********************************************************************/
void LocalAbsLts::GetActsOut(const L2AbsState &state,set<Action> &res) const
{
  map< L2AbsState,map< Action,set<L2AbsState> > >::const_iterator i = succMap.find(state);
  if(i != succMap.end()) {
    for(map< Action,set<L2AbsState> >::const_iterator j = i->second.begin();j != i->second.end();++j) {
      res.insert(j->first);
    }
  }
}

/*********************************************************************/
//given an abstract state return the set of actions incoming into it
/*********************************************************************/
void LocalAbsLts::GetActsIn(const L2AbsState &state,set<Action> &res) const
{
  map< L2AbsState,map< Action,set<L2AbsState> > >::const_iterator i = predMap.find(state);
  if(i != predMap.end()) {
    for(map< Action,set<L2AbsState> >::const_iterator j = i->second.begin();j != i->second.end();++j) {
      res.insert(j->first);
    }
  }
}

/*********************************************************************/
//given an abstract state and an action return the successor abstract
//states
/*********************************************************************/
void LocalAbsLts::GetSuccsOnAction(const L2AbsState &from,const Action &act,set<L2AbsState> &res)
{
  //if successors of this state have not been computed
  if(succMap.count(from) == 0) {
    const ProcManager &pmgr = *(static_cast<ProcManager*>(Database::components[pmid]));
    for(set<const ImplState*>::const_iterator i = pmgr.GetReachImpl().begin();i != pmgr.GetReachImpl().end();++i) {
      if((*i)->GetAbsId() == from) {
	map< Action,set<const ImplState*> > sa;
	(*i)->GetSuccsAndActions(set<Action>(),set<Action>(),sa);
	for(map< Action,set<const ImplState*> >::const_iterator k = sa.begin();k != sa.end();++k) {
	  for(set<const ImplState*>::const_iterator l = k->second.begin();l != k->second.end();++l) {
	    succMap[from][k->first].insert((*l)->GetAbsId());
	    predMap[(*l)->GetAbsId()][k->first].insert(from);
	  }
	}
      }
    }
  } 
  map< int,map< Action,set<int> > >::const_iterator i = succMap.find(from);
  if(i == succMap.end()) res.clear();
  else {
    map< Action,set<int> >::const_iterator j = i->second.find(act);
    if(j == i->second.end()) res.clear();
    else res = j->second;
  }
}

/*********************************************************************/
//check validity of a transition
/*********************************************************************/
bool LocalAbsLts::IsTrans(const L2AbsState &from,const Action &act,const L2AbsState &to)
{
  //if successors of this state have not been computed
  if(succMap.count(from) == 0) {
    const ProcManager &pmgr = *(static_cast<ProcManager*>(Database::components[pmid]));
    for(set<const ImplState*>::const_iterator i = pmgr.GetReachImpl().begin();i != pmgr.GetReachImpl().end();++i) {
      if((*i)->GetAbsId() == from) {
	map< Action,set<const ImplState*> > sa;
	(*i)->GetSuccsAndActions(set<Action>(),set<Action>(),sa);
	for(map< Action,set<const ImplState*> >::const_iterator k = sa.begin();k != sa.end();++k) {
	  for(set<const ImplState*>::const_iterator l = k->second.begin();l != k->second.end();++l) {
	    succMap[from][k->first].insert((*l)->GetAbsId());
	    predMap[(*l)->GetAbsId()][k->first].insert(from);
	  }
	}
      }
    }
  } 
  map< int,map< Action,set<int> > >::const_iterator i = succMap.find(from);
  if(i == succMap.end()) return false;
  map< Action,set<int> >::const_iterator j = i->second.find(act);
  if(j == i->second.end()) return false;
  else return (j->second.count(to) != 0);
}

/*********************************************************************/
//given an abstract state and an action return the predecessor
//abstract states
/*********************************************************************/
void LocalAbsLts::GetPredsOnAction(const L2AbsState &from,const Action &act,set<L2AbsState> &res) const
{
  map< int,map< Action,set<int> > >::const_iterator i = predMap.find(from);
  if(i == predMap.end()) res.clear();
  else {
    map< Action,set<int> >::const_iterator j = i->second.find(act);
    if(j == i->second.end()) res.clear();
    else res = j->second;
  }
}

/*********************************************************************/
//given an abstract state and an action return the successor abstract
//states in the previous iteration
/*********************************************************************/
void LocalAbsLts::GetPrevSuccsOnAction(const L2AbsState &from,const Action &act,set<L2AbsState> &res) const
{
  map< int,map< Action,set<int> > >::const_iterator i = prevSuccMap.find(from);
  if(i == prevSuccMap.end()) res.clear();
  else {
    map< Action,set<int> >::const_iterator j = i->second.find(act);
    if(j == i->second.end()) res.clear();
    else res = j->second;
  }
}

/*********************************************************************/
//given an abstract state, return its refusal set
/*********************************************************************/
void LocalAbsLts::GetRefusal(const L2AbsState &state,set<Action> &refusal) const
{
  map< L2AbsState,set<Action> >::const_iterator i = refMap.find(state);
  assert(i != refMap.end());
  refusal = i->second;
}

/*********************************************************************/
//display
/*********************************************************************/
void LocalAbsLts::Display()
{
  Util::Message(2,"******************* state ******************\n");
  const ProcManager &pmgr = *(static_cast<ProcManager*>(Database::components[pmid]));
  for(size_t j = 0;j < stateNum;++j) {
    for(set<const ImplState*>::const_iterator k = pmgr.GetReachImpl().begin();k != pmgr.GetReachImpl().end();++k) {
      if((*k)->GetAbsId() == (int)(j)) Util::Message(2,"%d : %s\n",j,(*k)->ToString().c_str());
    }
  }
  Util::Message(2,"******************* inits ******************\n");
  for(set<L2AbsState>::const_iterator i = init.begin();i != init.end();++i) {
    Util::Message(2,"%d ",*i);
  }
  Util::Message(2,"\n");
  Util::Message(2,"****************** actions *****************\n");
  for(set<Action>::const_iterator i = acts.begin();i != acts.end();++i) {
    Util::Message(2,"%s ",i->ToString().c_str());
  }
  Util::Message(2,"\n");
  Util::Message(2,"******************* trans ******************\n");
  for(set<Action>::const_iterator i = acts.begin();i != acts.end();++i) {
    for(size_t j = 0;j < stateNum;++j) {
      set<int> x; GetSuccsOnAction(j,*i,x);
      for(set<int>::const_iterator k = x.begin();k != x.end();++k) {
       	Util::Message(2,"%d === %s ===> %d\n",j,i->ToString().c_str(),*k);
      }
    }
  }
  Util::Message(2,"********************************************\n");
}

/*********************************************************************/
//end of LocalAbsLts.cpp
/*********************************************************************/
