/******************************** CPPFile *****************************

* FileName [InputProcessor.cpp]

* PackageName [main]

* Synopsis [This file contains method definitions of the
* InputProcessor class.]

* SeeAlso [InputProcessor.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <gmp.h>
#include <string>
#include <list>
#include <map>
#include <vector>
#include <set>
using namespace std;

#include "BigInt.h"
#include "Util.h"
#include "Node.h"
#include "LtsInfo.h"
#include "DFSAdapter.h"
#include "ProgAbs.h"
#include "ProgInfo.h"
#include "Action.h"
#include "SeawManager.h"
#include "ProcAbs.h"
#include "ProcInfo.h"
#include "LtsTrans.h"
#include "FspInfo.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "ContLoc.h"
#include "CallGraph.h"
#include "CallGraphCreator.h"
#include "ContLocExtractor.h"
#include "ProcInliner.h"
#include "AbsRefiner.h"
#include "LocalAbsLts.h"
#include "ConcCEDag.h"
#include "Component.h"
#include "ProcManager.h"
#include "TheoremProver.h"
#include "LtlFormula.h"
#include "LtlManager.h"
#include "Buchi.h"
#include "LtlToBuchi.h"
#include "LtsaChecker.h"
#include "InputProcessor.h"
#ifdef MAGIC_FULL
#include "PaccAPI.h"
#endif //MAGIC_FULL
using namespace magic;

/*********************************************************************/
//do all the stuff
/*********************************************************************/
bool InputProcessor::Run()
{
  //parse command line arguments and check if there is some
  //abstraction to be verified
  Database::ParseCommandLine();
  //restore if necessary
  if(Database::RESTORE) Restore();
  //initialise the theorem prover
  TheoremProver::Initialise();
  //initialise the SE-AW formula manager
  SeawManager::Initialise();
  //open the output dot file if necessary
  if(Database::CREATE_DOT_FILE) {
    assert(Database::dotFilePtr == NULL);
    Database::dotFilePtr = fopen(Database::DOT_FILE_NAME.c_str(),"w");
    assert(Database::dotFilePtr != NULL);    
    fprintf(Database::dotFilePtr,"digraph magic_dot_graph {\n\n");
    fprintf(Database::dotFilePtr,"  node [color=blue];\n");
    fprintf(Database::dotFilePtr,"  edge [color=red];\n\n");
  }
  //open the output model file if necessary
  if(Database::OUTPUT_MODELS) {
    assert(Database::modelFilePtr == NULL);
    Database::modelFilePtr = fopen(Database::modelFileName.c_str(),"w");
    assert(Database::modelFilePtr != NULL);    
  }
  //parse the input files and return if this is all to be done
  Database::ParseFiles();
  if(Database::PARSE_ONLY) return false;
  //if we need to simply translate ltl formulae to buchi automata
  if(Database::LTL_TO_BUCHI) {
    LtlToBuchi::Run();
    return false;
  }
  //if no abstractions to be verified
  if(Database::ABSTRACTION_NAME == "") return false;
  //extract the information about the procedures
  Database::ExtractProcInfos();
  //get the abstraction that needs to be checked
  for(map<string,ProgInfo>::const_iterator i = Database::progInfos.begin();
      i != Database::progInfos.end();++i) {
    const list<ProgAbs> &pabs = i->second.GetAbstractions();
    for(list<ProgAbs>::const_iterator j = pabs.begin();j != pabs.end();++j) {
      if(Database::ABSTRACTION_NAME == j->GetName()) {       
	Database::specLtsName = j->GetLtsName();
	//if checking for invalid pointer dereference
	if(Database::CHECK_INVALID_PTR_DEREF) {
	  Database::CONF_TYPE = Database::CONF_REACH;
	  Database::specLtsName = Database::INVALID_PTR_DEREF_SPEC;
	}
	//if checking for assertion failure
	else if(Database::CHECK_ASSERTION_FAILURE) {
	  Database::CONF_TYPE = Database::CONF_REACH;
	  Database::specLtsName = Database::ASSERTION_FAILURE_SPEC;
	}
	//convert trace containment to reachability
	else if(Database::CONF_TYPE == Database::CONF_TRACE) {
	  Database::CONF_TYPE = Database::CONF_REACH;
	  const LtsInfo &negLts = Database::GetSpecLtsNeg(Database::specLtsName);
	  //save original spec name for use by LTSA
	  if(Database::CONF_CHECK_MECHANISM == Database::CONF_CHECK_LTSA) {
	    LtsaChecker::SetSpecLtsName(Database::specLtsName);
	  }
	  Database::specLtsName = negLts.GetName();
	}
	//get the program name and the names of the root procedures
	Database::progName = i->first;
	Database::rootProcNames = i->second.GetProcs();
	bool res = false;
	//if the implementation is described in C
	if(Database::IMPL_TYPE == Database::IMPL_C) res = RunC(j);
#ifdef MAGIC_FULL
	//if the implementation is described as a PACC process
	else if(Database::IMPL_TYPE == Database::IMPL_PACC) res = RunPacc();
#endif //MAGIC_FULL
	//impossible
	else assert(false);
	//initialise the buchi automaton
	if(Database::CONF_TYPE == Database::CONF_LTL) {
	  //update the LTL formula with fair loop constraints
	  LtlFormula specFormula = Database::GetLtlFormula(Database::specLtsName);
          set< pair<Action,Action> > fairLoopActs;
          for(vector<Component*>::const_iterator i = Database::components.begin();i != Database::components.end();++i) {
            set< pair<Action,Action> > x;
            (*i)->GetFairLoopActs(x);
            fairLoopActs.insert(x.begin(),x.end());
          }
          for(set< pair<Action,Action> >::const_iterator i = fairLoopActs.begin();i != fairLoopActs.end();++i) {
	    //then and else
	    LtlFormula tform = LtlManager::GetActLtl(i->first);
	    LtlFormula eform = LtlManager::GetActLtl(i->second);
	    //(F then) and (F else)
	    LtlFormula ftform = LtlManager::GetUnaryLtl(tform,BasicLtl::LTL_F);
	    LtlFormula feform = LtlManager::GetUnaryLtl(eform,BasicLtl::LTL_F);
	    //(X F then) and (X F else)
	    LtlFormula xftform = LtlManager::GetUnaryLtl(ftform,BasicLtl::LTL_X);
	    LtlFormula xfeform = LtlManager::GetUnaryLtl(feform,BasicLtl::LTL_X);
	    //(then & X F then) and (else & X F else)
	    LtlFormula txft = LtlManager::GetBinaryLtl(tform,xftform,BasicLtl::LTL_AND);
	    LtlFormula exfe = LtlManager::GetBinaryLtl(eform,xfeform,BasicLtl::LTL_AND);
	    //(then & X F then) => (F else)
	    LtlFormula ntxft = LtlManager::GetUnaryLtl(txft,BasicLtl::LTL_NOT);
	    LtlFormula timpl = LtlManager::GetBinaryLtl(ntxft,feform,BasicLtl::LTL_OR);
	    //(else & X F else) => (F then)
	    LtlFormula nexfe = LtlManager::GetUnaryLtl(exfe,BasicLtl::LTL_NOT);
	    LtlFormula eimpl = LtlManager::GetBinaryLtl(nexfe,ftform,BasicLtl::LTL_OR);
	    //G((then & X F then) => (F else))
	    LtlFormula tglob = LtlManager::GetUnaryLtl(timpl,BasicLtl::LTL_G);
	    //G((else & X F else) => (F then))
	    LtlFormula eglob = LtlManager::GetUnaryLtl(eimpl,BasicLtl::LTL_G);
	    //G((then & X F then) => (F else)) & G((else & X F else) => (F then))
	    LtlFormula andf = LtlManager::GetBinaryLtl(tglob,eglob,BasicLtl::LTL_AND);
	    //implication
	    LtlFormula negf = LtlManager::GetUnaryLtl(andf,BasicLtl::LTL_NOT);
	    specFormula = LtlManager::GetBinaryLtl(negf,specFormula,BasicLtl::LTL_OR);
          }
	  //create the buchi automaton
	  int res = Buchi::Initialize(specFormula);
	  if(res == Buchi::LTL_TAUTOLOGY) {
	    Util::Error("LTL formula is a tautology ...\n");
	  } else if(res == Buchi::LTL_UNSATISFIABLE) {
	    Util::Error("LTL formula is unsatisfiable ...\n");
	  }
	  Buchi::Display();
	  Util::Message(2,"Buchi automaton constructed ...\n");
	}	
	//all done
	return res;
      }
    }
  }
  //abstraction to be checked not found
  return false;
}

/*********************************************************************/
//restore state from a backup file
/*********************************************************************/
void InputProcessor::Restore()
{
  //reset data structures
  Database::FILE_NAMES.clear();
  //open restore FILE
  Backupin = fopen(Database::restoreFileName.c_str(),"r");
  assert(Backupin != NULL);  
  //parse restore file
  Backupparse();
  //close restore file
  fclose(Backupin);
}

/*********************************************************************/
//process C implementations
/*********************************************************************/
bool InputProcessor::RunC(list<ProgAbs>::const_iterator &pait)
{
  //construct the call graph for the root procedures
  CallGraph callGraph;
  ConstructCallGraph(callGraph);
  
  //get the nodes of the call graph
  const set<string> &scc = callGraph.GetNodes();
  
  //if there are no procedures in the call graph we are done
  if(scc.empty()) return false;
  
  //create the control locations for the procedures
  for(set<string>::const_iterator k = scc.begin();k != scc.end();++k) {
    //create the control locations
    ProcInfo &y = Database::procInfos[*k];
    assert(y.GetContLocs().empty());
    ContLocExtractor cle(*k);
    y.GetAst().Apply(cle);
    //y.DisplayContLocs();
  }
  Util::Message(2,"control locations created ...\n");
  
  //remove unnecessary control locations - cases, defaults,
  //labels, gotos and breaks. these act as skip locations.
  RemoveSkipLocations(callGraph);
  Util::Message(2,"skip locations eliminated ...\n");
  
  //remove useless branches i.e. branches whose true and false
  //successors are the same
  RemoveUselessBranches(callGraph);
  Util::Message(2,"useless branch locations eliminated ...\n");
  
  //inline procedure bodies at unabstracted call sites and
  //associate info about spec LTSs to abstracted call sites
  ProcInliner::Inline(callGraph);
  Util::Message(2,"procedures inlined ...\n");
  
  //create the managers for the root procedures. also create the
  //array of all control locations for each root procedure and
  //backup fields and compute predecessors.
  const vector<Expr> &absGuards = pait->GetGuards();
  assert(Database::rootProcNames.size() == absGuards.size());
  for(size_t k = 0;k < Database::rootProcNames.size();++k) {
    //create the list of parameters and local variables
    const list<string> &pars =  Database::procInfos[Database::rootProcNames[k]].GetParams();
    const list<string> &locs =  Database::procInfos[Database::rootProcNames[k]].GetLocals();
    char nsTag[32];
    snprintf(nsTag,32,"P%d::",k);
    list<string> params;
    for(list<string>::const_iterator l = pars.begin();l != pars.end();++l) {
      params.push_back(nsTag + (*l));
    }
    list<string> locals;
    for(list<string>::const_iterator l = locs.begin();l != locs.end();++l) {
      locals.push_back(nsTag + (*l));
    }
    //create the guard
    list<Expr> repList; ExprManager::StringListToExprList(params,repList);
    Expr absGuard = ExprManager::ReplaceDummyVarsOne(absGuards[k],repList);
    //create the proc manager
    ProcManager *procManager = new ProcManager(k,Database::rootProcNames[k],params,locals,absGuard);
    procManager->Initialise();
    procManager->CreateAllContLocs();
    Database::components.push_back(procManager);
    Util::Message(2,"procedure manager for " + Database::rootProcNames[k] + " created ...\n");
  }
  
  //free control locations in the procedure infos
  for(map<string,ProcInfo>::iterator k = Database::procInfos.begin();k != Database::procInfos.end();++k) {
    k->second.Cleanup();
  }
  
  //compute relevant LTL or SE-AW propositions
  if((Database::CONF_TYPE == Database::CONF_LTL) || (Database::CONF_TYPE == Database::CONF_SEAW)) {
    vector< set<Expr> > props(Database::components.size());
    set<Expr> specProps;
    if(Database::CONF_TYPE == Database::CONF_LTL) {
      specProps = LtlManager::GetProps(Database::GetLtlFormula(Database::specLtsName));
    } else {
      specProps = SeawManager::GetProps(Database::specLtsName);
    }
    for(set<Expr>::const_iterator i = specProps.begin();i != specProps.end();++i) {
      set<int> localComps;
      set<string> lvs = Util::ComputeIdLvalues(i->GetExpr());
      for(set<string>::const_iterator j = lvs.begin();j != lvs.end();++j) {
	size_t pos;
	if((pos = j->find(':')) != string::npos) {
	  string x = j->substr(1,pos - 1);
	  localComps.insert(atoi(x.c_str()));
	}
      }
      if(localComps.size() == 1) {
	size_t x = *(localComps.begin());
	assert(x < Database::components.size());
	props[x].insert(*i);
      } else {
	assert(localComps.empty());
	props[0].insert(*i);
      }
    }
    for(size_t k = 0;k < Database::components.size();++k) {
      static_cast<ProcManager*>(Database::components[k])->SetProps(props[k]);
    }
    Util::Message(2,"relevant propositions computed ...\n");
  }
  
  //do the live variable analysis for the control locations
  if(Database::USE_LIVE_VARS) {
    DoLiveVarAnalysis();
    Util::Message(2,"live variable analysis done ...\n");
  }	
  return true;
}

/*********************************************************************/
//construct the call graph of the root procedures
/*********************************************************************/
void InputProcessor::ConstructCallGraph(CallGraph &callGraph)
{
  //the list of processes to be handled
  list<string> workSet;
  workSet.insert(workSet.end(),Database::rootProcNames.begin(),Database::rootProcNames.end());
  //the set of processes already handled
  set<string> done;
 
  //process till no more procedures left
  while(!workSet.empty()) {
    string x = workSet.front();
    workSet.pop_front();
    //check if it has been processed already
    if(done.count(x) != 0) continue;
    done.insert(x);
    //get the info about the procedure and simplify it
    if(Database::procInfos.count(x) == 0) {
      Util::Error("procedure " + x + " reachable from but not defined ...\n");
    }
    //get the proc info
    ProcInfo &y = Database::procInfos[x];
    //add a node for this procedure to the call graph
    callGraph.AddNode(x);
    //get the names of procedures called by the current
    //procedure and are not abstracted by actions
    CallGraphCreator cgc(y);
    y.GetAst().Apply(cgc);
    set<string> a = cgc.GetCalledProcs();
    //add these called procedure names to the workset and the edges
    for(set<string>::const_iterator i = a.begin();i != a.end();++i) {
      workSet.push_back(*i);
      callGraph.AddEdge(x,*i);
    }
  }
}

/*********************************************************************/
//remove unnecessary control locations - cases, defaults, labels,
//gotos and breaks. these act as skip locations and need not be
//analysed.
/*********************************************************************/
void InputProcessor::RemoveSkipLocations(const CallGraph &callGraph)
{
  const set<string> &scc = callGraph.GetNodes();
  
  //redirect the predecessors of the skip locations to the
  //successors of the skip locations.
  for(set<string>::const_iterator i = scc.begin();i != scc.end();++i) {
    ProcInfo &b = Database::procInfos[*i];

    //set the initial location
    if(b.GetInitLoc()->IsSkipLoc()) {
      b.SetInitLoc(GetClosestNonSkipLoc(b.GetInitLoc()));
    }

    for(list<ContLoc*>::iterator j = b.GetContLocs().begin();j != b.GetContLocs().end();++j) {
      ContLoc *c = *j;
      if(c->IsSkipLoc()) continue;
      if((c->GetLocType() == ContLoc::FINAL) || (c->GetLocType() == ContLoc::RETURN)) {
      } else if(c->GetLocType() == ContLoc::ASSIGN_EXPR) {
	c->SetTSucc(GetClosestNonSkipLoc(c->GetTSucc()));
      } else if(c->GetLocType() == ContLoc::ASSIGN_CALL) {
	c->SetTSucc(GetClosestNonSkipLoc(c->GetTSucc()));
	ContLoc *d = c->GetCallSucc();
	if(d != NULL) {
	  c->SetCallSucc(GetClosestNonSkipLoc(d));
	}
      } else if(c->GetLocType() == ContLoc::BRANCH) {
	c->SetTSucc(GetClosestNonSkipLoc(c->GetTSucc()));
	c->SetESucc(GetClosestNonSkipLoc(c->GetESucc()));
      } else Util::Error("control location of unknown type ...\n");
    }
  }
  //now remove all skip locations
  for(set<string>::const_iterator i = scc.begin();i != scc.end();++i) {
    ProcInfo &b = Database::procInfos[*i];
    for(list<ContLoc*>::iterator j = b.GetContLocs().begin();j != b.GetContLocs().end();) {
      if((*j)->IsSkipLoc()) {
	delete (*j);
	j = b.GetContLocs().erase(j);
      } else ++j;
    }
  }
}

/*********************************************************************/
//given a control location, if it is non-skip return it. otherwise
//return the non-skip location closest to it. if there is a loop of
//skip locations including this one then it throws an error.
/*********************************************************************/
ContLoc *InputProcessor::GetClosestNonSkipLoc(ContLoc *arg)
{
  set<ContLoc*> seen;
  while(true) {
    //if this is non-skip
    if(!arg->IsSkipLoc()) return arg;
    //otherwise check for skip loop
    if(seen.count(arg) != 0) {
      Util::Error("skip location loop detected ...\n");
    }
    seen.insert(arg);
    arg = arg->GetTSucc();
  }
}

/*********************************************************************/
//eliminate branch locations whose true and false successors are
//identical
/*********************************************************************/
void InputProcessor::RemoveUselessBranches(const CallGraph &callGraph)
{
  const set<string> &scc = callGraph.GetNodes();
 
  for(set<string>::const_iterator i = scc.begin();i != scc.end();++i) {
    ProcInfo &b = Database::procInfos[*i];
    while(true) {
      //get a useless branch
      ContLoc* ulb = NULL;
      for(list<ContLoc*>::iterator j = b.GetContLocs().begin();j != b.GetContLocs().end();++j) {
	if(((*j)->GetLocType() == ContLoc::BRANCH) && ((*j)->GetTSucc() == (*j)->GetESucc())) {
	  assert((*j) != (*j)->GetTSucc());
	  ulb = *j;
	  break;
	}
      }

      //if there are no useless branches we are done
      if(ulb == NULL) break;

      //set the initial location
      if(b.GetInitLoc() == ulb) {
	b.SetInitLoc(b.GetInitLoc()->GetTSucc());
      }

      //redirect all predecessors of useless branches to the
      //successors
      for(list<ContLoc*>::iterator j = b.GetContLocs().begin();j != b.GetContLocs().end();++j) {
	ContLoc *loc = (*j);
	if(loc == ulb) continue;
	int locType = loc->GetLocType();
	if((locType == ContLoc::FINAL) || (locType == ContLoc::RETURN)) {
	} else if(locType == ContLoc::ASSIGN_EXPR) {
	  if(loc->GetTSucc() == ulb) {
	    loc->SetTSucc(loc->GetTSucc()->GetTSucc());
	    assert(loc != loc->GetTSucc());
	  }
	} else if(locType == ContLoc::ASSIGN_CALL) {
	  if(loc->GetTSucc() == ulb) {
	    loc->SetTSucc(loc->GetTSucc()->GetTSucc());
	    assert(loc != loc->GetTSucc());
	  }
	  if(loc->GetCallSucc() == ulb) {
	    loc->SetCallSucc(loc->GetCallSucc()->GetTSucc());
	    assert(loc != loc->GetCallSucc());
	  }
	} else if(locType == ContLoc::BRANCH) {
	  if(loc->GetTSucc() == ulb) {
	    loc->SetTSucc(loc->GetTSucc()->GetTSucc());
	    assert(loc != loc->GetTSucc());
	  }
	  if(loc->GetESucc() == ulb) {
	    loc->SetESucc(loc->GetESucc()->GetTSucc());
	    assert(loc != loc->GetESucc());
	  }
	} else assert(false);
      }

      //remove all the useless branches
      for(list<ContLoc*>::iterator j = b.GetContLocs().begin();j != b.GetContLocs().end();++j) {
	if((*j) == ulb) {
	  delete (*j);
	  b.GetContLocs().erase(j);
	  break;
	}
      }
    }
  }
}

/*********************************************************************/
//do the live variable analysis for each root procedure
/*********************************************************************/
void InputProcessor::DoLiveVarAnalysis()
{
  for(vector<Component*>::const_iterator i = Database::components.begin();i != Database::components.end();++i) {
    static_cast<ProcManager*>(*i)->DoLiveVariableAnalysis();
  }
}

/*********************************************************************/
//process PACC implementations
/*********************************************************************/
#ifdef MAGIC_FULL
bool InputProcessor::RunPacc()
{
  for(size_t i = 0;i < Database::rootProcNames.size();++i) {
    Database::components.push_back(new PaccProcess(i,Database::rootProcNames[i]));
    if(!Database::components[i]->Initialise()) return false;
  }
  return true;
}
#endif //MAGIC_FULL

/*********************************************************************/
//end of InputProcessor.cpp
/*********************************************************************/
