/******************************** CPPFile *****************************

* FileName [ImplState.cpp]

* PackageName [main]

* Synopsis [Method definitions of ImplState class.]

* SeeAlso [ImplState.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <gmp.h>
#include <string>
#include <list>
#include <set>
#include <map>
#include <vector>
#include <typeinfo>
using namespace std;

#include "BigInt.h"
#include "Util.h"
#include "Node.h"
#include "ProcAbs.h"
#include "Action.h"
#include "ActionManager.h"
#include "LtsTrans.h"
#include "FspInfo.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "ProcInfo.h"
#include "LtsInfo.h"
#include "ImplState.h"
#include "ContLoc.h"
#include "LocalAbsLts.h"
#include "ConcCEDag.h"
#include "Component.h"
#include "ProcManager.h"
#define YYSTYPE int
#include "StdcParser.h"
using namespace magic;

/*********************************************************************/
//constructors
/*********************************************************************/
ImplState::ImplState(short c,const PVal &p,short ss,char si)
  : contLocId(c),predVal(p),specStateId(ss),specInfoId(si),absId(-1) {}

ImplState::ImplState(const ImplState &rhs)
  : contLocId(rhs.contLocId),predVal(rhs.predVal),specStateId(rhs.specStateId),
    specInfoId(rhs.specInfoId),absId(rhs.absId) {}

/*********************************************************************/
//operators
/*********************************************************************/
const ImplState &ImplState::operator = (const ImplState &rhs)
{
  contLocId = rhs.contLocId;
  predVal = rhs.predVal;
  specStateId = rhs.specStateId;
  specInfoId = rhs.specInfoId;
  absId = rhs.absId;
  return *this;
}

/*********************************************************************/
//return the set of specification propositions true in this state
/*********************************************************************/
void ImplState::GetPropositions(set<Expr> &res) const
{
  GetContLoc()->GetPropositions(GetKey(),res);
}

/*********************************************************************/
//return the refusal set of this state
/*********************************************************************/
void ImplState::GetRefusal(set<Action> &res) const
{
  const set<Action> &acts = GetProcManager().GetAbsLts().GetActs();
  for(set<Action>::const_iterator i = acts.begin();i != acts.end();++i) {
    set<const ImplState*> x; GetSuccsOnAction(*i,x);
    if(x.empty()) res.insert(*i);
  }
}

/*********************************************************************/
//return the set of successors of this state alongwith the
//corresponding actions
/*********************************************************************/
void ImplState::GetSuccsAndActions(const set<Action> &globActs,const set<Action> &chanActs,map< Action,set<const ImplState*> > &res) const
{
  //first check if the successors and actions have been already
  //computed before
  ContLoc *contLoc = GetContLoc();
  string key = GetKey();
  if((Database::PRECOMPUTE_IMPL_TRANS) && globActs.empty() && chanActs.empty()) {
    if(contLoc->ContainsSuccsAndActions(key)) {
      res = contLoc->GetSuccsAndActions(key);
      return;
    }
  }  
  //last control location
  if(contLoc->GetLocType() == ContLoc::FINAL) {
    GetSuccsAndActionsFinal(globActs,chanActs,res);
  }
  //return location
  else if(contLoc->GetLocType() == ContLoc::RETURN) {
    GetSuccsAndActionsReturn(globActs,chanActs,res);
  }
  //branch statement
  else if(contLoc->GetLocType() == ContLoc::BRANCH) {
    GetSuccsAndActionsBranch(globActs,chanActs,res);
  }
  //assignment statement with rhs not a procedure call
  else if(contLoc->GetLocType() == ContLoc::ASSIGN_EXPR) {
    GetSuccsAndActionsAssign(globActs,chanActs,res);
  }
  //assignment statement with rhs being a procedure call
  else if(contLoc->GetLocType() == ContLoc::ASSIGN_CALL) {
    GetSuccsAndActionsCall(globActs,chanActs,res);
  }
  //unknown 
  else assert(false);
  //update the control location map
  if(Database::PRECOMPUTE_IMPL_TRANS) {
    contLoc->AddSuccsAndActions(key,res);
  }
}

/*********************************************************************/
//return the set of successors of this state on the argument action
/*********************************************************************/
void ImplState::GetSuccsOnAction(const Action &act,set<const ImplState*> &succs) const
{
  map< Action,set<const ImplState*> > allSuccs;
  GetSuccsAndActions(set<Action>(),set<Action>(),allSuccs);
  map< Action,set<const ImplState*> >::const_iterator i = allSuccs.find(act);
  if(i == allSuccs.end()) succs.clear();
  else succs = i->second;
}

/*********************************************************************/
//return the set of predecessors of this state on the argument action
/*********************************************************************/
void ImplState::GetPredsOnAction(const Action &act,set<const ImplState*> &preds) const
{
  map< Action,set<const ImplState*> > allPreds = GetContLoc()->GetPredsAndActions(GetKey());
  map< Action,set<const ImplState*> >::const_iterator i = allPreds.find(act);
  if(i == allPreds.end()) preds.clear();
  else preds = i->second;
}

/*********************************************************************/
//create the key for this implementation state by appending the
//predicate valuation, a colon and the inlined spec LTS state. the key
//is used to map the successors and actions of the implementation
//state in the map associated with the control location for this
//implementation state.
/*********************************************************************/
string ImplState::GetKey() const
{
  if(specStateId == -1) {
    return predVal.ToString() + ":null";
  } else {
    char x[100];
    snprintf(x,100,":%d:%d",specStateId,(short)(specInfoId));
    return predVal.ToString() + x;
  }
}

/*********************************************************************/
//return successors of the final control location
/*********************************************************************/
void ImplState::GetSuccsAndActionsFinal(const set<Action> &globActs,const set<Action> &chanActs,map< Action,set<const ImplState*> > &res) const
{
  //if we are doing ltl model checking or deadlock detection or SE-AW
  //model checking and this is the first parallel component then allow
  //the final location to stutter
  if((Database::CONF_TYPE == Database::CONF_LTL) || (Database::CONF_TYPE == Database::CONF_DLOCK) ||
     (Database::CONF_TYPE == Database::CONF_SEAW)) {
    if(GetProcManagerId() == 0) {
      set<Action> acts; acts.insert(ActionManager::GetImplEpsilonAction(GetProcManagerId()));
      AddSuccAndActions(this,acts,res);
    }
  }
  //otherwise the final location has no successors. if there are any
  //return successors they will be handled by the return location
  else assert((Database::CONF_TYPE == Database::CONF_SIMUL) || (Database::CONF_TYPE == Database::CONF_REACH));
}

/*********************************************************************/
//create the successors and actions when the next control locations is
//in the same procedure and there is an additional constraint apart
//from those due to the predicate valuation. the first argument is the
//successor control location. the second argument is the constraint. a
//NULL value indicates that there is no additional constraint. the
//third argument is the set of actions. the fourth argument is the
//list where the results are collected.
/*********************************************************************/
void ImplState::HandleConstraint(ContLoc *succ,const Expr &cons,const set<Action> &acts,
				 map< Action,set<const ImplState*> > &res) const
{
  const ContLoc *contLoc = GetContLoc();
  const PredSet &preds = contLoc->GetPreds();
  set<Expr> state; preds.ToExpr(preds.PValToPredVal(predVal),state);
  
  set<Expr> negList;
  if(!cons.IsNull()) negList.insert(ExprManager::NegateExpr(cons));  

  if(!ExprManager::ProveImplies(state,negList)) {
    if(!cons.IsNull()) state.insert(cons);
    const PredSet &succPreds = succ->GetPreds();   
    //get the set of possible predicate valuations in the successor
    //assuming there is no assignment.
    vector<PVal> pvals;
    succPreds.GetPossibleValuations(preds,predVal,list<Expr>(),list<Expr>(),pvals);
    //try each possibility
    for(size_t i = 0;i < pvals.size();++i) {
      const PVal &pval = pvals[i];
      //get the expression for the successor state
      set<Expr> expr; succPreds.ToExpr(succPreds.PValToPredVal(pval),expr);
      //negate the expressions
      set<Expr> neg; ExprManager::NegateExprSet(expr,neg);
      //if the constraints cannot imply this negation then we
      //have to consider this predicate valuation as a possible
      //successor
      if(!ExprManager::ProveImplies(state,neg)) {
	set<const ImplState*> a = succ->GetImplStates(pval);
	for(set<const ImplState*>::const_iterator j = a.begin();j != a.end();++j) {
	  AddSuccAndActions(*j,acts,res);
	}
      }
    }
  }
}

/*********************************************************************/
//create the successors and actions when the next control locations is
//in the same procedure and there is a parallel assignment. the first
//argument is the successor control location. the second argument is a
//set of expressions implicitly conjuncted that express any additional
//constraints on the state before the assignment.  the third and
//fourth arguments are the lhs-list and rhs-list of the
//assignment. the fifth argument is the set of actions. the sixth
//argument is the list where the results are collected. the last
//argument (if not equal to -1), indicates the id of the spec state to
//be associated with the successor states.
/*********************************************************************/
void ImplState::HandleAssign(ContLoc *succ,const set<Expr> &addCons,const list<Expr> &lhsList,const list<Expr> &rhsList,
			     const set<Action> &acts,
			     const short fsid,const short fiid,
			     map< Action,set<const ImplState*> > &res) const
{
  ContLoc *contLoc = GetContLoc();  
  //first compute the set of possible valuation of the successors
  PredSet &preds = contLoc->GetPreds();
  PredSet &succPreds = succ->GetPreds();
  vector<PVal> pvals;
  succPreds.GetPossibleValuations(preds,predVal,lhsList,rhsList,pvals);  
  //for each possible predicate valuation of the successor check
  //if it is a possible next state valuation. if so add the
  //corresponding successor impl state.
  for(size_t i = 0;i < pvals.size();++i) {
    const PVal &pval = pvals[i];
    if(PossibleNextValuation(addCons,lhsList,rhsList,preds,predVal,succPreds,pval)) {
      if(fsid == -1) {
	set<const ImplState*> implStates = succ->GetImplStates(pval);
	for(set<const ImplState*>::const_iterator j = implStates.begin();j != implStates.end();++j) {
	  AddSuccAndActions(*j,acts,res);
	}
      } else {
	AddSuccAndActions(succ->GetImplState(pval,fsid,fiid),acts,res);
      }
    }
  }
}

/*********************************************************************/
//add successors on global variable assignments
/*********************************************************************/
void ImplState::HandleGlobalVarAssigns(const set<Action> &globActs,const short fsid,const short fiid,map< Action,set<const ImplState*> > &res) const
{
  //variable to replace "$0" in the channel data constraint
  Expr rexpr = ExprManager::GetIdExpr(Util::NewTempVar());
  list<Expr> rlist; rlist.push_back(rexpr);
  //stutter on all global variable actions
  ContLoc *contLoc = GetContLoc();
  for(set<Action>::const_iterator i = globActs.begin();i != globActs.end();++i) {
    set<Action> acts; acts.insert(i->Complement());
    const Expr &lhs = i->GetLhsExpr();
    Expr cons = ExprManager::ReplaceDummyVarsZero(i->GetRhsExpr(),rlist);
    set<Expr> x; x.insert(cons);
    list<Expr> y; y.push_back(lhs);
    list<Expr> z; z.push_back(rexpr);
    HandleAssign(contLoc,x,y,z,acts,fsid,fiid,res);
  }
}

/*********************************************************************/
//add successors on invalid pointer dereferences
/*********************************************************************/
void ImplState::HandleInvalidPtrDeref(map< Action,set<const ImplState*> > &res) const
{
  ContLoc *contLoc = GetContLoc();
  const ProcManager &procManager = GetProcManager();
  if(contLoc == procManager.GetMarkedLoc()) {
    Expr vpp = ExprManager::GetIdExpr(Database::VALID_PTR_PRED);
    list<Expr> el; el.push_back(contLoc->GetIpdExpr());
    set<Expr> x; x.insert(ExprManager::GetParExpr(vpp,el));
    set<Expr> state; contLoc->GetPreds().ToExpr(contLoc->GetPreds().PValToPredVal(predVal),state);
    if(!ExprManager::ProveImplies(state,x)) {
      set<Action> acts;
      acts.insert(ActionManager::GetInvalidPtrDerefAction());
      ContLoc *fin = procManager.GetFinalLoc();
      HandleConstraint(fin,Expr(),acts,res);
    }
  }
}

/*********************************************************************/
//add successors on assertion failures
/*********************************************************************/
void ImplState::HandleAssertionFailure(map< Action,set<const ImplState*> > &res) const
{
  ContLoc *contLoc = GetContLoc();
  const ProcManager &procManager = GetProcManager();
  if(contLoc == procManager.GetMarkedLoc()) {
    pair< Expr,list<Expr> > comps; ExprManager::GetCalledProcDetails(contLoc->GetLocRhs(),comps);
    assert(comps.second.size() == 1);
    set<Expr> x; x.insert(comps.second.front());
    set<Expr> state; contLoc->GetPreds().ToExpr(contLoc->GetPreds().PValToPredVal(predVal),state);
    set<Action> acts;
    //if the current state satisfies the assertion
    if(ExprManager::ProveImplies(state,x)) {
      acts.insert(ActionManager::GetImplEpsilonAction(GetProcManagerId()));
      HandleConstraint(GetContLoc()->GetTSucc(),Expr(),acts,res);
    }
    //if the current state violates the assertion
    else {
      acts.insert(ActionManager::GetAssertionFailureAction());
      HandleConstraint(procManager.GetFinalLoc(),Expr(),acts,res);
    }
  } else {
    ContLoc *succ = contLoc->GetTSucc();
    set<Action> acts; acts.insert(ActionManager::GetImplEpsilonAction(GetProcManagerId()));
    HandleConstraint(succ,Expr(),acts,res);
  }
}

/*********************************************************************/
//return successors of a return control location. 
/*********************************************************************/
void ImplState::GetSuccsAndActionsReturn(const set<Action> &globActs,const set<Action> &chanActs,map< Action,set<const ImplState*> > &res) const
{
  //get the successor location. this should be a final location.
  ContLoc *contLoc = GetContLoc();
  ContLoc *fin = contLoc->GetTSucc();
  assert(fin->GetLocType() == ContLoc::FINAL);  
  //get the return expression
  const Expr &expr = contLoc->GetLocExpr();
  //get the concrete expression for the return state
  set<Expr> state; contLoc->GetPreds().ToExpr(contLoc->GetPreds().PValToPredVal(predVal),state);  
  //the list of actions associated
  set<Action> acts;
  //the program will terminate with this return
  Database::GetSpecReturnActions(expr,GetProcManager().GetParams(),state,acts);
  if(acts.empty()) acts.insert(ActionManager::GetDefaultRetAction());
  HandleConstraint(fin,Expr(),acts,res);
  //handle global variable assignment actions
  HandleGlobalVarAssigns(globActs,-1,-1,res);
  //if checking for null pointer dereferences
  if(Database::CHECK_INVALID_PTR_DEREF) {
    HandleInvalidPtrDeref(res);
  }  
}

/*********************************************************************/
//return the successors of a branching statement
/*********************************************************************/
void ImplState::GetSuccsAndActionsBranch(const set<Action> &globActs,const set<Action> &chanActs,map< Action,set<const ImplState*> > &res) const
{
  ContLoc *contLoc = GetContLoc();
  const Expr &branchCond = contLoc->GetLocExpr();
  Expr falseCond = ExprManager::NegateExpr(branchCond);
  //get the actions - make sure to include fair actions 
  set<Action> thenActs,elseActs; 
  if((Database::CONF_TYPE == Database::CONF_LTL) && contLoc->IsFairLoop()) {
    const pair<Action,Action> &fairLoopActs = contLoc->GetFairLoopActs();
    thenActs.insert(fairLoopActs.first);
    elseActs.insert(fairLoopActs.second);
  } else {
    thenActs.insert(ActionManager::GetImplEpsilonAction(GetProcManagerId()));
    elseActs.insert(ActionManager::GetImplEpsilonAction(GetProcManagerId()));
  }
  //get the successors and actions
  HandleConstraint(contLoc->GetTSucc(),branchCond,thenActs,res);
  HandleConstraint(contLoc->GetESucc(),falseCond,elseActs,res);
  //handle global variable assignment actions
  HandleGlobalVarAssigns(globActs,-1,-1,res);
  //if checking for null pointer dereferences
  if(Database::CHECK_INVALID_PTR_DEREF) {
    HandleInvalidPtrDeref(res);
  }
}

/*********************************************************************/
//process a bunch of assignment actions
/*********************************************************************/
inline void ImplState::ProcessAssignActs(ContLoc *succ,const Expr &lhs,const Expr &rhs,
					 const set<Expr> &state,const Expr &assignGuard,
					 const set<Expr> &addCons,
					 const short fsid,const short fiid,
					 map< Action,set<const ImplState*> > &res) const
{
  //find the set of actions associated with the assignment
  set<Action> acts; Database::GetSpecAssignActions(lhs,rhs,GetProcManager().GetParams(),state,acts);
  if(acts.empty()) {
    if(GetProcManager().IsConcGlobalVar(lhs)) {
      set<Expr> lvc; Util::LvalueClosure(assignGuard,state,lvc);
      Expr clvc = ExprManager::ConjunctExprSet(lvc);
      acts.insert(Action(lhs,clvc));
    } else {
      acts.insert(Action(lhs,assignGuard));
    }
    list<Expr> lhsList,rhsList;
    lhsList.push_back(lhs);
    rhsList.push_back(rhs);
    HandleAssign(succ,addCons,lhsList,rhsList,acts,fsid,fiid,res);
  } else {
    list<Expr> repList; ExprManager::StringListToExprList(GetProcManager().GetParams(),repList);
    for(set<Action>::const_iterator j = acts.begin();j != acts.end();++j) {
      list<Expr> lhsList,rhsList;
      lhsList.push_back(ExprManager::ReplaceDummyVarsOne(j->GetLhsExpr(),repList));
      rhsList.push_back(rhs);
      set<Action> thisAct; thisAct.insert(*j);
      HandleAssign(succ,addCons,lhsList,rhsList,thisAct,fsid,fiid,res);
    }
  }
}

/*********************************************************************/
//return the successors of an assignment statement where the rhs is
//not a procedure call
/*********************************************************************/
void ImplState::GetSuccsAndActionsAssign(const set<Action> &globActs,const set<Action> &chanActs,map< Action,set<const ImplState*> > &res) const
{
  ContLoc *contLoc = GetContLoc();
  //get the lhs and rhs of the assignment
  const Expr &lhs = contLoc->GetLocLhs();
  const Expr &rhs = contLoc->GetLocRhs(); 
  //create the aliases set of the lhs
  set<Expr> lhsAliases = contLoc->GetLhsAliases();
  lhsAliases.insert(lhs);
  //try each possible aliasing scenario
  set<Expr> state; contLoc->GetPreds().ToExpr(contLoc->GetPreds().PValToPredVal(predVal),state);
  for(set<Expr>::const_iterator i = lhsAliases.begin();i != lhsAliases.end();++i) {
    //find the set of actions associated with the assignment
    Expr dummy = ExprManager::GetIdExpr("$0");
    Expr assignGuard = ExprManager::GetBinaryExpr(dummy,rhs,MAGIC_EQ_OP);
    ProcessAssignActs(contLoc->GetTSucc(),*i,rhs,state,assignGuard,set<Expr>(),-1,-1,res);
  }
  //handle global variable assignment actions
  HandleGlobalVarAssigns(globActs,-1,-1,res);
  //if checking for null pointer dereferences
  if(Database::CHECK_INVALID_PTR_DEREF) {
    HandleInvalidPtrDeref(res);
  }
}

/*********************************************************************/
//return the successors of an call statement. first argument is the
//lvalue where the return value is stored. second argument is the
//actual call.
/*********************************************************************/
void ImplState::GetSuccsAndActionsCall(const set<Action> &globActs,const set<Action> &chanActs,map< Action,set<const ImplState*> > &res) const
{
  ContLoc *contLoc = GetContLoc();
  Expr lhs = contLoc->GetLocLhs();
  Expr rhs = contLoc->GetLocRhs();
  //get the successor control location
  ContLoc *succ = contLoc->GetTSucc();
  //get the concrete expression for the current state
  set<Expr> state; contLoc->GetPreds().ToExpr(contLoc->GetPreds().PValToPredVal(predVal),state);
  //find the list of arguments to the call
  pair< Expr,list<Expr> > comps; ExprManager::GetCalledProcDetails(rhs,comps);
  list<Expr> &args = comps.second;
  //handle calls to exit
  if(Util::TrimString(comps.first.ToString()) == Database::EXIT_C_PROC) {
    set<Action> acts; acts.insert(ActionManager::GetImplEpsilonAction(GetProcManagerId()));
    AddSuccAndActions(this,acts,res);
    return;
  }
  //handle calls to assert
  if(Database::CHECK_ASSERTION_FAILURE && Util::TrimString(comps.first.ToString()) == Database::ASSERT_C_PROC) {
    HandleAssertionFailure(res);
    return;
  }
  //handle entry state
  if(specStateId == -1) {
    set<Action> acts; acts.insert(ActionManager::GetImplEpsilonAction(GetProcManagerId()));
    const PredSet &preds = contLoc->GetPreds();
    set<Expr> conc; preds.ToExpr(preds.PValToPredVal(predVal),conc);
    //get the names of the spec LTSs that abstract this call
    set<string> ltsNames = contLoc->GetProcInfo()->GetCallAbsLtsNames(rhs,contLoc->GetAbsProc(),conc);
    for(set<string>::const_iterator j = ltsNames.begin();j != ltsNames.end();++j) {
      const LtsInfo &li = Database::GetLtsInfo(*j);
      const ImplState *is = contLoc->GetImplState(predVal,li.GetStateId(*j),li.GetIndex());
      AddSuccAndActions(is,acts,res);
    }
    return;
  }
  //we have to inline the abstraction LTSs. find the successors and
  //actions of the spec state associated.
  LtsInfo &specInfo = Database::ltsInfos[specInfoId];
  string ss = specInfo.GetIdState(specStateId);
  list<LtsTrans> outTrans; specInfo.GetOutTrans(ss,outTrans);
  //add the implementation state for the destination state
  //of each outgoing transition
  for(list<LtsTrans>::const_iterator i = outTrans.begin();i != outTrans.end();++i) {
    string fs = i->GetFinalState();
    //if the destination state is a STOP state we have to
    //consider the effect of return actions
    if(fs == Database::STOP_STATE) {
      Action x = i->GetAction();
      assert(x.IsReturn());
      string tv = Util::NewTempVar();
      Expr tvExpr = ExprManager::GetIdExpr(tv);
      args.push_front(tvExpr);     
      Expr retExpr = x.GetConcreteReturnExpr(args);
      args.pop_front();
      //if the return expression is void
      if(retExpr.IsEmptyExpr()) {
	set<Action> acts; acts.insert(ActionManager::GetImplEpsilonAction(GetProcManagerId()));
	HandleConstraint(succ,Expr(),acts,res);
      }
      //if this is a default return
      else if(retExpr.ToString() == "! ") {
	set<Action> acts; acts.insert(ActionManager::GetDefaultRetAction());
	HandleConstraint(succ,Expr(),acts,res);
      }
      //if the return expression is non-void and non-default
      else {
	set<Expr> newState = state;
	newState.insert(retExpr);
	set<Expr> addCons;
	addCons.insert(retExpr);
	Expr assignGuard = ExprManager::ReplaceDummyVarsNonZero(x.GetRetExpr(),args);
	ProcessAssignActs(succ,lhs,tvExpr,newState,assignGuard,addCons,-1,-1,res);
      }      
    }
    //otherwise the final state is not stop
    else {
      Action x = i->GetAction();
      assert(!x.IsReturn());
      short fsid = specInfo.GetStateId(fs);
      //otherwise if this is an assign action
      if(x.IsAssign()) {
	string tv = Util::NewTempVar();
	Expr tvExpr = ExprManager::GetIdExpr(tv);
	args.push_front(tvExpr);
	pair<Expr,Expr> concAssign = x.GetConcreteAssignExprs(args);
	args.pop_front();
	set<Expr> newState = state;
	newState.insert(concAssign.second);
	set<Expr> addCons;
	addCons.insert(concAssign.second);
	Expr assignGuard = ExprManager::ReplaceDummyVarsNonZero(x.GetRhsExpr(),args);
	ProcessAssignActs(contLoc,concAssign.first,tvExpr,newState,assignGuard,addCons,fsid,specInfoId,res);
      }
      //otherwise if this is a send/bcast action
      else if(x.IsBcast() || x.IsSend()) {
	list<Expr> cargs;
	for(list<Expr>::const_iterator j = x.GetMessage().begin();j != x.GetMessage().end();++j) {
	  cargs.push_back(ExprManager::ReplaceDummyVarsNonZero(*j,args));	  
	}
	set<Action> acts;
	acts.insert(Action(x.IsBcast() ? BasicAction::ACTION_BCAST : BasicAction::ACTION_SEND,x.GetChannel(),cargs));
        const ImplState *sis = contLoc->GetImplState(predVal,fsid,specInfoId);
        AddSuccAndActions(sis,acts,res);
      }
      //otherwise if this is a receive action
      else if(x.IsRecv()) {
	//check if empty receive
	if(x.GetStore().empty()) {
	  set<Action> acts; 
	  acts.insert(Action(BasicAction::ACTION_BCAST,x.GetChannel(),x.GetStore()).Complement());
	  acts.insert(Action(BasicAction::ACTION_SEND,x.GetChannel(),x.GetStore()).Complement());
	  const ImplState *sis = contLoc->GetImplState(predVal,fsid,specInfoId);
	  AddSuccAndActions(sis,acts,res);
	} else {
	  list<Expr> lhsList;
	  for(list<Expr>::const_iterator j = x.GetStore().begin();j != x.GetStore().end();++j) {
	    lhsList.push_back(ExprManager::ReplaceDummyVarsOne(*j,args));	    
	  }
	  //do a RECV action. the target location is assumed to
	  //receive a non-deterministic value
	  if(chanActs.empty() && globActs.empty()) {	  
	    set<Action> acts; acts.insert(Action(BasicAction::ACTION_RECV,x.GetChannel(),lhsList));
	    list<Expr> rhsList;
	    for(list<Expr>::const_iterator k = x.GetStore().begin();k != x.GetStore().end();++k) {
	      rhsList.push_back(ExprManager::GetIdExpr(Util::NewTempVar()));
	    }	    
	    HandleAssign(contLoc,set<Expr>(),lhsList,rhsList,acts,fsid,specInfoId,res);
	  }
	  //do a complement of some SEND or BCAST action
	  for(set<Action>::const_iterator j = chanActs.begin();j != chanActs.end();++j) {
	    if(x.GetChannel() == j->GetChannel()) {
	      if(x.GetStore().size() != j->GetMessage().size()) {
		Util::Error("ERROR: mismatch in number of channel arguments ...\n"
			    "HINT: inspect receive %s and broadcase or send %s\n",
			    x.ToString().c_str(),j->ToString().c_str());
	      }
	      set<Expr> addCons; list<Expr> rhsList;
	      for(list<Expr>::const_iterator k = j->GetMessage().begin();k != j->GetMessage().end();++k) {
		string tv = Util::NewTempVar();
		Expr tvExpr = ExprManager::GetIdExpr(tv);
		args.push_front(tvExpr);
		Expr rhs = ExprManager::ReplaceDummyVarsZero(*k,args);
		args.pop_front();
		addCons.insert(rhs);
		rhsList.push_back(tvExpr);
	      }	    
	      set<Action> acts; acts.insert(j->Complement());
	      HandleAssign(contLoc,addCons,lhsList,rhsList,acts,fsid,specInfoId,res);
	    }
	  }
	}
      }
      //otherwise this is a basic action
      else {
	//if this is a spec epsilon action we change it to a impl epsilon action
	if(x.IsSpecEpsilon()) x = ActionManager::GetImplEpsilonAction(GetProcManagerId());
	set<Action> acts; acts.insert(x);
	const ImplState *sis = contLoc->GetImplState(predVal,fsid,specInfoId);
	AddSuccAndActions(sis,acts,res);
      }
    }
  }
  //handle global variable assignment actions
  HandleGlobalVarAssigns(globActs,specStateId,specInfoId,res);
  //if checking for null pointer dereferences
  if(Database::CHECK_INVALID_PTR_DEREF) {
    HandleInvalidPtrDeref(res);
  }
}

/*********************************************************************/
//add a list of successors and actions to a list. the successor is the
//second argument. the list of actions is the third argument. for each
//action in the list create an object array of size 2. make the first
//element the successor and the second element the action. add this
//array to the first argument.
/*********************************************************************/
void ImplState::AddSuccAndActions(const ImplState *succ,const set<Action> &actions,map< Action,set<const ImplState*> > &res) const
{
  for(set<Action>::const_iterator i = actions.begin();i != actions.end();++i) {
    res[*i].insert(succ);
    Util::Message(3,ToString() + " === " + i->ToString() + " ==> " + succ->ToString() + "\n");
  }
}

/*********************************************************************/
//string representation
/*********************************************************************/
string ImplState::ToString() const
{
  ContLoc *contLoc = GetContLoc();
  //get the expression for the predicate valuation and convert it to a string
  set<Expr> a; contLoc->GetPreds().ToExpr(contLoc->GetPreds().PValToPredVal(predVal),a);
  string b = ExprManager::ExprSetToString(a);
  if(specStateId == -1) {
    return (contLoc->ToString() + ":" + "null" + ":" + b);
  } else {
    return (contLoc->ToString() + ":" + Database::ltsInfos[specInfoId].GetIdState(specStateId) + ":" + b);
  }
}

/*********************************************************************/
//check if a new predicate valuation is possible from an old valuation
//due to an assignment statement.
/*********************************************************************/
bool ImplState::PossibleNextValuation(const set<Expr> &addCons,const list<Expr> &lhsList,const list<Expr> &rhsList,
				      const PredSet &oldPreds,const PVal &oldVal,
				      const PredSet &newPreds,const PVal &newVal) const
{
  //get the concrete expression for the old valuation
  set<Expr> oldExpr; oldPreds.ToExpr(oldPreds.PValToPredVal(oldVal),oldExpr);
  oldExpr.insert(addCons.begin(),addCons.end());
  
  //get the concrete expression for the new valuation
  set<Expr> newExpr1; newPreds.ToExpr(newPreds.PValToPredVal(newVal),newExpr1);

  //get the negation of the weakest preconditions of the
  //concrete new valuation expressions
  set<Expr> newExpr2;
  for(set<Expr>::const_iterator i = newExpr1.begin();i != newExpr1.end();++i) {
    Expr a = ExprManager::ComputeWP(lhsList,rhsList,*i);
    newExpr2.insert(ExprManager::NegateExpr(a));
  }

  //finally check implication
  return !ExprManager::ProveImplies(oldExpr,newExpr2);
}

/*********************************************************************/
//return the index of the proc manager of this impl state
/*********************************************************************/
size_t ImplState::GetProcManagerId() const
{
  return (contLocId % Database::rootProcNames.size());
}

/*********************************************************************/
//return the proc manager of this impl state
/*********************************************************************/
const ProcManager &ImplState::GetProcManager() const
{
  return *(static_cast<ProcManager*>(Database::components[GetProcManagerId()]));
}

/*********************************************************************/
//return the pointer to the control location for this state
/*********************************************************************/
ContLoc *ImplState::GetContLoc() const
{
  //the proc manager index
  size_t pmi = contLocId % Database::rootProcNames.size();
  //the cont loc index
  size_t cli = contLocId / Database::rootProcNames.size();
  return static_cast<ProcManager*>(Database::components[pmi])->GetUsefulLocs()[cli];
}

/*********************************************************************/
//methods of class ImplTrans
/*********************************************************************/
ImplTrans::ImplTrans(const ImplState *s,const Action &a,const ImplState *d)
{
  source = s;
  act = a;
  dest = d;
}

const ImplTrans &ImplTrans::operator = (const ImplTrans &rhs)
{
  source = rhs.source;
  act = rhs.act;
  dest = rhs.dest;
  return *this;
}

bool ImplTrans::operator == (const ImplTrans &rhs) const
{
  return ((source == rhs.source) && (act == rhs.act) && (dest == rhs.dest));
}

bool ImplTrans::operator < (const ImplTrans &rhs) const
{
  if(source < rhs.source) return true;
  if(source > rhs.source) return false;
  if(act < rhs.act) return true;
  if(rhs.act < act) return false;
  return (dest < rhs.dest);
}

string ImplTrans::ToString() const
{
  return source->ToString() + " === " + act.ToString() + " ===> " + dest->ToString();
}
 
/*********************************************************************/
//methods of class ImplPath
/*********************************************************************/
const ImplPath &ImplPath::operator = (const ImplPath &rhs)
{
  data = rhs.data;
  return *this;
}

bool ImplPath::operator == (const ImplPath &rhs) const
{
  return (data == rhs.data);
}

bool ImplPath::operator < (const ImplPath &rhs) const
{
  return (data < rhs.data);
}

void ImplPath::PushBack(const Action &act,const ImplState *is)
{
  data.push_back(pair<Action,const ImplState*>(act,is));
}

void ImplPath::Append(const ImplPath &rhs)
{
  data.insert(data.end(),rhs.data.begin(),rhs.data.end());
}

/*********************************************************************/
//end of ImplState.cpp
/*********************************************************************/
