/******************************** CPPFile *****************************

* FileName [GlobalCEDag.cpp]

* PackageName [main]

* Synopsis [Method definitions of GlobalCEDag class.]

* SeeAlso [GlobalCEDag.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <gmp.h>
#include <string>
#include <list>
#include <set>
#include <map>
#include <vector>
using namespace std;

#include "BigInt.h"
#include "Node.h"
#include "Action.h"
#include "ActionManager.h"
#include "LtsInfo.h"
#include "SimChecker.h"
#include "Database.h"
#include "Component.h"
#include "GlobalAbsLts.h"
#include "LtlChecker.h"
#include "DeadlockChecker.h"
#include "LtsaChecker.h"
#include "GlobalCEDag.h"
using namespace magic;

/*********************************************************************/
//define static fields
/*********************************************************************/
vector<BigInt> GlobalCEDag::stateLabel;
vector<Action> GlobalCEDag::actLabel;
vector< set<int> > GlobalCEDag::trans;
set<BigInt> GlobalCEDag::allRoot;
map< BigInt,set< pair<Action,ExclInfo> > > GlobalCEDag::allTrans;
int GlobalCEDag::strategyMax;
int GlobalCEDag::strategyNum;

/*********************************************************************/
//initialise - clear all data structures
/*********************************************************************/
void GlobalCEDag::Initialize()
{
  allRoot.clear();
  allTrans.clear();
}

/*********************************************************************/
//compute the counter example dag from the results of the simulation
//or reachability check
/*********************************************************************/
bool GlobalCEDag::Compute()
{ 
  stateLabel.clear();
  actLabel.clear();
  trans.clear();
  //if checking simulation or ERROR state reachability
  if((Database::CONF_TYPE == Database::CONF_SIMUL) || (Database::CONF_TYPE == Database::CONF_REACH)) {
    return ComputeFsp();
  }
  //if checking LTL formula
  else if(Database::CONF_TYPE == Database::CONF_LTL) return ComputeLtl();
  //if checking deadlock
  else if(Database::CONF_TYPE == Database::CONF_DLOCK) return ComputeDeadlock();
  //if checking SE-AW formula
  else if(Database::CONF_TYPE == Database::CONF_SEAW) return ComputeSeaw();
  //impossible
  else assert(false);
}

/*********************************************************************/
//first macro for recursively calling ComputeWithRoot
/*********************************************************************/
inline void GlobalCEDag::MacroOne(const BigInt &elemId,const int &newNode,bool &newFlag,set<int> &stack,const int loop)
{
  stateLabel.push_back(elemId);
  actLabel.push_back(Action());
  trans.push_back(set<int>());
  stack.insert(newNode);
  ComputeWithRoot(newNode,newFlag,stack,loop);
  stack.erase(newNode);
}

/*********************************************************************/
//second macro. calls the first macro
/*********************************************************************/
inline void GlobalCEDag::MacroTwo(const int &node,bool &newFlag,set<int> &stack,const int loop,const ExclInfo &exclInfo,const Action &act)
{
  //update action label and allTrans
  actLabel[node] = act;
  for(set<BigInt>::const_iterator i = exclInfo.first.begin();i != exclInfo.first.end();++i) {
    bool found = false;
#ifdef WIN32
    if((rand() * 1.0 / RAND_MAX) > Database::CE_NEW_NODE_PROB) {
#else
    if((random() * 1.0 / RAND_MAX) > Database::CE_NEW_NODE_PROB) {
#endif //WIN32
      for(size_t l = 0;l < stateLabel.size();++l) {
	if((stack.count(l) == 0) && (stateLabel[l] == (*i))) {
	  trans[node].insert(l);
	  found = true;
	  break;
	}
      }	 
    }   
    if(!found) {
      int newLoop = (exclInfo.second.count(*i) == 0) ? (loop - 1) : loop;
      int newNode = stateLabel.size();
      MacroOne(*i,newNode,newFlag,stack,newLoop);
      trans[node].insert(newNode);
    }
  }
}

/*********************************************************************/
//third macro. calls the second macro
/*********************************************************************/
inline bool GlobalCEDag::MacroThree(const int &node,bool &newFlag,set<int> &stack,const int loop,const set<BigInt> &clause,const set<BigInt> &before,const Action &act)
{
  ExclInfo exclInfo(clause,before);
  //check if we can loop
  if((loop == 0) && (!(clause == before))) return false;
  //check if this successor has appeared before
  if(!newFlag) {
    if(IsValidSucc(node,act,clause)) newFlag = true;
    else return false;
  }
  //update action label and allTrans
  allTrans[stateLabel[node]].insert(pair<Action,ExclInfo>(act,exclInfo));
  MacroTwo(node,newFlag,stack,loop,exclInfo,act);
  return true;
}

/*********************************************************************/
//the top-level routine when we are checking for simulation or ERROR
//state reachability
/*********************************************************************/
bool GlobalCEDag::ComputeFsp()
{
  if(Database::CONF_CHECK_MECHANISM == Database::CONF_CHECK_SAT) {
    //the spec lts
    const LtsInfo &specInfo = Database::GetLtsInfo(Database::specLtsName);
    //the states of the specification
    const set<string> &specStates = specInfo.GetStates();
    //number of spec lts states
    int ssn = specStates.size();
    
    //the id of the initial state of the spec lts
    int initSpecId = specInfo.GetStateId(Database::specLtsName);
    //flag to indicate if a new dag was created
    bool newFlag = false;
    
    //find the first initial state pair that was excluded from the
    //maximal simulation relation and build the CE-DAG with it as the
    //root
    set<BigInt> initImpls; GlobalAbsLts::GetInitStates(initImpls);
    for(set<BigInt>::const_iterator i = initImpls.begin();i != initImpls.end();++i) {
	//if this element was excluded
      BigInt elemId = (*i) * ssn + initSpecId;
      if((allRoot.count(elemId) == 0) && (SimChecker::exclMap.count(elemId) != 0)) {
	newFlag = allRoot.empty();
	allRoot.insert(elemId);
	set<int> stack;
	MacroOne(elemId,stateLabel.size(),newFlag,stack,Database::MAX_CE_LOOP);
	return newFlag;
      }
    }
    
    //if we reached here it means no new roots were found. this must
    //not be the first attempt.
    assert(!allRoot.empty());
    vector<BigInt> ar = RandomizeIntSet(allRoot);
    newFlag = false;
    set<int> stack;
    MacroOne(ar[0],stateLabel.size(),newFlag,stack,Database::MAX_CE_LOOP);
    return newFlag;
  } 
  else {
    assert(Database::CONF_CHECK_MECHANISM == Database::CONF_CHECK_LTSA);
    set< list<Action> >::const_iterator i = LtsaChecker::ceList.begin();
    //if there is some remaining counterexample
    if(i != LtsaChecker::ceList.end()) {
      //simulate the model along the counterexample
      vector< set<BigInt> > regions(i->size() + 1);
      size_t pos = 0;
      GlobalAbsLts::GetInitStates(regions[pos]);
      for(list<Action>::const_iterator j = i->begin();j != i->end();++j,++pos) {
	assert(!regions[pos].empty());
	set<BigInt> haveSucc;
	for(set<BigInt>::const_iterator k = regions[pos].begin();k != regions[pos].end();++k) {
	  set<BigInt> x; GlobalAbsLts::GetSuccsOnAction(*k,*j,x);
	  regions[pos + 1].insert(x.begin(),x.end());
	  if(!x.empty()) haveSucc.insert(*k);
	}
	regions[pos] = haveSucc;
      }
      //create action labels
      actLabel.insert(actLabel.end(),i->begin(),i->end());
      //create state labels
      assert(!regions.back().empty());
      stateLabel.resize(pos);
      BigInt currState = *(regions.back().begin());
      for(int j = pos - 1;j >= 0;--j) {
	bool found = false;
	for(set<BigInt>::const_iterator k = regions[j].begin();k != regions[j].end();++k) {
	  set<BigInt> x; GlobalAbsLts::GetSuccsOnAction(*k,actLabel[j],x);
	  if(x.count(currState) != 0) {
	    stateLabel[j] = (*k) * Database::GetSpecStateNum();
	    currState = *k;
	    found = true;
	    break;
	  }
	}
	assert(found);
      }
      //create transition relation
      trans.resize(pos);
      for(size_t j = 0;j < pos - 1;++j) trans[j].insert(j + 1);
      //remove this counterexample
      LtsaChecker::ceList.erase(i);
      return true;
    }
    //no more counterexamples remaining
    else return false;
  }
}

/*********************************************************************/
//compute the CE dag with the argument as the root
/*********************************************************************/
void GlobalCEDag::ComputeWithRoot(const int &node,bool &newFlag,set<int> &stack,const int loop)
{
  //the spec lts
  const LtsInfo &specInfo = Database::GetLtsInfo(Database::specLtsName);
  //the states of the specification
  const set<string> &specStates = specInfo.GetStates();
  //number of spec lts states
  int ssn = specStates.size();

  //get the abstract state and spec state components
  BigInt absState, specStateBig;
  BigInt::Div(absState,specStateBig,stateLabel[node],ssn);
  int specState = specStateBig.ToSL();

  //get the actions of the implementation
  set<Action> iActs; GlobalAbsLts::GetActions(iActs);
  vector<Action> implActs = RandomizeActSet(iActs);

  //check for each possible action and successor
  for(vector<Action>::const_iterator j = implActs.begin();j != implActs.end();++j) {
    set<BigInt> iSuccs; GlobalAbsLts::GetSuccsOnAction(absState,*j,iSuccs);
    vector<BigInt> implSuccs = RandomizeIntSet(iSuccs);
    set<string> specSuccs; specInfo.GetSuccsOnActionSpec(specState,*j,specSuccs);
    for(vector<BigInt>::const_iterator k = implSuccs.begin();k != implSuccs.end();++k) {
      set<BigInt> clause,before;
      bool flag = true;
      for(set<string>::const_iterator m = specSuccs.begin();m != specSuccs.end();++m) {
	BigInt succId = (*k) * ssn + specInfo.GetStateId(*m);
	pair<bool,bool> excl = ExcludedBefore(succId,stateLabel[node]);
	if(excl.first) {
	  clause.insert(succId);
	  if(excl.second) before.insert(succId);
	}
	//for simulation we are done
	else if(Database::CONF_TYPE == Database::CONF_SIMUL) {
	  flag = false;
	  break;
	}
      }

      //if we are checking for simulation
      if(Database::CONF_TYPE == Database::CONF_SIMUL) {
	//check if this is the successor and action to be expanded
	if(flag) {
	  if(MacroThree(node,newFlag,stack,loop,clause,before,*j)) return;
	}
      }
      //else if we are checking for reachability
      else if(Database::CONF_TYPE == Database::CONF_REACH) {
	//try each successor to see if it is suitable
	for(set<BigInt>::const_iterator l = clause.begin();l != clause.end();++l) {
	  set<BigInt> newClause,newBefore;
	  //if the successor has an ERROR component, make it a leaf
	  //node, else extend this path else make
	  BigInt x,y; BigInt::Div(x,y,*l,ssn);
	  if(y.ToSL() != specInfo.GetStateId(Database::ERROR_STATE)) {
	    newClause.insert(*l);
	    if(before.count(*l) != 0) newBefore.insert(*l);
	  }
	  if(MacroThree(node,newFlag,stack,loop,newClause,newBefore,*j)) return;
	}
      }
      //this should not happen
      else assert(false);
    }
  }

  //the OR choices
  vector< pair<Action,ExclInfo> > orChoices = RandomizeChoices(allTrans[stateLabel[node]]);
  for(vector< pair<Action,ExclInfo> >::const_iterator j = orChoices.begin();j != orChoices.end();++j) {
    if((loop == 0) && (!(j->second.first == j->second.second))) continue;
    MacroTwo(node,newFlag,stack,loop,j->second,j->first);
    return;
  }
  //this should not happen
  assert(false);
}

/*********************************************************************/
//return true if the first argument was excluded from the maximal
//simulation relation before the second argument and false otherwise
/*********************************************************************/
pair<bool,bool> GlobalCEDag::ExcludedBefore(const BigInt &succ,const BigInt &pred)
{
  if(SimChecker::exclMap.count(succ) == 0) return pair<bool,bool>(false,false);
  else return pair<bool,bool>(true,SimChecker::exclMap[succ] < SimChecker::exclMap[pred]);
}

/*********************************************************************/
//return true if the choice of successor in the global CE Dag is valid
//- is not the same as the previous choice
/*********************************************************************/
bool GlobalCEDag::IsValidSucc(const int &node,const Action &act,const set<BigInt> &succs)
{
  map< BigInt,set< pair<Action,ExclInfo> > >::const_iterator i = allTrans.find(stateLabel[node]);
  if(i == allTrans.end()) return true;
  const LtsInfo &specInfo = Database::GetLtsInfo(Database::specLtsName);
  int ssn = specInfo.GetStates().size();
  //decompose successors
  vector< set<int> > succComps(Database::components.size());
  for(set<BigInt>::const_iterator i = succs.begin();i != succs.end();++i) {
    vector<L2AbsState> x = GlobalAbsLts::CompToAbsStates((*i) / ssn);
    for(size_t j = 0;j < Database::components.size();++j) {
      succComps[j].insert(x[j]);
    }
  }
  //make sure that the control locations are different
  for(set< pair<Action,ExclInfo> >::const_iterator k = i->second.begin();k != i->second.end();++k) {
    vector< set<int> > newComps(Database::components.size());
    for(set<BigInt>::const_iterator i = k->second.first.begin();i != k->second.first.end();++i) {
      vector<L2AbsState> x = GlobalAbsLts::CompToAbsStates((*i) / ssn);
      for(size_t j = 0;j < Database::components.size();++j) {
	newComps[j].insert(x[j]);
      }
    }
    //check if this is a distinct set
    bool flag = false;
    for(size_t j = 0;j < Database::components.size();++j) {
      if(Database::components[j]->DistinctL2AbsStates(succComps[j],newComps[j])) {
	flag = true;
	break;
      }
    }
    if(flag) return true;
  }
  //this choice is inadmissible
  return false;
}

/*********************************************************************/
//the top-level routine when we are checking for LTL property
/*********************************************************************/
bool GlobalCEDag::ComputeLtl()
{
  if(LtlChecker::statePaths.empty()) return false;
  const list<BigInt> &statePath = LtlChecker::statePaths.front();
  const list<Action> &actPath = LtlChecker::actPaths.front();

  assert(!actPath.empty());
  list<BigInt>::const_iterator i = statePath.begin();
  list<Action>::const_iterator j = actPath.begin();
  //for infinite counterexamples
  if(statePath.size() == (actPath.size() + 1)) {
    int loopBack = -1;
    for(;(i !=  statePath.end()) && (j !=  actPath.end());++i,++j) {
      size_t newNode = stateLabel.size();
      trans.push_back(set<int>());
      if((*i) ==  statePath.back()) loopBack = newNode;
      stateLabel.push_back(*i);
      actLabel.push_back(*j);
      ++j;
      if(j == actPath.end()) {
	assert(loopBack != -1);
	trans[newNode].insert(loopBack);
      } else {
	trans[newNode].insert(newNode + 1);
      }
      --j;
    }
  }
  //for finite counterexamples
  else if(statePath.size() == actPath.size()) {
    for(;(i !=  statePath.end()) && (j !=  actPath.end());++i,++j) {
      size_t newNode = stateLabel.size();
      trans.push_back(set<int>());
      stateLabel.push_back(*i);
      actLabel.push_back(*j);
      ++j;
      if(j != actPath.end()) trans[newNode].insert(newNode + 1);
      --j;
    }
  } else assert(false);
  LtlChecker::statePaths.pop_front();
  LtlChecker::actPaths.pop_front();
  return true;
}

/*********************************************************************/
//the top-level routine when we are checking for deadlock
/*********************************************************************/
bool GlobalCEDag::ComputeDeadlock()
{
  assert(DeadlockChecker::statePath.size() == (DeadlockChecker::actPath.size() + 1));
  list<BigInt>::const_iterator i = DeadlockChecker::statePath.begin();
  list<Action>::const_iterator j = DeadlockChecker::actPath.begin();
  for(;i !=  DeadlockChecker::statePath.end();++i,++j) {
    size_t newNode = stateLabel.size();
    trans.push_back(set<int>());
    stateLabel.push_back(*i);
    if(j != DeadlockChecker::actPath.end()) {
      actLabel.push_back(*j);
      trans[newNode].insert(newNode + 1);
    }
  }
  return true;
}

/*********************************************************************/
//the top-level routine when we are checking for SE-AW formula
/*********************************************************************/
bool GlobalCEDag::ComputeSeaw()
{
  return true;
}

/*********************************************************************/
//randomize a set of actions
/*********************************************************************/
vector<Action> GlobalCEDag::RandomizeActSet(const set<Action> &arg)
{
  set<Action> remaining = arg;
  vector<Action> res;
  while(!remaining.empty()) {
#ifdef WIN32
    int randInd = static_cast<int>((rand() * 1.0 / RAND_MAX) * remaining.size());
#else
    int randInd = static_cast<int>((random() * 1.0 / RAND_MAX) * remaining.size());
#endif //WIN32
    int pos = 0;
    for(set<Action>::const_iterator i = remaining.begin();i != remaining.end();++i,++pos) {
      if(pos == randInd) {
	res.push_back(*i);
	remaining.erase(i);
	break;
      }
    }
  }
  assert(res.size() == arg.size());
  return res;
}

/*********************************************************************/
//randomize a set of integers
/*********************************************************************/
vector<BigInt> GlobalCEDag::RandomizeIntSet(const set<BigInt> &arg)
{
  set<BigInt> remaining = arg;
  vector<BigInt> res;
  while(!remaining.empty()) {
#ifdef WIN32
    int randInd = static_cast<int>((rand() * 1.0 / RAND_MAX) * remaining.size());
#else
    int randInd = static_cast<int>((random() * 1.0 / RAND_MAX) * remaining.size());
#endif
    int pos = 0;
    for(set<BigInt>::const_iterator i = remaining.begin();i != remaining.end();++i,++pos) {
      if(pos == randInd) {
	res.push_back(*i);
	remaining.erase(i);
	break;
      }
    }
  }
  assert(res.size() == arg.size());
  return res;
}

/*********************************************************************/
//randomize a set of choices for the impl machine
/*********************************************************************/
vector< pair<Action,ExclInfo> > GlobalCEDag::RandomizeChoices(const set< pair<Action,ExclInfo> > &arg)
{
  set< pair<Action,ExclInfo> > remaining = arg;
  vector< pair<Action,ExclInfo> > res;
  while(!remaining.empty()) {
#ifdef WIN32
    int randInd = static_cast<int>((rand() * 1.0 / RAND_MAX) * remaining.size());
#else
    int randInd = static_cast<int>((random() * 1.0 / RAND_MAX) * remaining.size());
#endif //WIN32
    int pos = 0;
    for(set< pair<Action,ExclInfo> >::const_iterator i = remaining.begin();i != remaining.end();++i,++pos) {
      if(pos == randInd) {
	res.push_back(*i);
	remaining.erase(i);
	break;
      }
    }
  }
  assert(res.size() == arg.size());
  return res;
}

/*********************************************************************/
//end of GlobalCEDag.cpp
/*********************************************************************/
