/******************************** CPPFile *****************************

* FileName [GlobalAbsLts.cpp]

* PackageName [main]

* Synopsis [Method definitions of GlobalAbsLts class.]

* SeeAlso [GlobalAbsLts.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <gmp.h>
#include <string>
#include <list>
#include <map>
#include <set>
#include <vector>
using namespace std;

#include "BigInt.h"
#include "Node.h"
#include "Action.h"
#include "Database.h"
#include "Component.h"
#include "GlobalAbsLts.h"
using namespace magic;

/*********************************************************************/
//define static fields
/*********************************************************************/
map< Action,set<size_t> > GlobalAbsLts::actToComp;
set<Action> GlobalAbsLts::singActs;
set<BigInt> GlobalAbsLts::reach;

/*********************************************************************/
//compute the singleton action map
/*********************************************************************/
void GlobalAbsLts::ComputeActInfo()
{
  //create actToComp
  actToComp.clear();
  for(size_t i = 0;i < Database::components.size();++i) {
    set<Action> acts; Database::components[i]->GetL2AbsActs(acts);
    for(set<Action>::const_iterator j = acts.begin();j != acts.end();++j) {
      actToComp[*j].insert(i);
    }
  }
  //create singActs
  singActs.clear();
  for(map< Action,set<size_t> >::const_iterator i = actToComp.begin();i != actToComp.end();++i) {
    if(i->first.IsSend()) {
      map< Action,set<size_t> >::const_iterator j = actToComp.find(i->first.Complement());
      if(j == actToComp.end()) singActs.insert(i->first);
      else {
	if((i->second.size() < 1) || (j->second.size() < 1)) singActs.insert(i->first);
	else if((i->second.size() == 1) && (j->second.size() == 1)) {
	  if((*(i->second.begin())) == (*(j->second.begin()))) singActs.insert(i->first);
	}
      }
    } else if(i->first.IsRecv()) {
      bool found = false;
      for(map< Action,set<size_t> >::const_iterator j = actToComp.begin();j != actToComp.end();++j) {
	if((i == j) || (!(j->first.IsBcast() || j->first.IsSend()))) continue;
	if(i->first.GetChannel() != j->first.GetChannel()) continue;
	if((i->second.size() > 1) || (j->second.size() > 1)) {
	  found = true;
	  break;
	} else if((i->second.size() == 1) && (j->second.size() == 1)) {
	  if((*(i->second.begin())) != (*(j->second.begin()))) {
	    found = true;
	    break;
	  }
	}
      }
      if(!found) singActs.insert(i->first);
    }
  }
}

/*********************************************************************/
//get the set of global initial states
/*********************************************************************/
void GlobalAbsLts::GetInitStates(set<BigInt> &res)
{
  vector< set<L2AbsState> > absInits;
  bool emptyFlag = false;
  for(vector<Component*>::const_iterator i = Database::components.begin();i != Database::components.end();++i) {
    absInits.push_back(set<int>());
    (*i)->GetL2AbsInit(absInits.back());
    if(absInits.back().empty()) {
      emptyFlag = true;
      break;
    }
  }
  if(!emptyFlag) CartesianProduct(absInits,0,1,0,res);
}

/*********************************************************************/
//get the set of global initial states in the previous iteration. this
//should only be called if we are doing incremental verification and
//the previous refinement was an LTS refinement.
/*********************************************************************/
void GlobalAbsLts::GetPrevInitStates(set<BigInt> &res)
{
  assert(Database::INC_VERIFY && (Database::lastRefComp != -1));
  vector< set<L2AbsState> > absInits;
  bool emptyFlag = false;
  for(size_t i = 0;i < Database::components.size();++i) {
    absInits.push_back(set<int>());
    if(i == static_cast<size_t>(Database::lastRefComp)) {
      Database::components[i]->GetPrevL2AbsInit(absInits.back());
    } else {
      Database::components[i]->GetL2AbsInit(absInits.back());
    }
    if(absInits.back().empty()) {
      emptyFlag = true;
      break;
    }
  }
  if(!emptyFlag) CartesianProduct(absInits,0,1,0,res);
}

/*********************************************************************/
//given a vector of sets of component states return the set of
//composite states obtained by their cartesian products
/*********************************************************************/
void GlobalAbsLts::CartesianProduct(const vector< set<L2AbsState> > &arg,size_t index,const BigInt &factor,const BigInt &curr,set<BigInt> &res)
{
  if(index == arg.size()) {
    res.insert(curr);
  } else {
    for(set<int>::const_iterator i = arg[index].begin();i != arg[index].end();++i) {
      CartesianProduct(arg,index + 1,factor * Database::components[index]->GetL1AbsStateNum(),
		       curr + (factor * (*i)),res);
    }
  }
}

/*********************************************************************/
//compute successors or predecessors
/*********************************************************************/
void GlobalAbsLts::ComputeSuccsOrPreds(const size_t index,const L2AbsState &state,const Action &act,const bool succFlag,const bool prevFlag,set<L2AbsState> &res)
{
  if(prevFlag) {
    assert(succFlag);
    if(index == static_cast<size_t>(Database::lastRefComp)) {
      Database::components[index]->GetPrevL2AbsSuccsOnAction(state,act,res);      
    } else {
      Database::components[index]->GetL2AbsSuccsOnAction(state,act,res);
    }
  } else {
    if(succFlag) Database::components[index]->GetL2AbsSuccsOnAction(state,act,res);
    else Database::components[index]->GetL2AbsPredsOnAction(state,act,res);
  }
}

/*********************************************************************/
//arbitrary interleaving
/*********************************************************************/
void GlobalAbsLts::Interleave(const BigInt &comp,const Action &act,bool succFlag,bool prevFlag,set<BigInt> &res)
{
  vector<L2AbsState> absStates = CompToAbsStates(comp);
  for(size_t i = 0;i < Database::components.size();++i) {      
    vector< set<L2AbsState> > absSuccs(Database::components.size());
    ComputeSuccsOrPreds(i,absStates[i],act,succFlag,prevFlag,absSuccs[i]);
    if(absSuccs[i].empty()) continue;
    for(size_t j = 0;j < Database::components.size();++j) {
      if(j != i) absSuccs[j].insert(absStates[j]);
    }
    CartesianProduct(absSuccs,0,1,0,res);
  }
}

/*********************************************************************/
//global synchronization
/*********************************************************************/
void GlobalAbsLts::Synchronize(const BigInt &comp,const Action &act,bool succFlag,bool prevFlag,set<BigInt> &res)
{
  vector<L2AbsState> absStates = CompToAbsStates(comp);
  //for basic actions
  if(act.IsBasic()) {
    vector< set<L2AbsState> > absSuccs;
    bool emptyFlag = false;
    for(size_t i = 0;i < Database::components.size();++i) {
      absSuccs.push_back(set<L2AbsState>());
      if(Database::components[i]->IsL2AbsAct(act)) {
	ComputeSuccsOrPreds(i,absStates[i],act,succFlag,prevFlag,absSuccs.back());
	if(absSuccs.back().empty()) {
	  emptyFlag = true;
	  break;
	}
      } else {
	absSuccs.back().insert(absStates[i]);
      }
    }
    if(!emptyFlag) CartesianProduct(absSuccs,0,1,0,res);
  }
  //for broadcast or assignment actions
  else if(act.IsAssign() || act.IsBcast()) {
    for(size_t i = 0;i < Database::components.size();++i) {      
      if(!Database::components[i]->IsL2AbsAct(act)) continue;
      vector< set<L2AbsState> > absSuccs(Database::components.size());
      ComputeSuccsOrPreds(i,absStates[i],act,succFlag,prevFlag,absSuccs[i]);
      bool emptyFlag = absSuccs[i].empty();
      if(!emptyFlag) {
	for(size_t j = 0;j < Database::components.size();++j) {
	  if(j != i) {
	    if(Database::components[j]->IsL2AbsAct(act.Complement())) {
	      ComputeSuccsOrPreds(j,absStates[j],act.Complement(),succFlag,prevFlag,absSuccs[j]);
	      if(absSuccs[j].empty()) {
		emptyFlag = true;
		break;
	      }
	    } else {
	      absSuccs[j].insert(absStates[j]);
	    }
	  }
	}
      }
      if(!emptyFlag) CartesianProduct(absSuccs,0,1,0,res);
    }
  }
  else assert(false);
}

/*********************************************************************/
//handshake between two processes
/*********************************************************************/
void GlobalAbsLts::Handshake(const BigInt &comp,const Action &act,bool succFlag,bool prevFlag,set<BigInt> &res)
{
  vector<L2AbsState> absStates = CompToAbsStates(comp);
  for(size_t i = 0;i < Database::components.size();++i) {      
    if(!Database::components[i]->IsL2AbsAct(act)) continue;
    for(size_t j = 0;j < Database::components.size();++j) {
      if((i == j) || (!Database::components[j]->IsL2AbsAct(act.Complement()))) continue;
      vector< set<L2AbsState> > absSuccs(Database::components.size());
      ComputeSuccsOrPreds(i,absStates[i],act,succFlag,prevFlag,absSuccs[i]);
      ComputeSuccsOrPreds(j,absStates[j],act.Complement(),succFlag,prevFlag,absSuccs[j]);
      if(absSuccs[i].empty() || absSuccs[j].empty()) continue;
      for(size_t k = 0;k < Database::components.size();++k) {
	if((k != i) && (k != j)) absSuccs[k].insert(absStates[k]);
      }
      CartesianProduct(absSuccs,0,1,0,res);
    }
  }
}

/*********************************************************************/
//do actions that happen in just one component
/*********************************************************************/
bool GlobalAbsLts::SingletonAction(const BigInt &comp,const Action &act,bool succFlag,bool prevFlag,set<BigInt> &res)
{
  if(singActs.count(act) == 0) return false;
  vector<L2AbsState> absStates = CompToAbsStates(comp);
  for(set<size_t>::const_iterator i = actToComp[act].begin();i != actToComp[act].end();++i) {
    vector< set<L2AbsState> > absSuccs(Database::components.size());
    ComputeSuccsOrPreds(*i,absStates[*i],act,succFlag,prevFlag,absSuccs[*i]);
    if(absSuccs[*i].empty()) continue;
    for(size_t j = 0;j < Database::components.size();++j) {      
      if(j != *i) absSuccs[j].insert(absStates[j]);
    }
    CartesianProduct(absSuccs,0,1,0,res);
  }
  return true;
}

/*********************************************************************/
//compute the map from outgoing actions to next states given a global
//state
/*********************************************************************/
void GlobalAbsLts::GetSuccsAndActions(const BigInt &comp,map< Action,set<BigInt> > &res)
{
  vector<L2AbsState> absStates = CompToAbsStates(comp);
  set<Action> outActs;
  for(size_t i = 0;i < Database::components.size();++i) {      
    set<Action> acts;
    Database::components[i]->GetL2AbsActsOut(absStates[i],acts);
    outActs.insert(acts.begin(),acts.end());
  }
  for(set<Action>::const_iterator i = outActs.begin();i != outActs.end();++i) {
    GetSuccsOnAction(comp,*i,res[*i]);
  }
}

/*********************************************************************/
//given a global implementation state and an action return the
//possible successors
/*********************************************************************/
void GlobalAbsLts::GetSuccsOnAction(const BigInt &comp,const Action &act,set<BigInt> &res)
{
  //basic actions force global sync
  if(act.IsBasic()) Synchronize(comp,act,true,false,res);
  //assign action
  else if(act.IsAssign()) {
    //if this is an assignment to a global variable then synchronize
    //globally
    if(Database::components[0]->IsConcGlobalVar(act.GetLhsExpr())) Synchronize(comp,act,true,false,res);
    //otherwise do not synchronize
    else Interleave(comp,act,true,false,res);
  }
  //return actions force no sync at all
  else if(act.IsReturn()) Interleave(comp,act,true,false,res);
  //broadcast actions force global synchronization
  else if(act.IsBcast()) Synchronize(comp,act,true,false,res);
  //send actions: if this action appears in just one component then
  //just do it, otherwise synchronize
  else if(act.IsSend()) {
    if(!SingletonAction(comp,act,true,false,res)) {
      Handshake(comp,act,true,false,res);
    }
  }
  //receive actions: if this action appears in just one component then
  //just do it
  else if(act.IsRecv()) {
    SingletonAction(comp,act,true,false,res);
  }
}

/*********************************************************************/
//compute the map from incoming actions to previous states given a
//global state
/*********************************************************************/
void GlobalAbsLts::GetPredsAndActions(const BigInt &comp,map< Action,set<BigInt> > &res)
{
  vector<L2AbsState> absStates = CompToAbsStates(comp);
  set<Action> inActs;
  for(size_t i = 0;i < Database::components.size();++i) {      
    set<Action> acts;
    Database::components[i]->GetL2AbsActsIn(absStates[i],acts);
    inActs.insert(acts.begin(),acts.end());
  }
  for(set<Action>::const_iterator i = inActs.begin();i != inActs.end();++i) {
    GetPredsOnAction(comp,*i,res[*i]);
  }
}

/*********************************************************************/
//given a global implementation state and an action return the
//possible predecessors
/*********************************************************************/
void GlobalAbsLts::GetPredsOnAction(const BigInt &comp,const Action &act,set<BigInt> &res)
{
  //basic actions force global sync
  if(act.IsBasic()) Synchronize(comp,act,false,false,res);
  //assign action
  else if(act.IsAssign()) {
    //if this is an assignment to a global variable then synchronize
    //globally
    if(Database::components[0]->IsConcGlobalVar(act.GetLhsExpr())) Synchronize(comp,act,false,false,res);
    //otherwise do not synchronize
    else Interleave(comp,act,false,false,res);
  }
  //return actions force no sync at all
  else if(act.IsReturn()) Interleave(comp,act,false,false,res);
  //broadcast actions force global synchronization
  else if(act.IsBcast()) Synchronize(comp,act,false,false,res);
  //send actions: if this action appears in just one component then
  //just do it, otherwise synchronize
  else if(act.IsSend()) {
    if(!SingletonAction(comp,act,false,false,res)) Handshake(comp,act,false,false,res);
  }
  //receive actions: if this action appears in just one component then
  //just do it
  else if(act.IsRecv()) {
    SingletonAction(comp,act,false,false,res);
  }
}

/*********************************************************************/
//get the actions
/*********************************************************************/
void GlobalAbsLts::GetActions(set<Action> &res)
{
  for(vector<Component*>::const_iterator i = Database::components.begin();i != Database::components.end();++i) {
    set<Action> x; (*i)->GetL2AbsActs(x);
    res.insert(x.begin(),x.end());
  }
}

/*********************************************************************/
//get the number of states
/*********************************************************************/
BigInt GlobalAbsLts::GetStateNum()
{
  BigInt res(1);
  for(size_t i = 0;i < Database::components.size();++i) {
    res = res * Database::components[i]->GetL2AbsStateNum();
  }
  return res;
}

/*********************************************************************/
//get the number of states in the previous iteration. this should only
//be called if we are doing incremental verification and the previous
//refinement was an LTS refinement.
/*********************************************************************/
int GlobalAbsLts::GetPrevStateNum()
{
  assert(Database::INC_VERIFY && (Database::lastRefComp != -1));
  int res = 1;
  for(size_t i = 0;i < Database::components.size();++i) {
    if(i == static_cast<size_t>(Database::lastRefComp)) {
      res *= Database::components[i]->GetPrevL2AbsStateNum();
    } else {
      res *= Database::components[i]->GetL2AbsStateNum();
    }
  }
  return res;
}

/*********************************************************************/
//get the set of global states whose successor might have changed due
//to abstraction refinement. this should only be called if we are
//doing incremental verification and the previous refinement was an
//LTS refinement.
/*********************************************************************/
void GlobalAbsLts::GetSuccsChanged(set<BigInt> &res)
{
  assert(Database::INC_VERIFY && (Database::lastRefComp != -1));
  vector< set<L2AbsState> > absInits;
  bool emptyFlag = false;
  for(size_t i = 0;i < Database::components.size();++i) {
    absInits.push_back(set<L2AbsState>());
    if(i == static_cast<size_t>(Database::lastRefComp)) {
      Database::components[i]->GetL2AbsSuccsChanged(absInits.back());
      if(absInits.back().empty()) {
	emptyFlag = true;
	break;
      }
    } else {
      for(int j = 0;j < Database::components[i]->GetL2AbsStateNum();++j) {
	absInits.back().insert(j);
      }
    }
  }
  if(!emptyFlag) CartesianProduct(absInits,0,1,0,res);
}

/*********************************************************************/
//given a global implementation state and an action return the
//possible successors in the previous iteration. this should only be
//called if we are doing incremental verification and the previous
//refinement was an LTS refinement.
/*********************************************************************/
void GlobalAbsLts::GetPrevSuccsOnAction(const BigInt &comp,const Action &act,set<BigInt> &res)
{
  assert(Database::INC_VERIFY && (Database::lastRefComp != -1));
  //basic actions force global sync
  if(act.IsBasic()) Synchronize(comp,act,true,true,res);
  //assign action
  else if(act.IsAssign()) {
    //if this is an assignment to a global variable then synchronize
    //globally
    if(Database::components[0]->IsConcGlobalVar(act.GetLhsExpr())) Synchronize(comp,act,true,true,res);
    //otherwise do not synchronize
    else Interleave(comp,act,true,true,res);
  }
  //return actions force no sync at all
  else if(act.IsReturn()) Interleave(comp,act,true,true,res);
  //broadcast actions force global synchronization
  else if(act.IsBcast()) Synchronize(comp,act,true,true,res);
  //send actions: if this action appears in just one component then
  //just do it, otherwise synchronize
  else if(act.IsSend()) {
    if(!SingletonAction(comp,act,true,true,res)) Handshake(comp,act,true,true,res);
  }
  //receive actions: if this action appears in just one component then
  //just do it
  else if(act.IsRecv()) {
    SingletonAction(comp,act,true,true,res);
  }
}

/*********************************************************************/
//return the set of propositions true in a global state
/*********************************************************************/
void GlobalAbsLts::GetPropositions(const BigInt &comp,set<Expr> &res)
{
  vector<L2AbsState> absStates = CompToAbsStates(comp);
  for(size_t i = 0;i < Database::components.size();++i) {
    set<Expr> prop;
    Database::components[i]->GetL2AbsProps(absStates[i],prop);
    res.insert(prop.begin(),prop.end());
  }
}

/*********************************************************************/
//given an action, a composite state and an index return the action
//from the state that produced it
/*********************************************************************/
Action GlobalAbsLts::GetCompAct(const Action &act,const BigInt &source,const BigInt &sink,int index)
{  
  vector<L2AbsState> absSrc = CompToAbsStates(source);
  vector<L2AbsState> absSink = CompToAbsStates(sink);
  char buf[64];
  snprintf(buf,64,"P%d::STUTTER",index);
  Action stutterAct(buf);
  //basic action
  if(act.IsBasic()) {
    return (Database::components[index]->IsL2AbsTrans(absSrc[index],act,absSink[index])) ? act : stutterAct;
  } 
  //assign action
  else if(act.IsAssign()) {
    if(Database::components[0]->IsConcGlobalVar(act.GetLhsExpr())) {
      for(int i = 0;i < index;++i) {
	if(Database::components[i]->IsL2AbsTrans(absSrc[i],act,absSink[i])) {
	  return (Database::components[index]->IsL2AbsTrans(absSrc[index],act.Complement(),absSink[index])) ? act.Complement() : stutterAct;
	}
      }
      if(Database::components[index]->IsL2AbsTrans(absSrc[index],act,absSink[index])) return act;
      else if(Database::components[index]->IsL2AbsTrans(absSrc[index],act.Complement(),absSink[index])) return act.Complement();
      else return stutterAct;
    } else {
      for(int i = 0;i < index;++i) {
	if(Database::components[i]->IsL2AbsTrans(absSrc[i],act,absSink[i])) return stutterAct;
      }
      return (Database::components[index]->IsL2AbsTrans(absSrc[index],act,absSink[index])) ? act : stutterAct;
    }
  }
  //return action
  else if(act.IsReturn()) {
    for(int i = 0;i < index;++i) {
      if(Database::components[i]->IsL2AbsTrans(absSrc[i],act,absSink[i])) return stutterAct;
    }
    return (Database::components[index]->IsL2AbsTrans(absSrc[index],act,absSink[index])) ? act : stutterAct;
  }
  //broadcast action
  else if(act.IsBcast()) {
    for(int i = 0;i < index;++i) {
      if(Database::components[i]->IsL2AbsTrans(absSrc[i],act,absSink[i])) {
	return (Database::components[index]->IsL2AbsTrans(absSrc[index],act.Complement(),absSink[index])) ? act.Complement() : stutterAct;
      }
    }
    if(Database::components[index]->IsL2AbsTrans(absSrc[index],act,absSink[index])) return act;
    else if(Database::components[index]->IsL2AbsTrans(absSrc[index],act.Complement(),absSink[index])) return act.Complement();
    else return stutterAct;
  }
  //send action
  else if(act.IsSend()) {
    for(int i = 0;i < index;++i) {
      if(Database::components[i]->IsL2AbsTrans(absSrc[i],act,absSink[i])) {
	return (Database::components[index]->IsL2AbsTrans(absSrc[index],act.Complement(),absSink[index])) ? act.Complement() : stutterAct;
      }
    }
    if(Database::components[index]->IsL2AbsTrans(absSrc[index],act,absSink[index])) return act;
    else if(Database::components[index]->IsL2AbsTrans(absSrc[index],act.Complement(),absSink[index])) return act.Complement();
    else return stutterAct;
  }
  //recv action
  else if(act.IsRecv()) {
    for(int i = 0;i < index;++i) {
      if(Database::components[i]->IsL2AbsTrans(absSrc[i],act,absSink[i])) return stutterAct;
    }
    if(Database::components[index]->IsL2AbsTrans(absSrc[index],act,absSink[index])) return act;
    else return stutterAct;
  }
  //impossible
  else assert(false);
}

/*********************************************************************/
//given a global state return true iff it is a deadlocked state
/*********************************************************************/
bool GlobalAbsLts::IsDeadlocked(const BigInt &comp)
{
  vector<L2AbsState> absStates = CompToAbsStates(comp);
  map< Action,set<size_t> > actMap,doMap;
  for(size_t i = 0;i < Database::components.size();++i) {      
    set<Action> acts; Database::components[i]->GetL2AbsActs(acts);
    for(set<Action>::const_iterator j = acts.begin();j != acts.end();++j) {
      actMap[*j].insert(i);
      doMap[*j].insert(i);
    }
  }
  for(size_t i = 0;i < Database::components.size();++i) {      
    set<Action> ref; Database::components[i]->GetL2AbsRefusal(absStates[i],ref);
    for(set<Action>::const_iterator j = ref.begin();j != ref.end();++j) {
      doMap[*j].erase(i);
    }
  }
  set<Action> acts; GetActions(acts);
  for(set<Action>::const_iterator i = acts.begin();i != acts.end();++i) {
    if(i->IsBcast()) {
      for(set<size_t>::const_iterator j = doMap[*i].begin();j != doMap[*i].end();++j) {
	set<size_t> x = doMap[i->Complement()]; x.insert(*j);
	if(x.size() == Database::components.size()) return false;
      }
    } else if(i->IsSend()) {
      if((singActs.count(*i) != 0) && (!doMap[*i].empty())) return false;
      if((doMap[*i].size() > 1) && (!doMap[i->Complement()].empty())) return false;
      else if((!doMap[*i].empty()) && (doMap[i->Complement()].size() > 1)) return false;
      else if((doMap[*i].size() == 1) && (doMap[i->Complement()].size() == 1)) {
	if((*(doMap[*i].begin())) != (*(doMap[i->Complement()].begin()))) return false;
      }
    } else if(i->IsRecv()) {
      if((singActs.count(*i) != 0) && (!doMap[*i].empty())) return false;
    } else if(i->IsAssign()) {
      if(!doMap[*i].empty()) {
	if(!Database::HANDLE_DATA_COMM) return false;
	if(Database::components[0]->IsConcGlobalVar(i->GetLhsExpr())) {
	  for(set<size_t>::const_iterator j = doMap[*i].begin();j != doMap[*i].end();++j) {
	    set<size_t> x = doMap[i->Complement()]; x.insert(*j);
	    if(x.size() == Database::components.size()) return false;
	  }
	} else return false;
      }
    } else if(i->IsReturn()) {
      if(!doMap[*i].empty()) return false;
    } else if(i->IsBasic()) {
      if(actMap[*i] == doMap[*i]) return false;
    } else assert(i->IsComplement());
  }
  return true;
}

/*********************************************************************/
//given a global state id return the vector of component state ids
/*********************************************************************/
vector<L2AbsState> GlobalAbsLts::CompToAbsStates(const BigInt &comp)
{
  vector<L2AbsState> res;
  BigInt quo = comp;
  for(vector<Component*>::const_iterator i = Database::components.begin();i != Database::components.end();++i) {
    int x = (*i)->GetL1AbsStateNum();
    BigInt rem;
    BigInt::Div(quo,rem,quo,x);
    res.push_back(rem.ToSL());
  }
  return res;
}

/*********************************************************************/
//given a global state id return the vector of component state ids
/*********************************************************************/
string GlobalAbsLts::CompToString(const BigInt &comp)
{
  string res = "( ";
  vector<L2AbsState> absStates = CompToAbsStates(comp);
  for(size_t i = 0;i < Database::components.size();++i) {      
    char x[128];
    snprintf(x,128,"%d ",absStates[i]);
    res += x;
  }
  res += ")";
  return res;
}

/*********************************************************************/
//end of GlobalAbsLts.cpp
/*********************************************************************/
