/******************************** Piffle *****************************

* FileName [Explain.cpp]

* PackageName [main]

* Synopsis [Method definitions of Explain class.]

* SeeAlso [Explain.h]

* Author [Alex Groce]

* Copyright [ Copyright (c) 2004 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <sys/types.h>
#include <gmp.h>
#include <unistd.h>
#include <string>
#include <list>
#include <set>
#include <map>
#include <vector>
using namespace std;

#include "BigInt.h"
#include "Util.h"
#include "Timer.h"
#include "Statistics.h"
#include "Node.h"
#include "Action.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "ContLoc.h"
#include "ImplState.h"
#include "LocalAbsLts.h"
#include "LtsInfo.h"
#include "Buchi.h"
#include "LtlChecker.h"
#include "LtlManager.h"
#include "LtlFormula.h"
#include "GlobalCEDag.h"
#include "ModelExtractor.h"
#include "ConcCEDag.h"
#include "CEDagVerifier.h"
#include "Component.h"
#include "ProcManager.h"
#include "Expr.h"
#include "Explain.h"
#include "string.h"
using namespace magic;

void Explain::Run()
{
  ProcManager &pm = static_cast<ProcManager&>(component);
  Util::Message(1,"Explaining the counterexample.\n");
  //CE.Display(component);  It's already printed, darn it

  if (Database::NO_SILENT_TRANS)
    Util::Error ("Explanation likely to fail without --silentTrans.\n");

  if ((Database::CONF_TYPE != Database::CONF_REACH) &&
      (Database::CONF_TYPE != Database::CONF_LTL))
    Util::Error 
      ("Can only explain reachability (--trace?) or LTL counterexamples.\n");

  LTL = (Database::CONF_TYPE == Database::CONF_LTL);

  CESize = CE.stateLabel.size();
  Util::Message(3, "CESize: %d\n",CESize);
  unwind = CESize + Database::ADDITIONAL_UNWIND;
  if (LTL) {
    if (CE.loopBegin != -1) {
      unwind += (CE.loop.size());
    }
  }

  // Extract the counterexample implementation locations.
  
  CEContLocs.clear(); CEContLocs.resize(CESize);
  CEPVals.clear(); CEPVals.resize(CESize);
  CEActions.clear(); CEActions.resize(CESize);
  
  for (size_t i = 0; i < CESize; ++i) {
    ImplState* is = (ImplState*)(static_cast<const CL1AbsState*>
				 (CE.stateLabel[i])->GetImplState());
    CEActions[i] = (Action)(CE.actLabel[i]);
    CEContLocs[i] = is->GetContLoc();
    CEPVals[i] = is->GetPredVal();
  }

  bool retry = true;
  bool first_time = true;

  while (retry && (first_time || unwind < (Database::MAX_UNWIND))) {
    first_time = false;
    
    Statistics::exUnwindTimer.Start();

    Statistics::explainIterDepthNum++;
    Util::Message(1, "starting depth iteration number %i\n", 
		  Statistics::explainIterDepthNum);
    Util::Message(1, "Unwinding to max depth of %i...\n",unwind);

    //initialise clause file and clause number
    clauseFile = fopen("explain.clauses","w");
    assert(clauseFile != NULL);
    clauseNum = 0;

    stop_unwind = unwind;
    SATVars = 1;
    
    maxPValDelta = 0;
    maxDelta = 0;
    ActDeltaWeight = Database::ACTION_WEIGHT;
    PredDeltaWeight = Database::PRED_WEIGHT;
    deltas.clear();

    PredValDeltas.clear();
    ActionDeltas.clear();
    ContLocDeletions.clear();
    ContLocInsertions.clear();
    
    // Extract the set of implementation locations.

    set<const ImplState*> CurrImpls = 
      (pm.GetInitLoc ())->GetImplStates(pm.GetGuard ());

    set<int> CurrLts;

    // Initialize mappings:
    
    ContLocMapping.clear(); ContLocMapping.resize(unwind);
    ActionMapping.clear(); ActionMapping.resize(unwind);
    RevActionMapping.clear();
    PValMapping.clear(); PValMapping.resize(unwind);
    PredValMapping.clear();PredValMapping.resize(unwind);
    SpecMapping.clear();SpecMapping.resize(unwind);

    AlignmentMapping.clear();
    RevAlignmentMapping.clear();
    Unalignable.clear();

    RevLoopMapping.clear();

    ImplMapping.clear(); ImplMapping.resize(unwind);
    RevImplMapping.clear();
    PossibleLoops.clear();
    AcceptingMapping.clear();
    TransMapping.clear(); TransMapping.resize(unwind);
    BuchiLtsMapping.clear();BuchiLtsMapping.resize(unwind);
    RevBuchiLtsMapping.clear();

    EmptyMapping.clear();

    // Set up initial constraints.

    AddInitialConstraints(CurrImpls, 0);

    // Handle the specification.
    
    if (!LTL) {
      const LtsInfo &specInfo = Database::GetLtsInfo(Database::specLtsName);
      errId = specInfo.GetStateId(Database::ERROR_STATE);
      Util::Message (3, "Error id = %d\n", errId);
      int InitId = specInfo.GetStateId(specInfo.GetInitState());
      CurrLts.insert(InitId);
      vector<int> InitLts;
      InitLts.push_back(GetBuchiLtsVar(InitId, 0));
      AddClause(InitLts);
    }

    // Unwind the transition relation.

    set<const ImplState*> NextImpls;
    set<int> NextLts;

    for (size_t i = 1; (i < unwind) && (!CurrImpls.empty()); ++i) {
      NextImpls.clear();
      NextLts.clear();
      AddTransitionConstraints(CurrImpls,i,NextImpls,CurrLts,
			       NextLts);
      CurrImpls = NextImpls;
      CurrLts = NextLts;
      if (CurrImpls.empty()) {
	Util::Message(1,"Stopped unwinding transition relation at %i.\n");
	stop_unwind = i;
      }
    }

    // Force actions!
    if (0) {
      set <size_t> AtMostOneAction;
 
      for (set<const ImplState*>::const_iterator i = CurrImpls.begin();
	   i != CurrImpls.end(); i++) {
	const ImplState* is = *i;
	size_t isi = GetImplVar(0,0,0,is,unwind-1);
	map<Action,set<const ImplState*> > sa;
	Action act;
	size_t acti;

	set<size_t> AtLeastOneAction;

	is->GetSuccsAndActions(set<Action>(),set<Action>(),sa);
	for (map<Action,set<const ImplState*> >::const_iterator j = sa.begin();
	     j != sa.end(); ++j) {
	  act = (*j).first;
	  acti = GetActionVar(act,unwind-1);
	  AtLeastOneAction.insert(acti);
	  AtMostOneAction.insert(acti);
	}

	vector<int> AtLeastOneActionClause;
	AtLeastOneActionClause.push_back(-isi);
	VectorInsert(AtLeastOneAction,AtLeastOneActionClause);
	AddClause(AtLeastOneActionClause);
      }
  
      AddMutexConstraints(AtMostOneAction);
    }

    Util::Message(3,"Possible final steps for counterexample:\n");
    for (set<const ImplState*>::const_iterator i = NextImpls.begin ();
	 i != NextImpls.end(); ++i) {
      Util::Message(3,(*i)->ToString() + "\n");
    }
  
    // If doing LTL, now unwind the Buchi states.
    if (LTL) {
      AddBuchiConstraints();
    }

    // Produce delta constraints.

    for (size_t i = 0; i < CESize; ++i) {
      AddAlignmentAndDeltas(i);
    }
  
    // Deltas for successful execution (only control location).
  
    for (size_t i = 0; i < stop_unwind; ++i) {
      vector<int> AllAligns;
      for (size_t j = 0; j < CESize; ++j) {
	int ai = GetAlignmentVar(j,i,CEContLocs[j]);
	if (ai != -1) 
	  AllAligns.push_back(-ai);
      }
      AllAligns.push_back(-EmptyMapping[i]);
      size_t NotAligned = SATVars++;
      Util::Message(3,"Not Aligned (insert) for %d = %d\n",i,NotAligned);
      ContLocInsertions[NotAligned] = i;
      AddEquivalence(AllAligns,NotAligned);
      deltas.push_back(pair<int,int>(NotAligned,maxPValDelta+ActDeltaWeight+1));
      maxDelta += maxPValDelta+ActDeltaWeight+1;
    }

    Statistics::exUnwindTimer.Stop();

    Statistics::exSearchTimer.Start();

    // Solve for non-spurious similar successful execution.

    bool spurious = true;
    retry = false;

    Util::Message(1,"Searching for non-spurious successful execution...\n");
    for (size_t i = 1; spurious && (i <= Database::SPURIOUS_TRIES); ++i) {
      Util::Message(1,"starting iteration number %i ...\n",i);
      Statistics::explainIterSpurNum++;
      if (GenerateAndSolvePBS()) {
	spurious = DisplayPBSResultsSpurious(pm);
	if (spurious) {
	  Util::Message(1,"Successful execution is spurious.\n");
	}
      } else {
	Util::Message(1,"** There are no successful executions! **\n");
	if ((Database::MORE_UNWIND) && (stop_unwind == unwind)) {
	  Util::Message(1, "Attempting to unwind additional steps...\n");
	  retry = true;
	}
	spurious = false;
      }
    }

    if (spurious) {
      Util::Message 
	(1,"Giving up on finding non-spurious successful execution.\n");
      retry = false;
    }
    unwind++;
    Statistics::exSearchTimer.Stop();
  }
  
  Util::Message(1,"Done explaining.\n");
}

void Explain::VectorInsert(const set<size_t> &s, vector<int> &v) {
  v.insert(v.end(),s.begin(),s.end());
}

void Explain::VectorNegInsert(const set<size_t> &s, vector<int> &v) {
  for (set<size_t>::const_iterator i = s.begin(); i != s.end(); ++i)
    v.push_back(-(*i));
}

size_t Explain::GetContLocVar(ContLoc* cl, size_t index) {
  map<ContLoc*,size_t>::const_iterator f = ContLocMapping[index].find(cl);
  if (f != ContLocMapping[index].end()){
    return (*f).second;
  }
  Util::Message(3,"%d : location %s gets SAT variable %d\n",
		index,cl->ToString().c_str(),SATVars);
  ContLocMapping[index][cl] = SATVars;
  return SATVars++;
}

size_t Explain::GetActionVar(const Action &act, size_t index) {
  map<Action,size_t>::const_iterator f = ActionMapping[index].find(act);
  if (f != ActionMapping[index].end()){
    return (*f).second;
  }
  Util::Message(3,"%d : action %s gets SAT variable %d\n",
		index,act.ToString().c_str(),SATVars);
  RevActionMapping[SATVars] = pair<size_t,Action>(index,act);
  ActionMapping[index][act] = SATVars;
  return SATVars++;
}

size_t Explain::GetPValVar(const PVal &pv, size_t index, ContLoc* cl) {
  map<pair<PVal,ContLoc*>,size_t>::const_iterator f = 
    PValMapping[index].find(pair<PVal,ContLoc*>(pv,cl));
  if (f != PValMapping[index].end()){
    return (*f).second;
  }
  Util::Message(3,"%d : PVal gets SAT variable %d\n",index,SATVars);
  PValMapping[index][pair<PVal,ContLoc*>(pv,cl)] = SATVars;

  assert(cl != 0);
  size_t cli = GetContLocVar(cl,index);
  PredSet PS = cl->GetPreds();
  PredVal PV = PS.PValToPredVal(pv);

  size_t pvi = SATVars;

  SATVars++;

  if (!PV.second.empty()) {
    
    if ((PV.second.size() * PredDeltaWeight) > maxPValDelta)
      maxPValDelta = (PV.second.size() * PredDeltaWeight);
    
    vector<int> PVClause;
    for (size_t i = 0; i < PV.second.size(); ++i) {
      size_t PVi = GetPredValVar
	(pair<size_t,pair<size_t,short> >
	 (cli,pair<size_t,short>(i,PV.second[i])), index);
      PVClause.push_back(PVi);
    }
    
    AddEquivalence(PVClause,pvi);
  }

  return pvi;
}

size_t Explain::GetPredValVar(const pair<size_t,pair<size_t,short> > &PV, 
			      size_t index) {
  map<pair<size_t,pair<size_t,short> >,size_t>::const_iterator f = 
    PredValMapping[index].find(PV);
  if (f != PredValMapping[index].end()){
    return (*f).second;
  }
  Util::Message(3,"%d : PredVal gets SAT variable %d\n",index,SATVars);
  PredValMapping[index][PV] = SATVars;

  return SATVars++;
}

size_t Explain::GetSpecVar(const pair <short,short> &s, size_t index) {
  map<pair<short,short>,size_t>::const_iterator f =
    SpecMapping[index].find(s);
  if (f != SpecMapping[index].end()){
    return (*f).second;
  }
  Util::Message(3,"%d : spec pair (%d,%d) gets SAT variable %d\n",
		index,s.first,s.second,SATVars);
  SpecMapping[index][s] = SATVars;

  return SATVars++;
}

size_t Explain::GetImplVar(size_t cli, size_t pvi, size_t speci, 
			   const ImplState* is, size_t index) {
  map<const ImplState*,size_t>::const_iterator f = 
    ImplMapping[index].find(is);
  if (f != ImplMapping[index].end()){
    return (*f).second;
  }
  ImplMapping[index][is] = SATVars;
  RevImplMapping[SATVars] = pair<size_t,const ImplState*>(index,is);

  // Add equivalence requirement clause to SAT.
  Util::Message(3,"%d : ImplState:  (%d, %d, %d) = %d\n",index, cli, pvi, 
		speci, SATVars);

  vector <int> v;
  v.push_back(cli);
  v.push_back(pvi);
  v.push_back(speci);
  AddEquivalence(v, SATVars);

  return SATVars++;
}

size_t Explain::GetTransVar(const pair<size_t,size_t> &t, size_t index) {
  map<pair<size_t,size_t>,size_t>::const_iterator f = 
    TransMapping[index].find(t);
  if (f != TransMapping[index].end()){
    return (*f).second;
  }
  TransMapping[index][t] = SATVars;

  // Add equivalence requirement clause to SAT.
  size_t acti = t.first;
  vector <int> v;
  v.push_back(acti);
  v.push_back(t.second);
  AddEquivalence(v, SATVars);

  Util::Message(3,"%d : Trans:  (%d, %d) = %d\n",
		index, acti, t.second, SATVars);

  return SATVars++;
}

size_t Explain::GetBuchiLtsVar(int id, size_t index) {
  map<int,size_t>::const_iterator f = BuchiLtsMapping[index].find(id);
  if (f != BuchiLtsMapping[index].end()){
    return (*f).second;
  }
  
  BuchiLtsMapping[index][id] = SATVars;
  RevBuchiLtsMapping[SATVars] = pair<size_t,int>(index,id);

  if ((!LTL) && (id == errId)) {
    Util::Message(3,"LTS %d (SAT %d) is error location.\n",id,SATVars);
    vector<int> NoError;
    NoError.push_back(-SATVars);
    AddClause(NoError);
  }

  Util::Message(3,"%d : LTS/Buchi %d = %d\n",index, id, SATVars);

  return SATVars++;
}

int Explain::GetAlignmentVar(size_t s0, size_t s1, ContLoc* cl) {
  if(ContLocMapping[s1].empty())
    Util::Message(1,"WARNING!  WARNING!  Empty ContLocMapping for %d\n",
		  s1);
  map<ContLoc*,size_t>::const_iterator cf = ContLocMapping[s1].find(cl);
  if (cf == ContLocMapping[s1].end()){
    return -1;
  }

  pair<size_t,size_t> a = pair<size_t,size_t>(s0,s1);

  map<pair<size_t,size_t>,size_t>::const_iterator f = AlignmentMapping.find(a);
  if (f != AlignmentMapping.end()) {
    Util::Message(3, "looking alignment up for %d,%d\n", s0,s1);
    Util::Message(3, "  result = %d\n", f->second);
    return (*f).second;
  }
  
  AlignmentMapping[a] = SATVars;
  RevAlignmentMapping[SATVars] = a;

  Util::Message(3,"Align (%d,%d) = %d\n", a.first, a.second, SATVars);

  return SATVars++;
}

bool Explain::PropCompatible(const ImplState* is, int specState) {
  set <Expr> implProps;
  is->GetPropositions(implProps);
  
  const set<Expr> posProps = Buchi::GetPosProps(specState);
  const set<Expr> negProps = Buchi::GetNegProps(specState);
  for(set<Expr>::const_iterator i = posProps.begin();i != posProps.end();++i) {
    if(implProps.count(*i) == 0) return false;
  }
  for(set<Expr>::const_iterator i = negProps.begin();i != negProps.end();++i) {
    if(implProps.count(*i) != 0) return false;
  }  
  return true;
}

/*********************************************************************/
//write a new clause to the file and increment the clause number
/*********************************************************************/
void Explain::AddClause(const vector<int> &arg)
{
  Util::Message(3, "Adding clause: ");
  for(vector<int>::const_iterator i = arg.begin();i != arg.end();++i) {
    fprintf(clauseFile,"%d ",*i);
    Util::Message(3,"%d ",*i);
  }
  fprintf(clauseFile,"0\n");
  Util::Message(3,"\n");
  ++clauseNum;
}

void Explain::AddEquivalence(const vector<int> &v, int iv) {
  vector<int> c1;
  c1.push_back(iv);
  for (vector<int>::const_iterator i = v.begin(); i != v.end(); ++i) {
    vector<int> c2;
    c2.push_back(-iv);
    c2.push_back(*i);
    AddClause(c2);
    c1.push_back(-(*i));
  }
  AddClause(c1);
}

void Explain::AddMutexConstraints (const set<size_t> &s) {
  for (set<size_t>::const_iterator i = s.begin(); i != s.end(); ++i) {
    for (set<size_t>::const_iterator j = s.begin(); 
	 j != s.lower_bound(*i); ++j) {
      vector<int> mutex;
      mutex.push_back(-(*i));
      mutex.push_back(-(*j));
      AddClause(mutex);
    }
  }
}

void Explain::AddInitialConstraints (const set<const ImplState*> &iset, 
				     size_t index) {
  set<size_t> AtLeastOne;
  ContLoc* cl;
  size_t cli;
  PVal pv;
  size_t pvi;
  
  for (set<const ImplState*>::const_iterator i = iset.begin();
       i != iset.end(); ++i) {
    cl = (*i)->GetContLoc();
    cli = GetContLocVar(cl,index);
    pv = (*i)->GetPredVal();
    pvi = GetPValVar(pv,index,cl);
    short sst = (*i)->GetSpecStateId();
    short sinfo = (*i)->GetSpecInfoId();
    pair<short,short> spec (sst, sinfo);
    size_t speci = GetSpecVar(spec,index);

    size_t isi = GetImplVar(cli,pvi,speci,*i,index);
    //RevImplMapping[isi] = pair<size_t,const ImplState*>(index, *i);
    AtLeastOne.insert(isi);
  }

  vector<int> AtLeastOneClause;
  VectorInsert(AtLeastOne,AtLeastOneClause);
  AddClause(AtLeastOneClause);
  AddMutexConstraints(AtLeastOne);

  size_t Empty = SATVars++;
  Util::Message (3,"Empty for %d = %d.\n",index,Empty);
  EmptyMapping.push_back(Empty);
  vector<int> NotEmptyClause;
  NotEmptyClause.push_back(-Empty);
  AddClause(NotEmptyClause);
}

void Explain::AddImplStateConstraints (const ImplState* is, size_t index,
				       set<const ImplState*> &NextStates,
				       set<size_t> &AllContLocs,
				       set<size_t> &AllPVals,
				       set<size_t> &AllActions,
				       set<size_t> &AllEmpty) {
  map<Action,set<const ImplState*> > sa;
  is->GetSuccsAndActions(set<Action>(),set<Action>(),sa);
  
  set<size_t> AtLeastOneAction;

  Util::Message(3,"STEP %d: \n", index);
  Util::Message(3,is->ToString());

  // if doing LTL, add to possible indexs == implstate)
  if (LTL) {
    map<const ImplState*,set<size_t> >::iterator floop =
      PossibleLoops.find(is);
    if (floop != PossibleLoops.end()) {
      (floop->second).insert(index-1);
    } else {
      set<size_t> newSet;
      newSet.insert(index-1);
      PossibleLoops[is] = newSet;
    }
  }

  Action act;
  size_t acti;
  size_t transi;
  ContLoc* cl = is->GetContLoc();
  size_t cli = GetContLocVar(cl,index-1);
  PVal pv = is->GetPredVal();
  size_t pvi = GetPValVar(pv,index-1,cl);
  short sst = is->GetSpecStateId();
  short sinfo = is->GetSpecInfoId();
  pair<short,short> spec (sst, sinfo);
  size_t speci = GetSpecVar(spec,index-1);

  size_t isi = GetImplVar(cli,pvi,speci,is,index-1);

  Util::Message(3," (%d)\n",isi);

  ContLoc* ncl;
  size_t ncli;
  PVal npv;
  size_t npvi;
  //size_t nisi;

  if (sa.empty()) {
    AllEmpty.insert(isi);
    return; // No point in checking through successors if there are none!
    // Actually, not returning here breaks things--adds a bad clause.
  }

  for (map<Action,set<const ImplState*> >::const_iterator i = sa.begin();
       i != sa.end(); ++i) {
    set <size_t> AtLeastOneSuccessor;
    act = (*i).first;
    acti = GetActionVar(act,index-1);
    AllActions.insert(acti);
    AtLeastOneAction.insert(acti);

    transi = GetTransVar(pair<size_t,size_t>(acti,isi),index);

    for (set<const ImplState*>::const_iterator j = (*i).second.begin();
	 j != (*i).second.end(); ++j) {
      NextStates.insert(*j);
      ncl = (*j)->GetContLoc();
      ncli = GetContLocVar(ncl,index);
      npv = (*j)->GetPredVal();
      npvi = GetPValVar(npv,index,ncl);
      short nsst = (*j)->GetSpecStateId();
      short nsinfo = (*j)->GetSpecInfoId();
      pair<short,short> nspec (nsst, nsinfo);
      size_t nspeci = GetSpecVar(nspec,index);

      size_t nisi = GetImplVar(ncli,npvi,nspeci,*j,index);
      //RevImplMapping[nisi] = pair<size_t,const ImplState*>(index,*j);

      AllContLocs.insert(ncli);
      AllPVals.insert(npvi);
      AtLeastOneSuccessor.insert(nisi);
    }

    vector<int> AtLeastOneSuccessorClause;
    AtLeastOneSuccessorClause.push_back(-transi);
    VectorInsert(AtLeastOneSuccessor,AtLeastOneSuccessorClause);
    AddClause(AtLeastOneSuccessorClause);
  }

  vector<int> AtLeastOneActionClause;
  AtLeastOneActionClause.push_back(-isi);
  VectorInsert(AtLeastOneAction,AtLeastOneActionClause);
  AddClause(AtLeastOneActionClause);
}


void Explain::AddTransitionConstraints (const set<const ImplState*> &iset, 
					size_t index,
					set<const ImplState*> &NextStates,
					const set<int> &CurrLts,
					set<int> &NextLts) {
  set<size_t> AllContLocs;
  set<size_t> AllPVals;
  set<size_t> AllActions;
  set<size_t> AllEmpty;

  for (set<const ImplState*>::const_iterator i = iset.begin(); 
       i != iset.end(); ++i) {
    //set<const ImplState*> Next;
    AddImplStateConstraints(*i,index,NextStates,AllContLocs,AllPVals,
			    AllActions,AllEmpty);
  }

  AddMutexConstraints(AllContLocs);
  AddMutexConstraints(AllPVals);
  AddMutexConstraints(AllActions);

  // Add predicate value mutexes.

  map<size_t,map<size_t,set<size_t> > > PredValMutexes;

  for (map<pair<size_t,pair<size_t,short> >, size_t>::const_iterator i =
	 PredValMapping[index].begin(); 
       i != PredValMapping[index].end(); ++i) {
    size_t cli = i->first.first;
    size_t predi = i->first.second.first;
    size_t mutex = i->second;
    map<size_t,map<size_t,set<size_t> > >::iterator fcli =
      PredValMutexes.find(cli);
    if (fcli == PredValMutexes.end()) {
      map<size_t,set<size_t> > prediMap;
      set<size_t> mutexSet;
      mutexSet.insert(mutex);
      prediMap[predi] = mutexSet;
      PredValMutexes[cli] = prediMap;
    } else {
      map<size_t,set<size_t> >::iterator fpredi =
	fcli->second.find(predi);
      if (fpredi == (fcli->second.end())) {
	set<size_t> mutexSet;
	mutexSet.insert(mutex);
	fcli->second[predi] = mutexSet;
      } else {
	fpredi->second.insert(mutex);
      }
    }
  }

  for (map<size_t,map<size_t,set<size_t> > >::const_iterator i =
	 PredValMutexes.begin(); i != PredValMutexes.end(); ++i) {
    for (map <size_t,set<size_t> >::const_iterator j = i->second.begin();
	 j != i->second.end(); ++j) {
      AddMutexConstraints(j->second);
    }
  }

  // Handle LTS.

  if (!LTL) {
    const LtsInfo &specInfo = Database::GetLtsInfo(Database::specLtsName);
    for (set<size_t>::const_iterator i = AllActions.begin(); 
	 i != AllActions.end (); ++i) {
      Action act = (RevActionMapping[*i]).second;
      Util::Message(3,"Action = " + (act.ToString()) + "\n");
      for (set<int>::const_iterator j = CurrLts.begin(); 
	   j!=CurrLts.end(); ++j) {
	Util::Message(3,"For LTS state %s (%d):\n",
		      specInfo.GetIdState(*j).c_str(),*j);
	size_t ltsi = GetBuchiLtsVar(*j,index-1);
	set<string> res;
	specInfo.GetSuccsOnActionSpec(*j,act,res);
	for (set<string>::const_iterator k = res.begin(); k!=res.end(); ++k) {
	  int id = specInfo.GetStateId(*k);
	  Util::Message(3,"  LTS Successor: %s (%d)\n",k->c_str(),id);
	  NextLts.insert(id);
	  size_t idi = GetBuchiLtsVar(id,index);
	  size_t transi = GetTransVar(pair<size_t,size_t>(*i,ltsi), index);
	  vector<int> LtsClause;
	  LtsClause.push_back(-transi);
	  LtsClause.push_back(idi);
	  AddClause(LtsClause);
	}
      }
    }
  }

  size_t Empty = SATVars++;
  Util::Message (3,"Empty for %d = %d.\n",index,Empty);
  EmptyMapping.push_back(Empty);
  vector<int> EmptyContLocs;
  vector<int> EmptyPVals;
  vector<int> EmptyActs;
  VectorNegInsert(AllContLocs,EmptyContLocs);
  VectorNegInsert(AllPVals,EmptyPVals);
  VectorNegInsert(AllActions,EmptyActs);
  AddEquivalence(EmptyContLocs,Empty);
  AddEquivalence(EmptyPVals,Empty);
  AddEquivalence(EmptyActs,Empty);

  vector<int> EmptyClause;
  EmptyClause.push_back(-(EmptyMapping[index-1]));
  EmptyClause.push_back(Empty);
  AddClause(EmptyClause);

  for (set<size_t>::const_iterator i = AllEmpty.begin(); 
       i != AllEmpty.end(); ++i) {
    vector<int> EmptyTrans;
    EmptyTrans.push_back(-(*i));
    EmptyTrans.push_back(Empty);
    AddClause(EmptyTrans);
  }
}

void Explain::AddAlignmentAndDeltas(size_t index) {
  
  Util::Message(3,"Calling AddAlignmentAndDeltas for %i\n",index);

  ContLoc* cl = CEContLocs[index];
  PVal pv = CEPVals[index];

  PredSet PS = cl->GetPreds();
  PredVal PV = PS.PValToPredVal(pv);

  set<size_t> AllAligns;

  for (size_t i = 0; i < stop_unwind; ++i) {
    int ai = GetAlignmentVar(index,i,cl);

    if (ai != -1) {
      size_t cli = GetContLocVar(cl,i);
      Util::Message(3,"Insert %d into AllAligns.\n",ai);
      AllAligns.insert(ai);
      
      vector<int> Matches;
      Matches.push_back(cli);

      for (size_t j = 0; j < CESize; ++j) {
	if (j != index) {
	  int xai = GetAlignmentVar(j,i,CEContLocs[j]);
	  if (xai != -1)
	    Matches.push_back(-xai);
	}
      }

      for (size_t j = 0; j < stop_unwind; ++j) {
	if (j != i) {
	  int xai = GetAlignmentVar(index,j,cl);
	  if (xai != -1)
	    Matches.push_back(-xai);
	}
      }

      for (size_t j = index+1; j < CESize; ++j) {
	for (size_t k = 0; k < i; ++k) {
	  int xai = GetAlignmentVar(j,k,CEContLocs[j]);
	  if (xai != -1)
	    Matches.push_back(-xai);
	}
      }

      for (size_t j = 0; j < index; ++j) {
	for (size_t k = i + 1; k < stop_unwind; ++k) {
	  int xai = GetAlignmentVar(j,k,CEContLocs[j]);
	  if (xai != -1)
	    Matches.push_back(-xai);
	}
      }
      
      if (index == (CESize-1))
	Util::Message(3,"POSSIBLE CULPRIT BEGIN\n");
      AddEquivalence(Matches,ai);
      if (index == (CESize-1))
	Util::Message(3,"POSSIBLE CULPRIT END\n");

      for (size_t j = 0; j < PV.second.size(); ++j) {
	size_t Delta = SATVars++;
	PredValDeltas[Delta] = pair<size_t,size_t>(index,j);
	Util::Message(3,"Predicate delta for %i @ %i = %i\n",index,j,Delta);
	Util::Message(3,"  for alignment %i\n",i);
	vector<int> DeltaClause;
	pair<size_t,pair<size_t,short> > lPV = 
	  (pair<size_t,pair<size_t,short> >
	   (cli,pair<size_t,short>(j,PV.second[j])));
	map<pair<size_t,pair<size_t,short> >,size_t>::const_iterator f = 
	  PredValMapping[i].find(lPV);
	DeltaClause.push_back(ai);
	if (f != PredValMapping[i].end()) {
	  DeltaClause.push_back(-((*f).second));
	}
	AddEquivalence(DeltaClause,Delta);
	deltas.push_back(pair<int,int>(Delta,PredDeltaWeight));
	maxDelta += PredDeltaWeight;
      }

      Action act = CEActions[index];
      size_t ActDelta = SATVars++;
      ActionDeltas[ActDelta] = index;
      vector<int> ActDeltaClause;
      Util::Message(3,"Adding action delta %d\n",ActDelta);
      map<Action,size_t>::const_iterator f = ActionMapping[i].find(act);
      ActDeltaClause.push_back(ai);
      if (f != ActionMapping[i].end()) {
	ActDeltaClause.push_back(-((*f).second));
      }
      AddEquivalence(ActDeltaClause,ActDelta);
      deltas.push_back(pair<int,int>(ActDelta,ActDeltaWeight));
      maxDelta += ActDeltaWeight;
    }
    
  }
  
  AddMutexConstraints(AllAligns);

  size_t NotAligned = SATVars++;
  ContLocDeletions[NotAligned] = index;
  Util::Message(3,"Not Aligned for %d = %d\n",index,NotAligned);
  Util::Message(3,"Control location = " + cl->ToString() + "\n");
  vector<int> NoAligns;
  VectorNegInsert(AllAligns,NoAligns);
  Util::Message(3,"Adding equivalence.\n");
  AddEquivalence(NoAligns,NotAligned);
  Util::Message(3,"Done adding equivalence.\n");
  if (AllAligns.empty()) {
    Unalignable.insert(index);
  }
  deltas.push_back(pair<int,int>(NotAligned,maxPValDelta+ActDeltaWeight+1));
  maxDelta += maxPValDelta+ActDeltaWeight+1;
}

void Explain::AddBuchiConstraints() {
  if (Database::LTL_NEG) {
    LtlFormula specFormula = Database::GetLtlFormula(Database::specLtsName);
    LtlFormula negSpec = LtlManager::NegateLtl(specFormula);
      
    Buchi::Initialize(negSpec); // NEGATED SPECIFICATION BUCHI
    Util::Message(1,"Automaton for explanation:\n");
    Buchi::Display();
  }

  Util::Message(3, "Unwinding Buchi\n");

  set<int> CurrBuchi;
  set<int> NextBuchi;

  // Handle index = 0;
  Buchi::GetInitStates(CurrBuchi);

  vector<int> SomeInitial;

  for (set<int>::const_iterator i = CurrBuchi.begin();
       i != CurrBuchi.end(); ++i) {
    size_t buchii = GetBuchiLtsVar(*i,0);

    SomeInitial.push_back(buchii);

    vector<int> ReasonsNot;
    ReasonsNot.push_back(buchii);

    set<int> next = Buchi::GetSuccs(*i);
    vector<int> NoSucc;
    for (set<int>::const_iterator j = next.begin(); j != next.end(); ++j) {
      int nbuchii = GetBuchiLtsVar(*j,1);
      NoSucc.push_back(-nbuchii);
    }

    size_t NoSuccessor = SATVars++;
    AddEquivalence(NoSucc,NoSuccessor);

    ReasonsNot.push_back(NoSuccessor);

    vector<int> NotClause;
    NotClause.push_back(-buchii);
    NotClause.push_back(-NoSuccessor);
    AddClause(NotClause);

    Util::Message(3,"No successor for %d = %d\n",*i,NoSuccessor);

    for (map<const ImplState*,size_t>::const_iterator j =
	   ImplMapping[0].begin(); j!=ImplMapping[0].end(); ++j) {
      size_t isi = (j->second);
      const ImplState* is = (j->first);
      if (!PropCompatible(is,*i)) {
	vector<int> NotClause;
	NotClause.push_back(-isi);
	NotClause.push_back(-buchii);
	AddClause(NotClause);
	ReasonsNot.push_back(isi);
      }
    }
      
    for (map<Action,size_t>::const_iterator j = ActionMapping[0].begin(); 
	 j != ActionMapping[0].end(); ++j) {
      size_t acti = (j->second);
      Action act = (j->first);
      if (!LtlChecker::ActCompatible(act,*i)) {
	Util::Message (3,"Action %s is NOT compatible with Buchi %i\n",
		       act.ToString().c_str(), *i);
	vector<int> NotClause;
	NotClause.push_back(-acti);
	NotClause.push_back(-buchii);
	AddClause(NotClause);
	ReasonsNot.push_back(acti);
      } else {
	Util::Message (3,"Action %s IS compatible with Buchi %i\n",
		       act.ToString().c_str(), *i);
      }
    }
      
    AddClause(ReasonsNot);
  }

  AddClause(SomeInitial);

  for (size_t index = 1; index < stop_unwind; ++index) {

    NextBuchi.clear ();

    map<int,set<size_t> > PredMap;
    for (set<int>::const_iterator i = CurrBuchi.begin();
	 i != CurrBuchi.end(); ++i) {
      size_t buchii = GetBuchiLtsVar(*i,index-1);
      set<int> next = Buchi::GetSuccs(*i);
      for (set<int>::const_iterator j = next.begin();
	   j != next.end(); ++j) {
	NextBuchi.insert(*j);
	map<int,set<size_t> >::iterator pmf = PredMap.find(*j);
	if (pmf != PredMap.end()) {
	  (pmf->second).insert(buchii);
	} else {
	  set<size_t> newSet;
	  newSet.insert(buchii);
	  PredMap[*j] = newSet;
	}
      }
    }
      
    for (set<int>::const_iterator i = NextBuchi.begin();
	 i != NextBuchi.end (); ++i) {
      size_t buchii = GetBuchiLtsVar(*i,index);
      set<size_t> preds = PredMap[*i];
	
      size_t NoPred = SATVars++;
      vector<int> NoPredClause;
      VectorNegInsert(preds,NoPredClause);
      AddEquivalence(NoPredClause,NoPred);
	
      Util::Message(3,"No predecessors for %i = %i\n",*i,NoPred);
	
      vector<int> NotClause;
      NotClause.push_back(-buchii);
      NotClause.push_back(-NoPred);
      AddClause(NotClause);

      vector<int> ReasonsNot;
      ReasonsNot.push_back(buchii);
      ReasonsNot.push_back(NoPred);

      if (index < (stop_unwind-1)) {
	set<int> next = Buchi::GetSuccs(*i);
	vector<int> NoSucc;
	for (set<int>::const_iterator j = next.begin(); j!=next.end(); ++j) {
	  int nbuchii = GetBuchiLtsVar(*j,index+1);
	  NoSucc.push_back(-nbuchii);
	}
	
	size_t NoSuccessor = SATVars++;
	AddEquivalence(NoSucc,NoSuccessor);
	  
	ReasonsNot.push_back(NoSuccessor);

	vector<int> NotClause;
	NotClause.push_back(-buchii);
	NotClause.push_back(-NoSuccessor);
	AddClause(NotClause);
	  
	Util::Message(3,"No successor for %d = %d\n",*i,NoSuccessor);
      }

      for (map<const ImplState*,size_t>::const_iterator j =
	     ImplMapping[index].begin(); j!=ImplMapping[index].end(); ++j) {
	size_t isi = (j->second);
	const ImplState* is = (j->first);
	if (!PropCompatible(is,*i)) {
	  vector<int> NotClause;
	  NotClause.push_back(-isi);
	  NotClause.push_back(-buchii);
	  AddClause(NotClause);
	  ReasonsNot.push_back(isi);
	}
      }

      for (map<Action,size_t>::const_iterator j = 
	     ActionMapping[index].begin(); 
	   j != ActionMapping[index].end(); ++j) {
	size_t acti = (j->second);
	Action act = (j->first);
	if (!LtlChecker::ActCompatible(act,*i)) {
	  Util::Message (3,"Action %s is NOT compatible with Buchi %i(%i) at %i\n",
			 act.ToString().c_str(), *i, buchii, index);
	  vector<int> NotClause;
	  NotClause.push_back(-acti);
	  NotClause.push_back(-buchii);
	  AddClause(NotClause);
	  ReasonsNot.push_back(acti);
	} else {
	  Util::Message (3,"Action %s IS compatible with Buchi %i(%i) at %i\n",
			 act.ToString().c_str(), *i, buchii, index);
	}
      }

      AddClause(ReasonsNot);
    }
    CurrBuchi = NextBuchi;
  }

  vector <int> SomeAccepting;

  for (map<const ImplState*,set<size_t> >::const_iterator i =
	 PossibleLoops.begin(); i != PossibleLoops.end(); ++i) {
    if ((i->second).size () > 1) {
      const ImplState* is = i->first;
      set<size_t> indices = i->second;
      for (set<size_t>::const_iterator j = indices.begin();
	   j != indices.end (); ++j) {
	size_t isij = ImplMapping[*j][is];
	for (set<size_t>::const_iterator k = indices.begin();
	     k != indices.lower_bound(*j); ++k) {
	  Util::Message(3,"Possible loop from %i to %i.\n",*k,*j);
	  size_t isik = ImplMapping[*k][is];

	  size_t IsLoop = SATVars++;
	  
	  RevLoopMapping[IsLoop] = pair<size_t,size_t>(*k,*j);
	  Util::Message(3,"Is a loop = %d\n",IsLoop);
	  vector<int> BothLoopEnds;
	  BothLoopEnds.push_back(isij);
	  BothLoopEnds.push_back(isik);
	  
	  for (map<int,size_t>::const_iterator kmap = 
		 BuchiLtsMapping[*k].begin(); kmap!=BuchiLtsMapping[*k].end();
	       ++kmap) {
	    map<int,size_t>::const_iterator jfound = 
	      BuchiLtsMapping[*j].find(kmap->first);
	    if (jfound == BuchiLtsMapping[*j].end()) {
	      BothLoopEnds.push_back(-(kmap->second));
	    } else {
	      size_t BothInOrNot = SATVars++;
	      vector<int> BothInOrNotClause;
	      BothInOrNotClause.push_back(-BothInOrNot);
	      size_t BothIn = SATVars++;
	      vector<int> BothInClause;
	      BothInClause.push_back(kmap->second);
	      BothInClause.push_back(jfound->second);
	      AddEquivalence(BothInClause,BothIn);
	      vector<int> BothInImp;
	      BothInImp.push_back(-BothIn);
	      BothInImp.push_back(BothInOrNot);
	      AddClause(BothInImp);
	      BothInOrNotClause.push_back(BothIn);
	      size_t BothNot = SATVars++;
	      vector<int> BothNotClause;
	      BothNotClause.push_back(-(kmap->second));
	      BothNotClause.push_back(-(jfound->second));
	      AddEquivalence(BothNotClause,BothNot);
	      vector<int> BothNotImp;
	      BothNotImp.push_back(-BothNot);
	      BothNotImp.push_back(BothInOrNot);
	      AddClause(BothNotImp);
	      BothInOrNotClause.push_back(BothNot);
	      AddClause(BothInOrNotClause);
	      
	      BothLoopEnds.push_back(BothInOrNot);
	    }
	  }

	  for (map<int,size_t>::const_iterator jmap = 
		 BuchiLtsMapping[*j].begin(); jmap!=BuchiLtsMapping[*j].end();
	       ++jmap) {
	    map<int,size_t>::const_iterator kfound = 
	      BuchiLtsMapping[*k].find(jmap->first);
	    if (kfound == BuchiLtsMapping[*k].end()) {
	      BothLoopEnds.push_back(-(jmap->second));
	    }
	  }

	  AddEquivalence(BothLoopEnds,IsLoop);
	    
	  size_t SomeAccepts = 0;
	  vector<int> ForceAtLeastOne;

	  if (Database::LTL_NEG) {
	    SomeAccepts = SATVars++;
	    Util::Message(3,"Some accepting = %d\n",SomeAccepts);

	    ForceAtLeastOne.push_back(-SomeAccepts);
	  }
	    
	  // Rule out anything accepting at stop_unwind:
	  // Cannot know future actions, etc.
	  for (size_t l = (*k); (l <= (*j)) && (l < (stop_unwind-1)); ++l) {
	    for (map<int,size_t>::const_iterator m = 
		   BuchiLtsMapping[l].begin();
		 m != BuchiLtsMapping[l].end(); ++m) {
	      if (Buchi::IsAccept(m->first)) {
		Util::Message(3,"Could accept if %d\n", m->second);
		if (Database::LTL_NEG) {
		  ForceAtLeastOne.push_back(m->second);
		  vector<int> OrAccepts;
		  OrAccepts.push_back(-(m->second));
		  OrAccepts.push_back(SomeAccepts);
		  AddClause(OrAccepts);
		} else {
		  vector<int> RuleOut;
		  RuleOut.push_back(-IsLoop);
		  RuleOut.push_back(-(m->second));
		  AddClause(RuleOut);
		}
	      }
	    }
	  }

	  AddClause(ForceAtLeastOne);
	    
	  if (Database::LTL_NEG) {
	    vector<int> ThisAccepts;
	    ThisAccepts.push_back(IsLoop);
	    ThisAccepts.push_back(SomeAccepts);
	      
	    size_t AcceptingLoop = SATVars++;
	    AcceptingMapping[AcceptingLoop] = pair<size_t,size_t>(*k,*j);
	    AddEquivalence(ThisAccepts,AcceptingLoop);
	      
	    Util::Message(3,"Is an accepting loop = %d\n",AcceptingLoop);
	      
	    SomeAccepting.push_back(AcceptingLoop);
	  }
	}
      }
    }
  }

  AddClause(SomeAccepting);
  
}

bool Explain::GenerateAndSolvePBS() {
  // Generate SAT and solve with PBS.
  
  string command;
  
  Util::Message(1,"Writing dimacs file...\n");
  //create the dimacs file. first write the number of variables and
  //clauses. then append the clause file and remove the clause file.
  FILE *pbscnf = fopen("explain.cnf","w");
  size_t varNo = SATVars;
  fprintf(pbscnf,"p cnf %d %d\n",varNo,clauseNum);
  fclose(pbscnf);
  fclose(clauseFile);
  system("cat explain.clauses >> explain.cnf");
  clauseFile = fopen("explain.clauses","a");
  
  //create the pb file
  FILE *pbspb = fopen("explain.cnf.pb","w");
  fprintf(pbspb,"# PBType: SE\n");
  fprintf(pbspb,"# PBGoal: %d\n",maxDelta+1);
  fprintf(pbspb,"# PBObj : MIN\n");
  fprintf(pbspb,"# NumCoef: %d\n",deltas.size());
  for (vector<pair<int, int> >::const_iterator i = deltas.begin();
       i != deltas.end(); ++i) {
    fprintf(pbspb,"v%d c%d\n", (*i).first, (*i).second);
  }
  fclose(pbspb);

  //check if environment variable MAGICROOT is defined
  char *envptr = getenv("MAGICROOT");
  if(envptr == NULL) Util::Error("MAGICROOT not set ... Aborting ...\n");
  //removed any old copies of the output file
  remove("results.tmp");
  //run PBS
#ifdef WIN32
  command = string(envptr) + 
    "\\pbs\\PBS.exe -f explain.cnf -D 1 -I -a -t 90000 > results.tmp";
#else
  command = string(envptr) + 
    "/pbs/PBS -f explain.cnf -D 1 -I -a -t 90000 > results.tmp";
#endif //WIN32
  system(command.c_str());
  //read results and get variable assignment
  FILE* results = fopen("results.tmp","r");
  char line[100000];
  bool sat = true;
  bool read_assign = false;
  while (!feof(results) && sat) {
    fgets(line, 100000, results);
    Util::Message(2,line);
    if (strstr(line,"Variable Assignments Satisfying CNF Formula:") != NULL) {
      read_assign = true;
      Util::Message(2,"Reading variable assignments.\n");
      assigned.clear ();
      for (size_t i = 0; i < varNo; ++i) {
        int v;
        fscanf (results,"%i",&v);
        if (v > 0) {
          assigned.insert(v);
        }
      }
    }
    else if (strcmp(line,"Result is     : UNS\n") == 0) {
      Util::Message(2, "UNSATISFIABLE\n");
      sat = false;
    }
    else if (strcmp(line,"Result is     : SAT\n") == 0) {
      Util::Message(2, "Finished (SAT)\n");
      sat = false;
    }
  }
  fclose(results);

  Util::Message(1,"Finished reading results.\n");

  return read_assign;
}

bool Explain::DisplayPBSResultsSpurious(ProcManager &pm) {
  vector<int> BlockingClause;

  SuccessImpls.clear();
  SuccessActions.clear();
  SuccessAlignments.clear();
  SuccessBuchi.clear();

  vector<set<size_t> > PredValChanges;
  set<size_t> ActionChanges;
  set<size_t> DeleteAligned;
  set<size_t> InsertAligned;

  PredValChanges.clear(); PredValChanges.resize(CESize);

  size_t LastImplState = 0;
  size_t LastAction = 0;
  
  for (set<size_t>::const_iterator i = assigned.begin();
       i != assigned.end(); ++i) {
    Util::Message(3,"SAT = %i\n",*i);
    map<size_t,pair<size_t, const ImplState*> >::const_iterator fis =
      RevImplMapping.find(*i);
    if (fis != RevImplMapping.end ()) {
      if (fis->second.first > LastImplState)
	LastImplState = fis->second.first;
      SuccessImpls[fis->second.first] = fis->second.second;
      BlockingClause.push_back(-(*i));
      Util::Message(3,"Saw ImplState for %i (%s)\n",fis->second.first);
    } else {
      map<size_t,pair<size_t,Action> >::const_iterator fact = 
	RevActionMapping.find(*i);
      if (fact != RevActionMapping.end()) {
	if (fact->second.first > LastAction)
	  LastAction = fact->second.first;
	SuccessActions[fact->second.first] = fact->second.second;
	BlockingClause.push_back(-(*i));
	Util::Message(3,"Saw Action for %i (%s)\n",fact->second.first,
		      fact->second.second.ToString().c_str());
      } else {
	map<size_t,pair<size_t,size_t> >::const_iterator fpvd =
	  PredValDeltas.find(*i);
	if (fpvd != PredValDeltas.end()) {
	  Util::Message(3,"Saw predicate delta for %i, %i\n",
			fpvd->second.first, fpvd->second.second);
	  PredValChanges[fpvd->second.first].insert(fpvd->second.second);
	} else {
	  map<size_t,size_t>::const_iterator factd =
	    ActionDeltas.find(*i);
	  if (factd != ActionDeltas.end()) {
	    ActionChanges.insert(factd->second);
	  } else {
	    map<size_t,size_t>::const_iterator fcld =
	      ContLocDeletions.find(*i);
	    if (fcld != ContLocDeletions.end()) {
	      Util::Message(3,"Deleted Alignment: %i\n",fcld->second);
	      DeleteAligned.insert(fcld->second);
	    } else {
	      map<size_t,size_t>::const_iterator fcli =
		ContLocInsertions.find(*i);
	      if (fcli != ContLocInsertions.end()) {
		Util::Message(3,"Inserted Alignment: %i\n",fcli->second);
		InsertAligned.insert(fcli->second);
	      } else {
		map<size_t,pair<size_t,size_t> >::const_iterator fra =
		  RevAlignmentMapping.find(*i);
		if (fra != RevAlignmentMapping.end()) {
		  SuccessAlignments[fra->second.second] = fra->second.first;
		  Util::Message(3,"ALIGNING: %d and %d\n",fra->second.first,
				fra->second.second);
		} else {
		  map<size_t,pair<size_t,int> >::const_iterator flts =
		    RevBuchiLtsMapping.find(*i);
		  if (flts != RevBuchiLtsMapping.end()) {
		    Util::Message(3,"Saw Buchi for %i (%i)\n", 
				  flts->second.first, flts->second.second);
		    map<size_t,set<int> >::iterator fsb =
		      SuccessBuchi.find(flts->second.first);
		    if (fsb == SuccessBuchi.end ()) {
		      set<int> NewSet;
		      NewSet.insert(flts->second.second);
		      SuccessBuchi[flts->second.first] = NewSet;
		    } else {
		      SuccessBuchi[flts->second.first].insert
			(flts->second.second);
		    }
		  } else {
		    map<size_t,pair<size_t,size_t> >::const_iterator floop =
		      RevLoopMapping.find(*i);
		    if (floop != RevLoopMapping.end()) {
		      Util::Message(3,"LOOP FROM %i to %i\n",
				    floop->second.first,floop->second.second);
		    } else {
		      map<size_t,pair<size_t,size_t> >::const_iterator facc =
			AcceptingMapping.find(*i);
		      if (facc != AcceptingMapping.end()) {
			Util::Message(3,"*ACC LOOP FROM %i to %i*\n",
				      facc->second.first,facc->second.second);
		      }
		    }
		  }
		}
	      }
	    }
	  }
	}
      }
    }
  }

  SuccessLength = LastImplState;
  //if (LastAction < SuccessLength)
  //  SuccessLength = LastAction;

  ConcCEDag success;
  for (size_t i = 0; i <= (SuccessLength); ++i) {
    const ImplState* is = SuccessImpls[i];
    CL1AbsState cl1 = CL1AbsState(is);
    success.stateLabel.push_back(cl1.Clone());
    success.actLabel.push_back(SuccessActions[i]);
    set<int> ts;
    if (i != (SuccessLength))
      ts.insert(i + 1);
    success.trans.push_back(ts);
  }

  success.Display(component);

  for (size_t i = 0; i <= SuccessLength; ++i) {
    Util::Message(3, "Buchi at %i : ", i);
    for (set<int>::const_iterator j = SuccessBuchi[i].begin();
	 j != SuccessBuchi[i].end(); j++) {
      if (j != SuccessBuchi[i].begin()) {
	Util::Message(3,", ");
      }
      Util::Message(3, "%i",*j);
    }
    Util::Message(3,"\n");
  }

  Util::Message(1,"\nError explanation deltas:\n\n");
  
  bool anyPrinted;
  size_t CEPos = 0;
  
  for(size_t i = 0; i < stop_unwind; ++i) {
    anyPrinted = false;

    map<size_t,size_t>::const_iterator af = SuccessAlignments.find(i);
    if (af != SuccessAlignments.end()) {
      Util::Message(3,"DELTAS: %i aligned with %i\n",i,af->second);
      for (size_t j = CEPos; j < (af->second); ++j) {
	bool anyPrintedCE = false;
	if (DeleteAligned.count(j) != 0) {
	  if (Unalignable.count(j) == 0) {
	    anyPrintedCE = true;
	    Util::Message(1,"Control location deleted (step #%i.%i):\n",i,j);
	    Util::Message(1,"  " + CEContLocs[j]->ToString() + "\n");
	    if (j < (CESize-1)) {
	      Util::Message(1,"  " + (CEActions[j].ToString()) + "\n");
	    }
	  }
	}
	if (anyPrintedCE) {
	  Util::Message(1,"------------------------\n");
	}
      }
	
      CEPos = (af->second);
      Util::Message (3,"CEPos = %i\n",CEPos);

      if (!(PredValChanges[CEPos].empty())) {
	assert (CEContLocs[CEPos] == SuccessImpls[i]->GetContLoc());
	PredSet PS = CEContLocs[CEPos]->GetPreds();
	PredVal PV = PS.PValToPredVal(CEPVals[CEPos]);
	PredVal PV2 = PS.PValToPredVal(SuccessImpls[i]->GetPredVal());
	Util::Message(3,"PredChange (lengths %i, %i)\n",PV.second.size(),
		      PV2.second.size());
	const vector <Predicate> Preds = PS.GetPreds ();
	for (set<size_t>::const_iterator j = PredValChanges[CEPos].begin ();
	     j != PredValChanges[CEPos].end(); ++j) {
	  Util::Message(3,"j = %i, vals = %i, %i\n",*j,(PV.second)[*j],
			(PV2.second)[*j]);
	  anyPrinted = true;
	  set<Expr> trueExprsOld;
	  set<Expr> trueExprsNew;
	  Preds[*j].ToExpr((PV.second)[*j],trueExprsOld);
	  Preds[*j].ToExpr((PV2.second)[*j],trueExprsNew);
	  Util::Message(1,"Predicate changed (step #%i):\n",i);
	  Util::Message(1,"  was:  ");
	  for (set<Expr>::const_iterator k = trueExprsOld.begin(); 
	       k!= trueExprsOld.end(); ++k) {
	    if (k != trueExprsOld.begin()) {
	      Util::Message(1,", ");
	    }
	    Util::Message(1,(*k).ToString());
	  }
	  Util::Message(1,"\n  now:  ");
	  for (set<Expr>::const_iterator k = trueExprsNew.begin(); 
	       k!= trueExprsNew.end(); ++k) {
	    if (k != trueExprsNew.begin()) {
	      Util::Message(1,", ");
	    }
	    Util::Message(1,(*k).ToString());
	  }
	  Util::Message(1,"\n");
	}
      }

      if (ActionChanges.count(CEPos) != 0) {
	anyPrinted = true;
	Util::Message(1,"Action changed (step #%i):\n",i);
	Util::Message(1,"  was:");
	Util::Message(1,"  " + CEActions[CEPos].ToString() + "\n");
	if (i <= LastAction) {
	  Util::Message(1,"  now:");
	  Util::Message(1,"  " + SuccessActions[i].ToString() + "\n");
	}
      }

    }

    if ((i <= SuccessLength) && (InsertAligned.count(i) != 0)) {
      anyPrinted = true;
      Util::Message(1,"Control location inserted (step #%i):\n",i);
      Util::Message(1,"  " + 
		    (SuccessImpls[i]->GetContLoc())->ToString() + "\n");
      if (i <= LastAction) {
	Util::Message(1,"  " + (SuccessActions[i].ToString()) + "\n");
      }
    }

    if (anyPrinted) {
      Util::Message(1,"------------------------\n");
    }

  }

  for (size_t i = CEPos+1; i < CESize; ++i) {
    if (DeleteAligned.count(i) != 0) {
      if (Unalignable.count(i) == 0) {
	anyPrinted = true;
	Util::Message(1,"Control location deleted (step #%i.%i):\n",
		      SuccessLength+1,i);
	Util::Message(1,"  " + CEContLocs[i]->ToString() + "\n");
	if (i < (CESize-1)) {
	  Util::Message(1,"  " + (CEActions[i].ToString()) + "\n");
	}
	Util::Message(1,"------------------------\n");
      }
    }
  }

  assert(Database::IMPL_TYPE == Database::IMPL_C);
  //the set of branch locations that can explain this path's
  //infeasibility
  set<ContLoc*> expLocs;
  
  //compute the weakest precondition at each node of the graph in
  //reverse topological order
  vector< set<Expr> > dagToWP(success.stateLabel.size());
  vector<int> topo; success.ComputeTopo(topo);
  for(vector<int>::const_iterator i = topo.begin();i != topo.end();++i) {
    for(set<int>::const_iterator j = success.trans[*i].begin();
	j != success.trans[*i].end();++j) {
      dagToWP[*i].insert(dagToWP[*j].begin(),dagToWP[*j].end());
    }
    map< int,pair<Action,Expr> > inlinedAssignMap;
    set<ContLoc*> dummySeeds;
    CEDagVerifier CED(component,false);
    CED.UpdatePrePost(dummySeeds,expLocs,dagToWP[*i],*i,success,
		      inlinedAssignMap,true,0);
  }
  
  //add the constraint for the initial state
  dagToWP[0].insert(pm.GetGuard());

  bool spurious = (ExprManager::ProveImplies(dagToWP[0],set<Expr>()));

  if (spurious) {
    AddClause(BlockingClause);
  }

  return spurious;
}



/*********************************************************************/
//end of Explain.cpp
/*********************************************************************/
