/**************************** CPPHeaderFile ***************************

* FileName [Database.h]

* PackageName [main]

* Synopsis [Header file containing all static constants, arguments and
* options to the tool as well as data structures.]

* Description []

* SeeAlso [arc.cpp]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University.  All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#ifndef __DATABASE_H__
#define __DATABASE_H__

//needed to interface with the parser
extern FILE *Stdcin;
int Stdcparse();

namespace magic {

//other classes needed
class StdcFile;
class BasicExpr;
class Expr;
class FspBody;
class FspTransSeq;
class Action;
class ProcAddInfo;
class ProgInfo;
class LtsTrans;
class FspInfo;
class LtlFormula;
class ProcInfo;
class LtsInfo;
class CallGraph;
class ContLoc;
class Component;
class PredSet;

class Database
{
 private:
  static void DisplayUsage();

 public:

  /*******************************************************************/
  //constants
  /*******************************************************************/
  //the current version of magic
  static const string MAGIC_VERSION;

  //parser related constants for the pre and post increment and
  //decrement operators
  static const short int PRE_INC;
  static const short int PRE_DEC;
  static const short int POST_INC;
  static const short int POST_DEC;

  //the epsilon action
  static const string EPSILON;
  
  //the epsilon action specification. just does an epsilon action and
  //stops.
  static const string EPSILON_SPEC;

  //the invalid pointer dereference action
  static const string INVALID_PTR_DEREF;

  //the invalid pointer dereference action spec. does an invalid
  //pointer dereference action and then goes to the error state
  static const string INVALID_PTR_DEREF_SPEC;

  //the predicate for valid pointers
  static const string VALID_PTR_PRED;

  //the assertion failure action
  static const string ASSERTION_FAILURE;

  //the assertion failure action spec. does an assertion failure
  //action and then goes to the error state
  static const string ASSERTION_FAILURE_SPEC;

  //the name of an epsilon procedure abstraction
  static const string EPSILON_PROC_ABS;
  
  //the default specification LTS abstracting call-sites. this LTS
  //cannot appear in any user-supplied specification. it has just a
  //single transition with the "default-return" action. hence any
  //call-site which is abstracted by this LTS essentially becomes
  //non-simulatable.
  static const string DEFAULT_SPEC;
  
  //the STOP lts state
  static const string STOP_STATE;

  //the ERROR lts state
  static const string ERROR_STATE;

  //a dummy C procedure used as a placeholder when nothing is known
  //about an indirect function call site
  static const string DUMMY_C_PROC;

  //some special functions with special semantics
  static const string EXIT_C_PROC;
  static const string MALLOC_C_PROC;
  static const string FREE_C_PROC;
  static const string ASSERT_C_PROC;

  //various formats in which implementations can be described. the
  //default is a concurrent c program. the only other type supported
  //currently is a PACC process.
  static const int IMPL_C;
  static const int IMPL_PACC;
  
  //various types of conformance checks. currently we support weak
  //simulation, error state reachability, trace containment, LTL model
  //checking, deadlock detection and SE-AW model checking
  static const int CONF_SIMUL;
  static const int CONF_REACH;
  static const int CONF_TRACE;
  static const int CONF_LTL;
  static const int CONF_DLOCK;
  static const int CONF_SEAW;
  static const int CONF_ANGLUIN;

  //mechanisms for carrying out the conformance check
  static const int CONF_CHECK_EXPLICIT;
  static const int CONF_CHECK_SAT;
  static const int CONF_CHECK_LTSA;

  //various C parsers that can be used
  static const int C_PARSER_DEFAULT;
  static const int C_PARSER_CTOOL;
  
  //various theorem provers that can be used
  static const int THEOREM_PROVER_SIMPLIFY;
  static const int THEOREM_PROVER_CPROVER;
  static const int THEOREM_PROVER_CVC;
  static const int THEOREM_PROVER_ICS;
  static const int THEOREM_PROVER_SVC;
  static const int THEOREM_PROVER_CVCL;

  //various SAT solvers that can be used
  static const int SAT_SOLVER_CHAFF;
  static const int SAT_SOLVER_SATO;
  static const int SAT_SOLVER_GRASP;
  static const int SAT_SOLVER_HORN_AI;

  //types of checks for the infeasibility of a path
  static const int PATH_CHECK_FORWARD;
  static const int PATH_CHECK_BACKWARD;

  //formats in which counterexamples are displayed
  static const int CE_LOC_ONLY;
  static const int CE_LOC_ACT;
  static const int CE_LOC_CONS;
  static const int CE_LOC_CONS_ACT;

  //various pattern matchers than can be used
  static const int PATTERN_MATCHER_NAIVE;

  //various types of CEGAR strategies - currently we support adding a
  //certain number of predicates that can eliminate the new
  //counterexamples every time, adding a certain number of useful
  //predicates every time and predicate optimization
  static const int CEGAR_ELIMINATE_PRED;
  static const int CEGAR_USEFUL_PRED;
  static const int CEGAR_OPTIMIZE_PRED;
  static const int CEGAR_GREEDY_PRED;

  //name of the temporary dot file
  static const string DOT_FILE_NAME;
  
  //parameters for predicate evaluation
  //static const int COVER_NUMBER_TRACES;
  //static const int NUMBER_TRACES;
  //static const int TRACES_LENGTH;
  //static const Random rnd;

  //types of output model formats
  static const int OUTPUT_MODEL_FSP;

  //types of abstraction refinement
  static const int REFINE_NONE;
  static const int REFINE_PRED;
  static const int REFINE_LTS;
  
  /*******************************************************************/
  //arguments and options
  /*******************************************************************/
  //the list of files to be parsed
  static list<string> FILE_NAMES;

  //the name of the abstraction to be checked
  static string ABSTRACTION_NAME;
  
  //the output file for storing the computed simulation relation
  static string SIM_REL_FILE;
  
  //if we just need to parse the input files
  static bool PARSE_ONLY;

  //if the files should be printed while parsing
  static bool PARSE_ECHO;

  //if statistics should be displayed at the end
  static bool DISPLAY_STATISTICS;
  
  //the format in which the implementation is described
  static int IMPL_TYPE;

  //the selected type of conformance check
  static int CONF_TYPE;

  //the selected mechanism for doing the conformance check
  static int CONF_CHECK_MECHANISM;

  //the selected C parser
  static int C_PARSER_TYPE;
  
  //type of theorem prover to be used
  static int THEOREM_PROVER_USED;

  //type of SAT solver to be used
  static int SAT_SOLVER_USED;

  //type of pattern matcher used
  static int PATTERN_MATCHER_USED;

  //whether to ignore pointers
  static bool IGNORE_POINTERS;

  //whether syntactic checks should be used before calling the theorem
  //prover
  static bool USE_SYNTACTIC_CHECKS;
  
  //whether results of theorem prover calls should be cached
  static bool THEOREM_PROVER_CACHE;

  //the maximum theorem prover cache size
  static int THEOREM_PROVER_CACHE_SIZE;  

  //whether to use PAM for alias analysis
  static bool USE_PAM;
  
  //the flag which indicates whether the predicates supplied by the
  //user are to be used or not
  static bool PREDS_FROM_SPEC;
  
  //whether the initial seeds come from the specification
  static bool INIT_SEEDS_FROM_SPEC;

  //the flag which indicates whether new predicates should be
  //generated from counterexamples
  static bool PREDS_FROM_CE;

  //the verbosity level of the tool
  static int VERBOSITY_LEVEL;

  //whether to handle data communication between processes - either
  //through global variables or by passing data on channels
  static bool HANDLE_DATA_COMM;

  //whether to use live variables
  static bool USE_LIVE_VARS;

  //whether to inline all routines whose code is available
  static bool INLINE_LIBS;

  //whether to check for any NULL pointer dereference
  static bool CHECK_INVALID_PTR_DEREF;

  //whether to check for assertion failures
  static bool CHECK_ASSERTION_FAILURE;

  //the starting and ending dereference or assertion numbers
  static int START_ITER;
  static int END_ITER;

  //whether to use only reachable implementation states
  static bool USE_REACH_IMPL;

  //whether the transition relation of the implementation LTS should
  //be precomputed and stored
  static bool PRECOMPUTE_IMPL_TRANS;

  //if the implementation state transitions should be cached
  static bool CACHE_IMPL_TRANS;

  //whether a simulation relation should be created even when using
  //SAT
  static bool SIM_REL_FROM_SAT;
  
  //if pre-optimization phase on just CGA should run
  static bool PREOPTIMIZE;

  //if pre-optimization phase should limit visits to states.
  static bool PREOPTVISIT;

  //limit for number of visits to a state in pre-optimization
  static int PREOPTVISITS;

  //minimum depth to which to perform pre-optimization searches
  static int PREOPTMINDEPTH;

  //minimum number of counterexamples
  static size_t PREOPTMINCES;

  //maximum depth to which to perform pre-optimization searches
  static int PREOPTMAXDEPTH;

  //maximum number of counterexamples
  static size_t PREOPTMAXCES;

  //if traces should be generated based on coverage
  static bool COVERAGE;
  
  //if traces should be generated based on control location vs. impl
  static bool COVERCONT;
  
  //if coverage should be shown (works only for control locations)
  static bool SHOWCOVER;

  //whether to do counter-example guided abstraction refinement
  static bool CEGAR;

  //the type of CEGAR we do
  static int CEGAR_TYPE;

  //whether to do incremental verification
  static bool INC_VERIFY;

  //the maximum number of predicates added in every iteration. this
  //applied only to adding eliminate or useful predicate.
  static int MAX_ADD_PRED;

  //whether we should check the validity of branching in CEs
  static bool CHECK_CE_BRANCHING;

  //what kind of check should be done to check for the feasibility of
  //a path. currently two types are supported - forward and backward.
  static int PATH_CHECK_TYPE;

  //format in which counterexamples are displayed
  static int CE_DISPLAY_TYPE;

  //if we are not doing CEGAR, whether we should check the validity of
  //the generated counterexample
  static bool CHECK_CE_VALIDITY;

  //whether to use a pattern matcher during cegar
  static bool USE_PATTERN_MATCHER;

  //the maximum number of global CE dags to be constructed
  static size_t MAX_GLOBAL_CE;

  //the maximum number of backward loops allowed on any path in the
  //global CE dag
  static size_t MAX_CE_LOOP;

  //the probability with which a new node node is created even if
  //there already exists another node with same state labeling
  static double CE_NEW_NODE_PROB;

  //if clauses should be built so that no new variables are introduced
  static bool CNF_NO_NEW_VARS;

  //how large to generate subsets
  static size_t LARGEST_SUBSET;

  //the smallest size of subsets that will be at least tried
  static size_t SMALLEST_SUBSET;

  //cutoff for generating subsets
  static size_t SUBSET_CUTOFF;

  //cutoff for generating conjunctive clauses per spurios CE
  static size_t CONSTRAINT_CUTOFF;

  //if sets of predicates of a larger size should be generated, even if smaller
  //size was sufficient to eliminate.
  static bool STOP_OPT_AFTER_ELIM;

  //if predicate abstraction should be used
  static bool USE_PRED_ABSTRACTION;

  //if LTS abstraction should be used
  static bool USE_LTS_ABSTRACTION;

  //if useless control locations should be removed
  static bool NO_USELESS_LOCS;

  //if silent transitions should be removed
  static bool NO_SILENT_TRANS;

  //the maximum number of times a location is allowed to repeat in a
  //predicate inference chain
  static int MAX_PRED_INFER_LOOP;

  //maximum amount of seconds tool should run
  static int MAX_TIME_LIMIT;

  //whether to translate input ltl formulae to buchi automata
  static bool LTL_TO_BUCHI;

  //whether to create any output dot file
  static bool CREATE_DOT_FILE;

  //whether to draw any spec LTSs
  static bool DRAW_LTS;

  //whether to draw the control flow graph
  static bool DRAW_CFG;

  //whether to draw the reduced CFG with inferred predicates
  static bool DRAW_INFERRED_PREDS;

  //whether to draw the LTS obtained after predicate abstraction
  static bool DRAW_PRED_ABS_LTS;

  //whether to draw the LTS obtained after LTS abstraction
  static bool DRAW_LTS_ABS_LTS;

  //whether to draw the LTS obtained after parallel composition
  static bool DRAW_COMPOSED_LTS;

  //draw the CE-DAG obtained
  static bool DRAW_CE_DAG;

  //draw projections of the CE-DAG on LTS abstraction components
  static bool DRAW_CE_PROJ_LTS;

  //draw projections of the CE-DAG on predicate abstraction components
  static bool DRAW_CE_PROJ_PRED;
  
  //whether to output models
  static bool OUTPUT_MODELS;

  //what kind of models are to be output
  static int OUTPUT_MODEL_TYPE;  

  //whether to explain real counterexamples
  static bool EXPLAIN_CE;

  //how much more to unwind in explanation
  static int ADDITIONAL_UNWIND;

  //maximum additional amount to unwind in explanation
  static size_t MAX_UNWIND;

  //whether to try to unwind more on getting UNS
  static bool MORE_UNWIND;

  //number of times to try to produce non-spurious successful run
  static size_t SPURIOUS_TRIES;

  //weight for action changes
  static size_t ACTION_WEIGHT;

  //weight for predicate changes
  static size_t PRED_WEIGHT;

  //LTL:  counterexample to negation of formula?
  static bool LTL_NEG;

  //whether to backup the state for later restoring
  static bool BACKUP;

  //whether to restore backed up state
  static bool RESTORE;

  //name of the file to which to print the counterexample
  static string CE_FILE;
  
  /*******************************************************************/
  //data structures
  /*******************************************************************/
  //the command line arguments passed to the tool
  static list<string> commandLine;
  //the canonical names of C files mapped to their asts
  static map<string,StdcFile*> cAsts;
  //the set of spec files already parsed
  static set<string> specFiles;
  //the map of procedure names to additional info specified about them
  static map<string,ProcAddInfo> procAddInfos;
  //the map of program names to info specified about them
  static map<string,ProgInfo> progInfos;
  //the fsp info specified in spec files
  static FspInfo fspInfo;
  //the map of enumeration constants to their values
  static map<string,const BasicExpr*> enums;
  //the names of C procedures mapped to info about them
  static map<string,ProcInfo> procInfos;  
  //the specification LTS name
  static string specLtsName;
  //the name of program to be verified
  static string progName;
  //the root implementation procedure names
  static vector<string> rootProcNames;
  //the vector of LTSs infos
  static vector<LtsInfo> ltsInfos;
  //the mapping from lts infos to their negations
  static map<string,string> ltsNegMap;
  //the components of the implementation
  static vector<Component*> components;
  //the type of abstraction refinement last performed
  static int lastRefType;
  //id of the component whose abstraction was last refined
  static int lastRefComp;
  //the priorities of the expressions
  static map<Expr,int> exprPriority;
  //the name of the output file containing the drawn stuff
  static string drawFileName;
  //FILE pointer for the dot output file
  static FILE *dotFilePtr;
  //name of the file containing output FSP models
  static string modelFileName;
  //FILE pointer for the model output file
  static FILE *modelFilePtr;
  //name of backup file
  static string backupFileName;
  //name of restore file
  static string restoreFileName;

  /*******************************************************************/
  //methods
  /*******************************************************************/
  static void ParseCommandLine();
  static void ParseFiles();
  static bool IsFileParsed(const string &name);
  static void AddFileAst(const string &name,StdcFile *ast);
  static void ExtractProcInfos();
  static const LtsInfo &GetLtsInfo(const string &name);
  static const LtsInfo &GetSpecLtsNeg(const string &name);
  static LtlFormula GetLtlFormula(const string &name);
  static const set<Action> &GetSpecActions();
  static void GetSpecAssignActions(const Expr &lhs,const Expr &rhs,
				   const list<string> &params,
				   const set<Expr> &context,set<Action> &res);
  static void GetSpecReturnActions(const Expr &expr,const list<string> &params,
				   const set<Expr> &context,set<Action> &res);  
  static void GetSpecAssignActExprs(set< pair<Expr,Expr> > &res);
  static void GetSpecRetActExprs(set<Expr> &res);
  static int GetSpecStateNum();
};

} //namespace magic

#endif //__DATABASE_H__

/*********************************************************************/
//end of Database.h
/*********************************************************************/
