/******************************** CPPFile *****************************

* FileName [Database.cpp]

* PackageName [main]

* Synopsis [This file contains procedures to access and manipulate
* global data structures.]

* SeeAlso [Database.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <gmp.h>
#include <string>
#include <list>
#include <map>
#include <set>
#include <vector>
using namespace std;

#include "BigInt.h"
#include "Util.h"
#include "Statistics.h"
#include "CSymbolTable.h"
#include "Node.h"
#include "DFSAdapter.h"
#include "Action.h"
#include "SeawManager.h"
#include "Alias.h"
#include "ProgAbs.h"
#include "ProgInfo.h"
#include "ProcAbs.h"
#include "ProcAddInfo.h"
#include "LtsTrans.h"
#include "FspInfo.h"
#include "LtlFormula.h"
#include "LtlManager.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "ContLoc.h"
#include "ProcInfo.h"
#include "LtsInfo.h"
#include "AbsRefiner.h"
#include "LocalAbsLts.h"
#include "ConcCEDag.h"
#include "Component.h"
#include "Buchi.h"
#ifdef MAGIC_FULL
#include "ctool.h"
#include "CtoolAPI.h"
#include "PamAPI.h"
using namespace ctool;
#endif //MAGIC_FULL
using namespace magic;

/*********************************************************************/
//define static constants
/*********************************************************************/
const string Database::MAGIC_VERSION = "1.0";
const short int Database::PRE_INC = 6000;
const short int Database::PRE_DEC = 6010;
const short int Database::POST_INC = 6020;
const short int Database::POST_DEC = 6030;
const string Database::EPSILON = "epsilon";
const string Database::EPSILON_SPEC = "EpsilonSpec";
const string Database::INVALID_PTR_DEREF = "invalid_ptr_deref";
const string Database::INVALID_PTR_DEREF_SPEC = "InvalidPtrDerefSpec";
const string Database::VALID_PTR_PRED = "$VP";
const string Database::ASSERTION_FAILURE = "assertion_failure";
const string Database::ASSERTION_FAILURE_SPEC = "AssertionFailureSpec";
const string Database::EPSILON_PROC_ABS = "EpsilonProcAbs";
const string Database::DEFAULT_SPEC = "DefaultSpec";
const string Database::STOP_STATE = "STOP";
const string Database::ERROR_STATE = "ERROR";
const string Database::DUMMY_C_PROC = "MagicDummyCProc";
const string Database::EXIT_C_PROC = "exit";
const string Database::MALLOC_C_PROC = "malloc";
const string Database::FREE_C_PROC = "free";
const string Database::ASSERT_C_PROC = "assert";
const int Database::IMPL_C = 2800;
const int Database::IMPL_PACC = 2810;
const int Database::CONF_SIMUL = 2500;
const int Database::CONF_REACH = 2510;
const int Database::CONF_TRACE = 2520;
const int Database::CONF_LTL = 2530;
const int Database::CONF_DLOCK = 2540;
const int Database::CONF_SEAW = 2550;
const int Database::CONF_ANGLUIN = 2560;
const int Database::CONF_CHECK_EXPLICIT = 1000;
const int Database::CONF_CHECK_SAT = 1010;
const int Database::CONF_CHECK_LTSA = 1020;
const int Database::C_PARSER_DEFAULT = 8000;
const int Database::C_PARSER_CTOOL = 8010;
const int Database::THEOREM_PROVER_SIMPLIFY = 5000;
const int Database::THEOREM_PROVER_CPROVER = 5010;
const int Database::THEOREM_PROVER_CVC = 5020;
const int Database::THEOREM_PROVER_ICS = 5030;
const int Database::THEOREM_PROVER_SVC = 5040;
const int Database::THEOREM_PROVER_CVCL = 5050;
const int Database::SAT_SOLVER_CHAFF = 5500;
const int Database::SAT_SOLVER_SATO = 5510;
const int Database::SAT_SOLVER_GRASP = 5520;
const int Database::SAT_SOLVER_HORN_AI = 5530;
const int Database::PATH_CHECK_FORWARD = 5700;
const int Database::PATH_CHECK_BACKWARD = 5710;
const int Database::CE_LOC_ONLY = 5800;
const int Database::CE_LOC_ACT = 5810;
const int Database::CE_LOC_CONS = 5820;
const int Database::CE_LOC_CONS_ACT = 5830;
const int Database::PATTERN_MATCHER_NAIVE = 7000;
const int Database::CEGAR_ELIMINATE_PRED = 7100;
const int Database::CEGAR_USEFUL_PRED = 7110;
const int Database::CEGAR_OPTIMIZE_PRED = 7120;
const int Database::CEGAR_GREEDY_PRED = 7130;
//const int Database::COVER_NUMBER_TRACES = 1000;
//const int Database::NUMBER_TRACES = 1000;
//const int Database::TRACES_LENGTH = 500;
//const Random Database::rnd = new Random ();
const string Database::DOT_FILE_NAME = "magic.dot";
const int Database::OUTPUT_MODEL_FSP = 9500;
const int Database::REFINE_NONE = 9900;
const int Database::REFINE_PRED = 9910;
const int Database::REFINE_LTS = 9920;

/*********************************************************************/
//define static arguments and options
/*********************************************************************/
list<string> Database::FILE_NAMES;
string Database::ABSTRACTION_NAME;
string Database::SIM_REL_FILE = "magic.sim";
bool Database::PARSE_ONLY = false;
bool Database::PARSE_ECHO = false;
bool Database::DISPLAY_STATISTICS = false;
int Database::IMPL_TYPE = IMPL_C;
int Database::CONF_TYPE = CONF_SIMUL;
int Database::CONF_CHECK_MECHANISM = CONF_CHECK_SAT;
int Database::C_PARSER_TYPE = C_PARSER_DEFAULT;
int Database::THEOREM_PROVER_USED = THEOREM_PROVER_SIMPLIFY;
bool Database::USE_PAM = false;
int Database::SAT_SOLVER_USED = SAT_SOLVER_HORN_AI;
int Database::PATTERN_MATCHER_USED = PATTERN_MATCHER_NAIVE;
bool Database::IGNORE_POINTERS = true;
bool Database::USE_SYNTACTIC_CHECKS = true;
bool Database::THEOREM_PROVER_CACHE = false;
int Database::THEOREM_PROVER_CACHE_SIZE = 100000;
bool Database::PREDS_FROM_SPEC = true;
bool Database::INIT_SEEDS_FROM_SPEC = false;
bool Database::PREDS_FROM_CE = false;
int Database::VERBOSITY_LEVEL = 2;
bool Database::HANDLE_DATA_COMM = false;
bool Database::USE_LIVE_VARS = false;
bool Database::INLINE_LIBS = false;
bool Database::CHECK_INVALID_PTR_DEREF = false;
bool Database::CHECK_ASSERTION_FAILURE = false;
int Database::START_ITER = 1;
int Database::END_ITER = 99999999;
bool Database::USE_REACH_IMPL = true;
bool Database::PRECOMPUTE_IMPL_TRANS = true;
bool Database::CACHE_IMPL_TRANS = true;
bool Database::SIM_REL_FROM_SAT = false;
bool Database::PREOPTIMIZE = false;
bool Database::PREOPTVISIT = false;
int Database::PREOPTMINDEPTH = 50;
size_t Database::PREOPTMINCES = 100;
int Database::PREOPTMAXDEPTH = 500;
size_t Database::PREOPTMAXCES = 2000;
int Database::PREOPTVISITS = 5;
bool Database::COVERAGE = false;
bool Database::COVERCONT = false;
bool Database::SHOWCOVER = false;
bool Database::CEGAR = false;
int Database::CEGAR_TYPE = CEGAR_ELIMINATE_PRED;
bool Database::INC_VERIFY = true;
int Database::MAX_ADD_PRED = 1;
bool Database::CHECK_CE_BRANCHING = false;
int Database::PATH_CHECK_TYPE = PATH_CHECK_FORWARD;
int Database::CE_DISPLAY_TYPE = CE_LOC_ONLY;
bool Database::CHECK_CE_VALIDITY = false;
bool Database::USE_PATTERN_MATCHER = false;
size_t Database::MAX_GLOBAL_CE = 1;
size_t Database::MAX_CE_LOOP = 0;
double Database::CE_NEW_NODE_PROB = 0.0;
bool Database::CNF_NO_NEW_VARS = true;
size_t Database::LARGEST_SUBSET = 10;
size_t Database::SMALLEST_SUBSET = 1;
size_t Database::SUBSET_CUTOFF = 50;
size_t Database::CONSTRAINT_CUTOFF = 20;
bool Database::STOP_OPT_AFTER_ELIM = true;
bool Database::USE_PRED_ABSTRACTION = true;
bool Database::USE_LTS_ABSTRACTION = false;
bool Database::NO_USELESS_LOCS = false;
bool Database::NO_SILENT_TRANS = true;
int Database::MAX_PRED_INFER_LOOP = 1;
int Database::MAX_TIME_LIMIT = 36000000;
bool Database::LTL_TO_BUCHI = false;
bool Database::CREATE_DOT_FILE = false;
bool Database::DRAW_LTS = false;
bool Database::DRAW_CFG = false;
bool Database::DRAW_INFERRED_PREDS = false;
bool Database::DRAW_PRED_ABS_LTS = false;
bool Database::DRAW_LTS_ABS_LTS = false;
bool Database::DRAW_COMPOSED_LTS = false;
bool Database::DRAW_CE_DAG = false;
bool Database::DRAW_CE_PROJ_LTS = false;
bool Database::DRAW_CE_PROJ_PRED = false;
bool Database::OUTPUT_MODELS = false;
int Database::OUTPUT_MODEL_TYPE = OUTPUT_MODEL_FSP;
bool Database::EXPLAIN_CE = false;
int Database::ADDITIONAL_UNWIND = 5;
size_t Database::MAX_UNWIND = 50;
bool Database::MORE_UNWIND = true;
size_t Database::SPURIOUS_TRIES = 50;
size_t Database::ACTION_WEIGHT = 1;
size_t Database::PRED_WEIGHT = 1;
bool Database::LTL_NEG = true;
bool Database::BACKUP = false;
bool Database::RESTORE = false;
string Database::CE_FILE = "ce.out";

/*********************************************************************/
//define static data structures
/*********************************************************************/
list<string> Database::commandLine;
map<string,StdcFile*> Database::cAsts;
set<string> Database::specFiles;
map<string,ProcAddInfo> Database::procAddInfos;
map<string,ProgInfo> Database::progInfos;
FspInfo Database::fspInfo;
map<string,const BasicExpr*> Database::enums;
map<string,ProcInfo> Database::procInfos;
string Database::specLtsName;
string Database::progName;
vector<string> Database::rootProcNames;
vector<LtsInfo> Database::ltsInfos;
map<string,string> Database::ltsNegMap;
vector<Component*> Database::components;
int Database::lastRefType = REFINE_NONE;
int Database::lastRefComp = -1;
map<Expr,int> Database::exprPriority;
string Database::drawFileName = "magic.ps";
FILE *Database::dotFilePtr = NULL;
string Database::modelFileName = "magic.fsp";
FILE *Database::modelFilePtr = NULL;
string Database::backupFileName = "magic.backup";
string Database::restoreFileName = "magic.backup";

/*********************************************************************/
//global shutdown method defined in arc.cpp needed
/*********************************************************************/
__attribute__ ((noreturn)) void GlobalShutdown(int sig);

/*********************************************************************/
//parse command line and get user specified values of various options
/*********************************************************************/
void Database::ParseCommandLine()
{
  for(list<string>::const_iterator i = commandLine.begin();
      i != commandLine.end();++i) {
    if(((*i) == "--help") || ((*i) == "-h")) {
      DisplayUsage();
      GlobalShutdown(-1);
    } else if((*i) == "--parse") {
      PARSE_ONLY = true;
    } else if((*i) == "--echo") {
      PARSE_ECHO = true;
    } else if((*i) == "--stat") {
      DISPLAY_STATISTICS = true;
    } else if((*i) == "--simFile") {
      if((++i) != commandLine.end()) {
	SIM_REL_FILE = (*i);
      } else --i;
    } else if((*i) == "--pacc") {
      IMPL_TYPE = IMPL_PACC;
    } else if((*i) == "--reach") {
      CONF_TYPE = CONF_REACH;
    } else if((*i) == "--trace") {
      CONF_TYPE = CONF_TRACE;
    } else if((*i) == "--ltl") {
      CONF_TYPE = CONF_LTL;
    } else if((*i) == "--deadlock") {
      CONF_TYPE = CONF_DLOCK;
    } else if((*i) == "--seaw") {
      CONF_TYPE = CONF_SEAW;
    } else if((*i) == "--learn") {
      CONF_TYPE = CONF_ANGLUIN;
    } else if((*i) == "--explicit") {
      CONF_CHECK_MECHANISM = CONF_CHECK_EXPLICIT;
    } else if((*i) == "--ltsa") {
      CONF_CHECK_MECHANISM = CONF_CHECK_LTSA;
    } else if((*i) == "--autoPred") {
      PREDS_FROM_SPEC = false;
    } else if((*i) == "--specSeed") {
      INIT_SEEDS_FROM_SPEC = true;
    } else if((*i) == "--cePred") {
      PREDS_FROM_CE = true;
    } else if((*i) == "--verbose") {
      if((++i) != commandLine.end()) {
	VERBOSITY_LEVEL = atoi(i->c_str());
	if(VERBOSITY_LEVEL < 0) {
	  Util::Error("verbosity level cannot be negative ...\n");
	}
      } else --i;
    } else if((*i) == "--dataComm") {
      HANDLE_DATA_COMM = true;
    } else if((*i) == "--liveVars") {
      USE_LIVE_VARS = true;
    } else if((*i) == "--inline") {
      INLINE_LIBS = true;
    } else if((*i) == "--invalidPtr") {
      CHECK_INVALID_PTR_DEREF = true;
    } else if((*i) == "--assert") {
      CHECK_ASSERTION_FAILURE = true;
    } else if((*i) == "--startIter") {
      if((++i) != commandLine.end()) {
	START_ITER = atoi(i->c_str());
	if(START_ITER < 1) {
	  Util::Error("starting iteration cannot be less than 1 ...\n");
	}
      } else --i;
    } else if((*i) == "--endIter") {
      if((++i) != commandLine.end()) {
	END_ITER = atoi(i->c_str());
	if(END_ITER < 1) {
	  Util::Error("ending iteration cannot be less than 1 ...\n");
	}
      } else --i;
    } else if((*i) == "--onlyIter") {
      if((++i) != commandLine.end()) {
	START_ITER = atoi(i->c_str());
	END_ITER = atoi(i->c_str());
	if(START_ITER < 1) {
	  Util::Error("starting and ending iteration cannot be less than 1 ...\n");
	}
      } else --i;
    } else if((*i) == "--noReachImpl") {
      USE_REACH_IMPL = false;
    } else if((*i) == "--abstraction") {
      if((++i) != commandLine.end()) {
	ABSTRACTION_NAME = (*i);
      } else --i;
    } else if((*i) == "--TPCache") {
      THEOREM_PROVER_CACHE = true;
    } else if((*i) == "--TPCacheSize") {
      if((++i) != commandLine.end()) {
	THEOREM_PROVER_CACHE_SIZE = atoi(i->c_str());
      } else --i;
    } else if((*i) == "--noImplTrans") {
      PRECOMPUTE_IMPL_TRANS = false;
    } else if((*i) == "--satSimRel") {
      SIM_REL_FROM_SAT = true;
    } else if((*i) == "--cprover") {
      THEOREM_PROVER_USED = THEOREM_PROVER_CPROVER;
    } else if((*i) == "--cvc") {
      THEOREM_PROVER_USED = THEOREM_PROVER_CVC;
    } else if((*i) == "--ics") {
      THEOREM_PROVER_USED = THEOREM_PROVER_ICS;
    } else if((*i) == "--svc") {
      THEOREM_PROVER_USED = THEOREM_PROVER_SVC;
    } else if((*i) == "--cvcl") {
      THEOREM_PROVER_USED = THEOREM_PROVER_CVCL;
    } else if((*i) == "--sato") {
      SAT_SOLVER_USED = SAT_SOLVER_SATO;
    } else if((*i) == "--grasp") {
      SAT_SOLVER_USED = SAT_SOLVER_GRASP;
    } else if((*i) == "--chaff") {
      SAT_SOLVER_USED = SAT_SOLVER_CHAFF;
    } else if((*i) == "--seed") {
      if((++i) != commandLine.end()) {
	//rnd.setSeed(Long.parseLong(*i));
      } else --i;
    } else if((*i) == "--preOpt") {
      PREOPTIMIZE = true;
    } else if((*i) == "--preOptVisit") {
      PREOPTVISIT = true;
    } else if((*i) == "--optPred") {
      CEGAR_TYPE = CEGAR_OPTIMIZE_PRED;
    } else if((*i) == "--usefulPred") {
      CEGAR_TYPE = CEGAR_USEFUL_PRED;
    } else if((*i) == "--greedy") {
      CEGAR_TYPE = CEGAR_GREEDY_PRED;
    } else if((*i) == "--largestSubset") {
      if((++i) != commandLine.end()) {
	LARGEST_SUBSET = atoi(i->c_str());
	if(LARGEST_SUBSET < 1) {
	  Util::Error("largest subset cannot be less than 1 ...\n");
	}
      } else --i;
    } else if((*i) == "--noIncremental") {
      INC_VERIFY = false;
    } else if((*i) == "--smallestSubset") {
      if((++i) != commandLine.end()) {
	SMALLEST_SUBSET = atoi(i->c_str());
	if(SMALLEST_SUBSET < 1) {
	  Util::Error("smallest subset cannot be less than 1 ...\n");
	}
      } else --i;
    } else if((*i) == "--subsetCutoff") {
      if((++i) != commandLine.end()) {
	SUBSET_CUTOFF = atoi(i->c_str());
	if(SUBSET_CUTOFF < 1) {
	  Util::Error("subset cutoff cannot be less than 1 ...\n");
	}
      } else --i;
    } else if((*i) == "--constraintCutoff") {
      if((++i) != commandLine.end()) {
	CONSTRAINT_CUTOFF = atoi(i->c_str());
	if(CONSTRAINT_CUTOFF < 1) {
	  Util::Error("constraint cutoff cannot be less than 1 ...\n");
	}
      } else --i;
    } else if((*i) == "--noStopAfterElim") {
      STOP_OPT_AFTER_ELIM = false;
    } else if((*i) == "--noNewVars") {
      CNF_NO_NEW_VARS = true;
    } else if((*i) == "--coverage") {
      COVERAGE = true;
    } else if((*i) == "--coverCont") {
      COVERCONT = true;
    } else if((*i) == "--showCoverage") {
      SHOWCOVER = true;
    } else if((*i) == "--cegar") {
      CEGAR = true;
    } else if((*i) == "--branching") {
      CHECK_CE_BRANCHING = true;
    } else if((*i) == "--pathBack") {
      PATH_CHECK_TYPE = PATH_CHECK_BACKWARD;
    } else if((*i) == "--ceShowAct") {
      CE_DISPLAY_TYPE = CE_LOC_ACT;
    } else if((*i) == "--ceShowCons") {
      CE_DISPLAY_TYPE = CE_LOC_CONS;
    } else if((*i) == "--ceShowAll") {
      CE_DISPLAY_TYPE = CE_LOC_CONS_ACT;
    } else if((*i) == "--checkCE") {
      CHECK_CE_VALIDITY = true;
    } else if((*i) == "--patMatch") {
      USE_PATTERN_MATCHER = true;
    } else if((*i) == "--noPredAbs") {
      USE_PRED_ABSTRACTION = false;
    } else if((*i) == "--ltsAbs") {
      USE_LTS_ABSTRACTION = true;
    } else if((*i) == "--noUselessLocs") {
      NO_USELESS_LOCS = true;
    } else if((*i) == "--silentTrans") {
      NO_SILENT_TRANS = false;
    } else if((*i) == "--ceDag") {
      if((++i) != commandLine.end()) {
	MAX_GLOBAL_CE = atoi(i->c_str());
	if(MAX_GLOBAL_CE < 1) {
	  Util::Error("max global CE dag cannot be less than 1 ...\n");
	}
      } else --i;
    } else if((*i) == "--noTransCache") {
      CACHE_IMPL_TRANS = false;
    } else if((*i) == "--predLoop") {
      if((++i) != commandLine.end()) {
	MAX_PRED_INFER_LOOP = atoi(i->c_str());
	if(MAX_PRED_INFER_LOOP < 1) {
	  Util::Error("predicate inference loop cannot be less than 1 ...\n");
	}
      } else --i;
    } else if((*i) == "--timeLimit") {
      if((++i) != commandLine.end()) {
	MAX_TIME_LIMIT = atoi(i->c_str());
	if(MAX_TIME_LIMIT < 0) {
	  Util::Error("maximum time limit cannot be negative ...\n");
	}
      } else --i;
    } else if((*i) == "--pointers") {
      IGNORE_POINTERS = false;
    } else if((*i) == "--noSyntactic") {
      USE_SYNTACTIC_CHECKS = false;
    } else if((*i) == "--ctool") {
      C_PARSER_TYPE = C_PARSER_CTOOL;
    } else if((*i) == "--pam") {
      USE_PAM = true;
    } else if((*i) == "--ltlToBuchi") {
      LTL_TO_BUCHI = true;
    } else if((*i) == "--drawLTS") {
      DRAW_LTS = true;
    } else if((*i) == "--drawCFG") {
      DRAW_CFG = true;
    } else if((*i) == "--drawInfPreds") {
      DRAW_INFERRED_PREDS = true;
    } else if((*i) == "--drawPredAbsLTS") {
      DRAW_PRED_ABS_LTS = true;
    } else if((*i) == "--drawFile") {
      if((++i) != commandLine.end()) {
	drawFileName = *i;
	if((drawFileName.size() < 3) || (drawFileName.rfind(".ps") != (drawFileName.size() - 3))) {
	  Util::Error("draw file name \"" + drawFileName + "\" does not end with \".ps\" ...\n");
	}
      } else --i;
    } else if((*i) == "--outputModels") {
      OUTPUT_MODELS = true;
    } else if((*i) == "--modelFile") {
      if((++i) != commandLine.end()) {
	modelFileName = *i;
      } else --i;
    } else if((*i) == "--explain") {
      EXPLAIN_CE = true;
    } else if((*i) == "--unwind") {
      if((++i) != commandLine.end()) {
	ADDITIONAL_UNWIND = atoi(i->c_str());
      }
      else --i;
    } else if((*i) == "--maxUnwind") {
      if((++i) != commandLine.end()) {
	MAX_UNWIND = atoi(i->c_str());
      }
      else --i;
    } else if((*i) == "--noMoreUnwind") {
      MORE_UNWIND = false;
    } else if((*i) == "--spuriousTries") {
      if((++i) != commandLine.end()) {
	SPURIOUS_TRIES = atoi(i->c_str());
      }
      else --i;
    } else if((*i) == "--actionWeight") {
      if((++i) != commandLine.end()) {
	ACTION_WEIGHT = atoi(i->c_str());
      }
      else --i;
    } else if((*i) == "--predWeight") {
      if((++i) != commandLine.end()) {
	PRED_WEIGHT = atoi(i->c_str());
      }
      else --i;
    } else if((*i) == "--noLTLNeg") {
      LTL_NEG = false;
    } else if((*i) == "--backup") {
      BACKUP = true;
    } else if((*i) == "--backupFile") {
      if((++i) != commandLine.end()) {
	backupFileName = *i;
      } else --i;
    } else if((*i) == "--restore") {
      RESTORE = true;
    } else if((*i) == "--restoreFile") {
      if((++i) != commandLine.end()) {
	restoreFileName = *i;
      } else --i;
    } else if((*i) == "--ceFile") {
      if((++i) != commandLine.end()) {
	CE_FILE = *i;
      } else --i;
    } else if((i->at(0)) == '-') {
      Util::Error("ERROR : unknown option %s ...\n"
		  "use magic --help for usage information ...\n",i->c_str());
    } else{
      FILE_NAMES.push_back(*i);
    }
  }

  //starting iteration cannot exceed ending iteration
  assert(START_ITER <= END_ITER);

  //if we are using action guided abstraction then we cannot check
  //simulation
  assert((!USE_LTS_ABSTRACTION) || (CONF_TYPE != CONF_SIMUL));

  //cannot check both assertion failure and invalid pointer
  //dereferences
  assert(!(CHECK_INVALID_PTR_DEREF && CHECK_ASSERTION_FAILURE));

  //no need to do incremental verification without LTS abstraction
  INC_VERIFY = USE_LTS_ABSTRACTION ? INC_VERIFY : false;

  //if we are using PAM then we must also use CTOOL
  C_PARSER_TYPE = USE_PAM ? C_PARSER_CTOOL : C_PARSER_TYPE;

  //if we have changed the CEGAR type from its default value then it
  //automatically implies we are doing counter example guided
  //abstraction refinement
  CEGAR = (CEGAR_TYPE != CEGAR_ELIMINATE_PRED) ? true : CEGAR;
  
  //only weak simulation can be checked in explicit state manner. all
  //other types of conformance checks are done using SAT.
  CONF_CHECK_MECHANISM = ((CONF_TYPE == CONF_SIMUL) || 
			  (((CONF_TYPE == CONF_TRACE) || (CONF_TYPE == CONF_REACH)) && 
			   (CONF_CHECK_MECHANISM == CONF_CHECK_LTSA))) ? CONF_CHECK_MECHANISM : CONF_CHECK_SAT;

  //if we are doing incremental verification we must use SAT for
  //verification and further we must use the HORNSAT solver
  CONF_CHECK_MECHANISM = INC_VERIFY ? CONF_CHECK_SAT : CONF_CHECK_MECHANISM;
  SAT_SOLVER_USED = INC_VERIFY ? SAT_SOLVER_HORN_AI : SAT_SOLVER_USED;

  //for counter-example guided abstraction refinement we need the
  //simulation relation even from SAT
  SIM_REL_FROM_SAT = CEGAR ? true : SIM_REL_FROM_SAT;

  //if we cache impl state transitions then the transitions must be
  //precomputed
  PRECOMPUTE_IMPL_TRANS = CACHE_IMPL_TRANS ? true : PRECOMPUTE_IMPL_TRANS;

  //drawing a more detailed graph implies drawing a less detailed
  //graph
  DRAW_CE_PROJ_LTS = DRAW_CE_PROJ_LTS  || DRAW_CE_PROJ_PRED; 
  DRAW_CE_DAG = DRAW_CE_DAG || DRAW_CE_PROJ_LTS;
  DRAW_COMPOSED_LTS = DRAW_COMPOSED_LTS || DRAW_CE_DAG;
  DRAW_LTS_ABS_LTS = DRAW_LTS_ABS_LTS || DRAW_COMPOSED_LTS;
  DRAW_PRED_ABS_LTS = DRAW_PRED_ABS_LTS || DRAW_LTS_ABS_LTS;
  DRAW_INFERRED_PREDS = DRAW_INFERRED_PREDS || DRAW_PRED_ABS_LTS;
  DRAW_CFG = DRAW_CFG || DRAW_INFERRED_PREDS;
  DRAW_LTS = DRAW_LTS || DRAW_CFG;

  //if we are to draw anything, we must create the output dot file
  CREATE_DOT_FILE = DRAW_LTS;

  //we cannot combine PACC with CEGAR
  assert((IMPL_TYPE != IMPL_PACC) || !CEGAR);

  //the initial seeds must come from the spec if one of the following
  //conditions is true: (i) we are not doing CEGAR (ii) we are doing
  //only action-guided abstraction
  INIT_SEEDS_FROM_SPEC = ((!CEGAR) || (!USE_PRED_ABSTRACTION)) ? true : INIT_SEEDS_FROM_SPEC;
}

/*********************************************************************/
//parse input files
/*********************************************************************/
void Database::ParseFiles()
{
#ifdef MAGIC_FULL
  Project *proj = NULL;
  if(C_PARSER_TYPE == C_PARSER_CTOOL) {
    proj = new Project();
    proj->SetPrintOption(false);
  }
#endif //MAGIC_FULL
  for(list<string>::iterator i = FILE_NAMES.begin();i != FILE_NAMES.end();++i) {
    //find the filename
    string name = *i;
    //check if the file has been already parsed
    if(IsFileParsed(name)) continue;
    //if it is a preprocessed C file
    if(name.rfind(".pp") == (name.length() - 3)) {
      Util::Message(2,"parsing preprocessed C file %s ...\n",name.c_str());
      if(C_PARSER_TYPE == C_PARSER_DEFAULT) {
	Stdcin = fopen(name.c_str(),"r");
	//if file could not be opened
	if(Stdcin == NULL) {
	  Util::Message(2,"could not open file %s for parsing ...\n",name.c_str());
	  continue;
	}
	//initialise the symbol table and set the file type
	CSymbolTable::fileType = CSymbolTable::STDC_FILE;
	CSymbolTable::Initialise();
	//parse and add the ast to the database if necessary
	Stdcparse();
	assert(CSymbolTable::ast != NULL);
	AddFileAst(name,CSymbolTable::ast);
      } 
#ifdef MAGIC_FULL
      else if(C_PARSER_TYPE == C_PARSER_CTOOL) {
	TransUnit *unit = proj->parse(name.c_str(),false);
	if(USE_PAM) PamAPI::ProcessFile(unit);
	AddFileAst(name,CtoolAPI::ToDefault(unit));
      }
#endif //MAGIC_FULL
      else assert(false);
    }
    //if it is a specification file
    else if(name.rfind(".spec") == (name.length() - 5)) {
      Util::Message(2,"parsing specification file %s ...\n",name.c_str());      
      Stdcin = fopen(name.c_str(),"r");
      //if file could not be opened
      if(Stdcin == NULL) {
	Util::Message(2,"could not open file %s for parsing ...\n",name.c_str());
	continue;
      }
      //initialise the symbol table and set the file type
      CSymbolTable::fileType = CSymbolTable::SPEC_FILE;
      CSymbolTable::Initialise();
      //parse and add the name of the spec file to the database
      Stdcparse();
      assert(CSymbolTable::ast == NULL);
      specFiles.insert(name);
    }
  }
  //sanity check on various SE-AW stuff
  SeawManager::SanityCheck();
#ifdef MAGIC_FULL
  if(USE_PAM) PamAPI::AliasAnalysis();
  if(proj != NULL) delete proj;
#endif //MAGIC_FULL
}

/*********************************************************************/
//returns true iff the file with given name has been parsed already
/*********************************************************************/
bool Database::IsFileParsed(const string &name)
{
  if(name.rfind(".pp") == (name.length() - 3)) {
    return (cAsts.count(name) != 0);
  } else if(name.rfind(".spec") == (name.length() - 5)) {
    return (specFiles.count(name) != 0);
  } else {
    Util::Error("ERROR: supplied file %s is of unknown type ...\n",name.c_str());
  }
}

/*********************************************************************/
//add a C filename and its ast to the appropriate map
/*********************************************************************/
void Database::AddFileAst(const string &name,StdcFile * ast)
{
  assert(name.rfind(".pp") == (name.length() - 3));
  cAsts[name] = ast;
}

/*********************************************************************/
//extract information about procedures
/*********************************************************************/
void Database::ExtractProcInfos()
{
  //for procedures with code available
  for(map<string,StdcFile*>::iterator i = cAsts.begin();i != cAsts.end();++i) {
    for(list<Proc*>::iterator j = i->second->procs.begin();j != i->second->procs.end();++j) {
      if(procAddInfos.count((*j)->name) == 0) {
	procInfos[(*j)->name] = ProcInfo(**j,i->first);
      } else {
	procInfos[(*j)->name] = ProcInfo(**j,i->first,procAddInfos[(*j)->name]);
      }
    }
  }
  //for procedures without code but only additional info
  for(map<string,ProcAddInfo>::const_iterator i = procAddInfos.begin();i != procAddInfos.end();++i) {
    if(procInfos.count(i->first) == 0) {
      procInfos[i->first] = ProcInfo(i->second);
    }
  }
  //make sure the procedure abstractions are mutually exclusive and
  //complete
  for(map<string,ProcInfo>::iterator i = procInfos.begin();i != procInfos.end();++i) {
    i->second.CheckAbstractions();
  }
}

/*********************************************************************/
//return info about LTS with given name
/*********************************************************************/
const LtsInfo &Database::GetLtsInfo(const string &name)
{
  //first check the global array
  for(size_t i = 0;i < ltsInfos.size();++i) {
    if(ltsInfos[i].GetName() == name) return ltsInfos[i];
  }
  //if not create the lts from the fsp info
  LtsInfo x = fspInfo.GetLtsInfo(name);
  if(DRAW_LTS) x.Draw();
  x.SetIndex(static_cast<char>(ltsInfos.size()));
  ltsInfos.push_back(x);
  return ltsInfos.back();
}

/*********************************************************************/
//return info about negation of the LTS with given name
/*********************************************************************/
const LtsInfo &Database::GetSpecLtsNeg(const string &name)
{
  //if we haven't negated this before
  if(ltsNegMap.count(name) == 0) {
    LtsInfo x = fspInfo.GetSpecLtsNeg(name);
    ltsNegMap[name] = x.GetName();
    x.SetIndex(static_cast<char>(ltsInfos.size()));
    ltsInfos.push_back(x);
    return ltsInfos.back();
  }
  //otherwise just look up the database
  else {
    string negName = ltsNegMap[name];
    for(size_t i = 0;i < ltsInfos.size();++i) {
      if(ltsInfos[i].GetName() == name) return ltsInfos[i];
    }
    assert(false);
  }
}

/*********************************************************************/
//return LTL formula with given name
/*********************************************************************/
LtlFormula Database::GetLtlFormula(const string &name)
{
  return LtlManager::GetFormula(name);
}

/*********************************************************************/
//return the actions of the specification. the spec could be either a
//FSP process or a LTL formula.
/*********************************************************************/
const set<Action> &Database::GetSpecActions()
{
  if((CONF_TYPE == CONF_SIMUL) || (CONF_TYPE == CONF_REACH) || 
     (CONF_TYPE == CONF_DLOCK) || (CONF_TYPE == CONF_ANGLUIN)) {
    return GetLtsInfo(specLtsName).GetActions();
  } else if(CONF_TYPE == CONF_LTL) {
    return LtlManager::GetActions(GetLtlFormula(specLtsName));
  } else if(CONF_TYPE == CONF_SEAW) {
    return SeawManager::GetActions(specLtsName);
  } else assert(false);
}

/*********************************************************************/
//return the assignment actions of the specification given a
//context. the spec could be either a FSP process or a LTL formula.
/*********************************************************************/
void Database::GetSpecAssignActions(const Expr &lhs,const Expr &rhs,
				    const list<string> &params,
				    const set<Expr> &context,set<Action> &res)
{
  if((CONF_TYPE == CONF_SIMUL) || (CONF_TYPE == CONF_REACH) || 
     (CONF_TYPE == CONF_DLOCK) || (CONF_TYPE == CONF_ANGLUIN)) {
    GetLtsInfo(specLtsName).GetAssignActions(lhs,rhs,params,context,res);
  } else if(CONF_TYPE == CONF_LTL) {
    GetLtlFormula(specLtsName).GetAssignActions(lhs,rhs,params,context,res);
  } else if(CONF_TYPE == CONF_SEAW) {
    SeawManager::GetAssignActions(specLtsName,lhs,rhs,params,context,res);
  } else assert(false);
}

/*********************************************************************/
//return the assignment actions of the specification given a
//context. the spec could be either a FSP process or a LTL formula.
/*********************************************************************/
void Database::GetSpecReturnActions(const Expr &expr,const list<string> &params,
				    const set<Expr> &context,set<Action> &res)
{
  if((CONF_TYPE == CONF_SIMUL) || (CONF_TYPE == CONF_REACH) || 
     (CONF_TYPE == CONF_DLOCK) || (CONF_TYPE == CONF_ANGLUIN)) {
    GetLtsInfo(specLtsName).GetReturnActions(expr,params,context,res);
  } else if(CONF_TYPE == CONF_LTL) {
    GetLtlFormula(specLtsName).GetReturnActions(expr,params,context,res);
  } else if(CONF_TYPE == CONF_SEAW) {
    SeawManager::GetReturnActions(specLtsName,expr,params,context,res);
  } else assert(false);
}

/*********************************************************************/
//return expressions associated with assign actions of the
//specification. the spec could be either a FSP process or a LTL
//formula.
/*********************************************************************/
void Database::GetSpecAssignActExprs(set< pair<Expr,Expr> > &res)
{
  if((CONF_TYPE == CONF_SIMUL) || (CONF_TYPE == CONF_REACH) || 
     (CONF_TYPE == CONF_DLOCK) || (CONF_TYPE == CONF_ANGLUIN)) {
    GetLtsInfo(specLtsName).GetAssignActExprs(res);
  } else if(CONF_TYPE == CONF_LTL) {
    GetLtlFormula(specLtsName).GetAssignActExprs(res);
  } else if(CONF_TYPE == CONF_SEAW) {
    SeawManager::GetAssignActExprs(specLtsName,res);
  } else assert(false);
}

/*********************************************************************/
//return expressions associated with return actions of the
//specification. the spec could be either a FSP process or a LTL
//formula.
/*********************************************************************/
void Database::GetSpecRetActExprs(set<Expr> &res)
{
  if((CONF_TYPE == CONF_SIMUL) || (CONF_TYPE == CONF_REACH) || 
     (CONF_TYPE == CONF_DLOCK) || (CONF_TYPE == CONF_ANGLUIN)) {
    GetLtsInfo(specLtsName).GetRetActExprs(res);
  } else if(CONF_TYPE == CONF_LTL) {
    GetLtlFormula(specLtsName).GetRetActExprs(res);
  } else if(CONF_TYPE == CONF_SEAW) {
    SeawManager::GetRetActExprs(specLtsName,res);
  } else assert(false);
}

/*********************************************************************/
//return number of states of the specification automata. the spec
//could be either a FSP process or a LTL formula. for a LTL formula
//return the number of states of the Buchi automaton.
/*********************************************************************/
int Database::GetSpecStateNum()
{
  if((CONF_TYPE == CONF_SIMUL) || (CONF_TYPE == CONF_REACH) ||  
     (CONF_TYPE == CONF_ANGLUIN)) {
    return GetLtsInfo(specLtsName).GetStates().size();
  } else if(CONF_TYPE == CONF_LTL) {
    return Buchi::GetStateNum();
  } else if(CONF_TYPE == CONF_DLOCK) return 1;
  else if(CONF_TYPE == CONF_SEAW) return 1;
  else assert(false);
}

/*********************************************************************/
//display usage information for MAGIC
/*********************************************************************/
void Database::DisplayUsage()
{
  Util::Message(1,"\n");
  Util::Message(1,"Usage: magic <option | input file> <option | input file> ...\n");
  Util::Message(1,"\n");
  Util::Message(1,"Common options:\n");
  Util::Message(1,
		"  --help or -h:\n"
		"\tgenerate this help message and exit.\n");
  Util::Message(1,
		"  --parse:\n"
		"\tonly parse input files and exit.\n");
  Util::Message(1,
		"  --echo:\n"
		"\tprint characters as they are being parsed.\n"
		"\tthis is helpful for debugging parse errors.\n");
  Util::Message(1,
		"  --stat:\n"
		"\tdisplay statistics at the end.\n");
  Util::Message(1,
		"  --simFile <file name>:\n"
		"\tname of file to output the computed maximal simulation\n"
		"\trelation. this could be broken.\n");
  Util::Message(1,
		"  --reach:\n"
		"\tcheck ERROR state reachability instead of simulation.\n");
  Util::Message(1,
		"  --trace:\n"
		"\tcheck trace containment instead of simulation.\n");
  Util::Message(1,
		"  --ltl:\n"
		"\tdo SE-LTL model checking. this is experimental.\\n");
  Util::Message(1,
		"  --deadlock:\n"
		"\tdo deadlock detection. this is experimental.\n");
  Util::Message(1,
		"  --seaw:\n"
		"\tdo SE-A-Omega model checking. this is experimental.\n");
  Util::Message(1,
		"  --learn:\n"
		"\tdo angluin learning. this is experimental.\n");
  Util::Message(1,
		"  --explicit:\n"
		"\tdo not use HORNSAT for verification. this could be broken.\n");
  Util::Message(1,
		"  --ltsa:\n"
		"\tuse the LTSA tool for verification. this lets us do assume-guarantee\n"
		"\tstyle reasoning along with learning.\n");
  Util::Message(1,
		"  --autoPred:\n"
		"\tautomatically add all branches in the code as seed predicates.\n"
		"\tnormally there are no predicates in the first iteration.\n");
  Util::Message(1,
		"  --specSeed:\n"
		"\tobtain initial seed predicates from spec files.\n");
  Util::Message(1,
		"  --verbose <level>:\n"
		"\tset the verbosity level. default is 2.\n");
  Util::Message(1,
		"  --dataComm:\n"
		"\tanalyse shared memory based communication between components.\n");
  Util::Message(1,
		"  --inline:\n"
		"\tinline all library routines whose code is available.\n");
  Util::Message(1,
		"  --invalidPtr:\n"
		"\tcheck for possible invalid pointer dereferences.\n");
  Util::Message(1,
		"  --assert:\n"
		"\tcheck for possible assertion failures.\n");
  Util::Message(1,
		"  --startIter <iteration number>:\n"
		"\tstarting iteration for invalid pointer dereference or assertion\n"
		"\tfailure check. default is 1.\n");
  Util::Message(1,
		"  --endIter <iteration number>:\n"
		"\tending iteration for invalid pointer dereference or assertion\n"
		"\tfailure check. default is 1.\n");
  Util::Message(1,
		"  --onlyIter <iteration number>:\n"
		"\tstarting and ending iteration for invalid pointer dereference or\n"
		"\tassertion failure check. default is 1.\n");
  Util::Message(1,
		"  --noReachImpl:\n"
		"\tdo not perform reachability implicitly. create all possible\n"
		"\tstates. this could increase memory requirement.\n");
  Util::Message(1,
		"  --abstraction <name>:\n"
		"\tthe name of the abstraction to be verified.\n"
		"\tin case of SE-LTL model checking this is the name\n"
		"\tof the formula to be verified.\n");
  Util::Message(1,
		"  --TPCache:\n"
		"\tcache theorem prover queries and their results.\n"
		"\tthis could increase memory requirement.\n");
  Util::Message(1,
		"  --TPCacheSize <size>:\n"
		"\tthe size of the theorem prover cache. the cache is cleared\n"
		"\tif this size is exceeded.\n");
  Util::Message(1,
		"  --optPred:\n"
		"\tdo pseudo-boolean constraint based predicate optimization.\n");
  Util::Message(1,
		"  --cegar:\n"
		"\tdo CEGAR without predicate optimization.\n");
  Util::Message(1,
		"  --noIncremental:\n"
		"\tdon't do incremental verification for two-level CEGAR.\n");
  Util::Message(1,
		"  --ceShowAct:\n"
		"\tshow actions when displaying counterexamples.\n");
  Util::Message(1,
		"  --ceShowCons:\n"
		"\tshow propositional constraints when displaying counterexamples.\n");
  Util::Message(1,
		"  --ceShowAll:\n"
		"\tshow actions and propositional constraints when displaying counterexamples.\n");
  Util::Message(1,
		"  --checkCE:\n"
		"\tcheck validity of counterexample if not doing CEGAR.\n");
  Util::Message(1,
		"  --noPredAbs:\n"
		"\tdo not do predicate refinement.\n");
  Util::Message(1,
		"  --ltsAbs:\n"
		"\tdo action-guided abstraction and refinement.\n");
  Util::Message(1,
		"  --silentTrans:\n"
		"\tdon't eliminate silent transitions.\n");
  Util::Message(1,
		"  --ceDag <number>:\n"
		"\tuse these many counterexamples in each iteration for refinement.\n");
  Util::Message(1,
		"  --predLoop <number>:\n"
		"\tnumber of times to go through a loop during predicate discovery.\n"
		"\tdefault value is 1.\n");
  Util::Message(1,
		"  --timeLimit <number>:\n"
		"\ttime limit in seconds. default value is 36000000.\n");
  Util::Message(1,
		"  --drawLTS:\n"
		"\tdraw all FSP processes used during verification.\n");
  Util::Message(1,
		"  --drawCFG:\n"
		"\tdraw control flow graph for each component. implies --drawLTS.\n");
  Util::Message(1,
		"  --drawInfPreds:\n"
		"\tdraw predicates used at each CFG node in each iteration.\n"
		"\timplies --drawLTS and --drawCFG.\n");
  Util::Message(1,
		"  --drawPredAbsLTS:\n"
		"\tdraw LTS created by predicate abstraction in each iteration.\n"
		"\timplies --drawLTS, --drawCFG and --drawInfPreds.\n");
  Util::Message(1,
		"  --drawFile <file name>:\n"
		"\tname of file where MAGIC will draw. file name must end with .ps.\n"
		"\tdefault is magic.ps.\n");
  Util::Message(1,
		"  --outputModels:\n"
		"\toutput models created in each iteration by predicate abstraction.\n"
		"\tby default, models are output as FSP processes.\n");
  Util::Message(1,
		"  --modelFile <file name>:\n"
		"\tname of file where MAGIC will output models. default is magic.fsp.\n");
  Util::Message(1,
		"  --explain:\n"
		"\tdo error explanation. this is experimental.\n");
  Util::Message(1,
		"  --backup:\n"
		"\tbackup MAGIC's state. this is experimental.\n");
  Util::Message(1,
		"  --backupFile <file name>:\n"
		"\tname of backup file. default is magic.backup.\n");
  Util::Message(1,
		"  --restore:\n"
		"\trestore MAGIC's state from backup file. this is experimental.\n");
  Util::Message(1,
		"  --restoreFile <file name>:\n"
		"\tname of backup file to restore from. default is magic.backup.\n");
  Util::Message(1,
		"  --ceFile <file name>:\n"
		"\tname of file to output counterexample to. default is empty string.\n"
		"\tif name is empty then counterexample is output to stdout.\n");
  Util::Message(1,"\n");
  Util::Message(1,"Less used options:\n");
  Util::Message(1,
		"  --pacc:\n"
		"\tthis option is meaningless.\n");
  Util::Message(1,
		"  --cePred:\n"
		"\tthis option is meaningless.\n");
  Util::Message(1,
		"  --liveVars:\n"
		"\tdo live variable analysis for better predicate discovery.\n"
		"\tthis could be broken.\n");
  Util::Message(1,
		"  --noImplTrans:\n"
		"\tdo not precompute the transitions of the implementation machine.\n");
  Util::Message(1,
		"  --satSimRel:\n"
		"\tconstruct the simulation relation from the HORNSAT solver's\n"
		"\tinternal data structures. this is automatically set to true\n"
		"\tif some abstraction refinement is to be done.\n");
  Util::Message(1,
		"  --cprover:\n"
		"\tuse CProver as the theorem prover. default is Simplify.\n"); 
  Util::Message(1,
		"  --cvc:\n"
		"\tuse CVC as the theorem prover.\n"); 
  Util::Message(1,
		"  --ics:\n"
		"\tuse ICS as the theorem prover.\n"); 
  Util::Message(1,
		"  --svc:\n"
		"\tuse SVC as the theorem prover.\n"); 
  Util::Message(1,
		"  --cvcl:\n"
		"\tuse CVC Lite as the theorem prover.\n"); 
  Util::Message(1,
		"  --sato:\n"
		"\tuse Sato as the SAT solver. default is the internal HORNSAT solver.\n");
  Util::Message(1,
		"  --grasp:\n"
		"\tuse FGrasp as the SAT solver.\n");
  Util::Message(1,
		"  --chaff:\n"
		"\tuse ZChaff as the SAT solver.\n");
  Util::Message(1,
		"  --seed:\n"
		"\tthis option is meaningless.\n");
  Util::Message(1,
		"  --preOpt:\n"
		"\tthis option is meaningless.\n");
  Util::Message(1,
		"  --predOptVisit:\n"
		"\tthis option is meaningless.\n");
  Util::Message(1,
		"  --usefulPred:\n"
		"\tuse only useful predicates for predicate discovery.\n");
  Util::Message(1,
		"  --greedy:\n"
		"\tdo greedy predicate optimization.\n");
  Util::Message(1,
		"  --largestSubset <size>:\n"
		"\tthe maximum size of subsets tried during predicate optimization.\n");
  Util::Message(1,
		"  --smallestSubset <size>:\n"
		"\tthe minimum size of subsets tried during predicate optimization.\n");
  Util::Message(1,
		"  --subsetCutoff <size>:\n"
		"\tthe maximum number of subsets tried during predicate optimization.\n");
  Util::Message(1,
		"  --constraintCutoff <size>:\n"
		"\tthe maximum number of constraints generated during predicate optimization.\n");
  Util::Message(1,
		"  --noStopAfterElim:\n"
		"\tdo not stop generating constraints even if the counterexample\n"
		"\thas been eliminated during predicate optimization.\n");
  Util::Message(1,
		"  --noNewVars:\n"
		"\tdon't use new variable for PBS encoding during predicate minimization.\n"
		"\tthis might be broken.\n");
  Util::Message(1,
		"  --coverage:\n"
		"\tthis option is meaningless.\n");
  Util::Message(1,
		"  --coverCont:\n"
		"\tthis option is meaningless.\n");
  Util::Message(1,
		"  --showCoverage:\n"
		"\tthis option is meaningless.\n");
  Util::Message(1,
		"  --branching:\n"
		"\tconsider branching when validating tree counterexamples.\n");
  Util::Message(1,
		"  --pathBack:\n"
		"\tvalidate counterexamples in a backward manner.\n");
  Util::Message(1,
		"  --patMatcher:\n"
		"\tthis option is meaningless.\n");
  Util::Message(1,
		"  --noUselessLocs:\n"
		"\tignore useless control locations. this might be broken.\n");
  Util::Message(1,
		"  --noTransCache:\n"
		"\tdo not cache transitions of predicate abstraction models.\n");
  Util::Message(1,
		"  --pointers:\n"
		"\thandle aliasing due to pointers.\n");
  Util::Message(1,
		"  --noSyntactic:\n"
		"\tno syntactic checks before calling theorem prover.\n");
  Util::Message(1,
		"  --ctool:\n"
		"\tuse CTOOL as the C parser. this might be broken.\n");
  Util::Message(1,
		"  --pam:\n"
		"\tuse PAM for alias analysis. this is experimental.\n");
  Util::Message(1,
		"  --ltlToBuchi:\n"
		"\tconvert SE-LTL formula to Buchi automaton. no model checking.\n");
  Util::Message(1,
		"  --unwind <depth>:\n"
		"\tunwind depth for error explanation.\n");
  Util::Message(1,
		"  --maxUnwind <depth>:\n"
		"\tmaximum unwind depth for error explanation.\n");
  Util::Message(1,
		"  --noMoreUnwind:\n"
		"\tstop unwinding during error explanation.\n");
  Util::Message(1,
		"  --spuriousTries <number>:\n"
		"\tnumber of attempts for spurious counterexamples during error explanation.\n");
  Util::Message(1,
		"  --actionWeight <weight>:\n"
		"\tset weightage of actions for error explanation.\n");
  Util::Message(1,
		"  --predWeight <weight>:\n"
		"\tset weightage of predicates for error explanation.\n");
  Util::Message(1,
		"  --noLTLNeg:\n"
		"\tdon't negate specification for error explanation.\n");
  Util::Message(1,"\n");
}

/*********************************************************************/
//end of Database.cpp
/*********************************************************************/
