/******************************** CPPFile *****************************

* FileName [ContLocSet.cpp]

* PackageName [main]

* Synopsis [Method definitions of ContLocSet class.]

* SeeAlso [ContLocSet.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <gmp.h>
#include <string>
#include <list>
#include <set>
#include <map>
#include <vector>
using namespace std;

#include "BigInt.h"
#include "Node.h"
#include "Action.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "ContLoc.h"
#include "ContLocSet.h"
using namespace magic;

ContLocSet::ContLocSet() 
{
  initLoc = NULL;
}

ContLocSet::ContLocSet(const ContLocSet &rhs)
{
  *this = rhs;
}

const ContLocSet &ContLocSet::operator = (const ContLocSet &rhs)
{
  contLocs = rhs.contLocs;
  initLoc = rhs.initLoc;
  finalLocs = rhs.finalLocs;
  returnLocs = rhs.returnLocs;
  labelLocs = rhs.labelLocs;
  breakLocs = rhs.breakLocs;
  continueLocs = rhs.continueLocs;
  gotoLocs = rhs.gotoLocs;
  finalExpr = rhs.finalExpr;
  return *this;
}

/*********************************************************************/
//concatenate another set of locations at the end of this
/*********************************************************************/
void ContLocSet::Concat(const ContLocSet &rhs)
{
  //check if either of the sets is empty
  if(contLocs.empty()) {
    *this = rhs;
    return;
  } else if(rhs.contLocs.empty()) return;

  //combine the locations
  ConcatLocs(contLocs,rhs.contLocs);
  ConcatLocs(returnLocs,rhs.returnLocs);
  ConcatLocs(labelLocs,rhs.labelLocs);
  ConcatLocs(breakLocs,rhs.breakLocs);
  ConcatLocs(continueLocs,rhs.continueLocs);
  ConcatLocs(gotoLocs,rhs.gotoLocs);

  //make the initial control location of the second set the
  //successor of all the final control locations of the first
  //set
  for(list<ContLoc*>::iterator i = finalLocs.begin();i != finalLocs.end();++i) {
    (*i)->SetSucc(rhs.initLoc);
  }

  //set the final locations of the first set to be the same as
  //the second
  finalLocs = rhs.finalLocs;
  
  //set the final expression of the first to be the same as the
  //second
  SetFinalExpr(rhs.finalExpr);
}

/*********************************************************************/
//merge with another set of locations
/*********************************************************************/
void ContLocSet::Merge(const ContLocSet &rhs)
{
  //combine the locations
  ConcatLocs(contLocs,rhs.contLocs);
  ConcatLocs(finalLocs,rhs.finalLocs);
  ConcatLocs(returnLocs,rhs.returnLocs);
  ConcatLocs(labelLocs,rhs.labelLocs);
  ConcatLocs(breakLocs,rhs.breakLocs);
  ConcatLocs(continueLocs,rhs.continueLocs);
  ConcatLocs(gotoLocs,rhs.gotoLocs);
}

/*********************************************************************/
//add a set of control locations before another. the difference with
//concatenation is that the final locations of the first set are not
//linked to the initial location of the second set.
/*********************************************************************/
void ContLocSet::Append(const ContLocSet &rhs)
{
  //combine the locations
  ConcatLocs(contLocs,rhs.contLocs);
  ConcatLocs(finalLocs,rhs.finalLocs);
  ConcatLocs(returnLocs,rhs.returnLocs);
  ConcatLocs(labelLocs,rhs.labelLocs);
  ConcatLocs(breakLocs,rhs.breakLocs);
  ConcatLocs(continueLocs,rhs.continueLocs);
  ConcatLocs(gotoLocs,rhs.gotoLocs);

  //set the final locations of the first set to be the same as
  //the second
  finalLocs = rhs.finalLocs;
  
  //set the final expression of the first to be the same as the
  //second
  SetFinalExpr(rhs.finalExpr);
}

/*********************************************************************/
//append the second list of control locations after the first
/*********************************************************************/
void ContLocSet::ConcatLocs(list<ContLoc*> &one,const list<ContLoc*> &two)
{
  one.insert(one.end(),two.begin(),two.end());
}

void ContLocSet::SetFinalExpr(const Expr &e)
{
  finalExpr = e;
}

/*********************************************************************/
//return the set of labeled locations with identifier labels i.e. not
//cases or defaults
/*********************************************************************/
list<ContLoc*> ContLocSet::GetIdLabelLocs()
{
  list<ContLoc*> res;
  for(list<ContLoc*>::iterator i = labelLocs.begin();i != labelLocs.end();++i) {
    if((*i)->GetLocType() == ContLoc::LABEL_ID) {
      res.push_back(*i);
    }
  }
  return res;
}

/*********************************************************************/
//return the set of labeled locations with case labels
/*********************************************************************/
list<ContLoc*> ContLocSet::GetCaseLocs()
{
  list<ContLoc*> res;
  for(list<ContLoc*>::iterator i = labelLocs.begin();i != labelLocs.end();++i) {
    if((*i)->GetLocType() == ContLoc::LABEL_CASE) {
      res.push_back(*i);
    }
  }
  return res;
}

/*********************************************************************/
//return the set of labeled locations with default labels
/*********************************************************************/
list<ContLoc*> ContLocSet::GetDefaultLocs()
{
  list<ContLoc*> res;
  for(list<ContLoc*>::iterator i = labelLocs.begin();i != labelLocs.end();++i) {
    if((*i)->GetLocType() == ContLoc::LABEL_DEFAULT) {
      res.push_back(*i);
    }
  }
  return res;
}

/*********************************************************************/
//make all the branch locations in this set useful
/*********************************************************************/
void ContLocSet::MakeBranchesUseful()
{
  for(list<ContLoc*>::iterator i = contLocs.begin();i != contLocs.end();++i) {
    if((*i)->GetLocType() == ContLoc::BRANCH) {
      (*i)->SetUseful(true);
    }
  }
}

/*********************************************************************/
//make all the branch locations in this set fair
/*********************************************************************/
void ContLocSet::MakeBranchesFair()
{
  for(list<ContLoc*>::iterator i = contLocs.begin();i != contLocs.end();++i) {
    if((*i)->GetLocType() == ContLoc::BRANCH) {
      (*i)->SetFairLoop(true);
    }
  }
}

/*********************************************************************/
//clear the list of case locations
/*********************************************************************/
void ContLocSet::HandleCases()
{
  list<ContLoc*> newLabelLocs;
  for(list<ContLoc*>::iterator i = labelLocs.begin();i != labelLocs.end();++i) {
    if((*i)->GetLocType() != ContLoc::LABEL_CASE) {
      newLabelLocs.push_back(*i);
    }
  }
  labelLocs = newLabelLocs;
}

/*********************************************************************/
//make break locations final locations and clear the list of break
//locations
/*********************************************************************/
void ContLocSet::HandleBreaks()
{
  ConcatLocs(finalLocs,breakLocs);
  breakLocs.clear();
}

/*********************************************************************/
//make the initial location the successor of all continue
//locations
/*********************************************************************/
void ContLocSet::HandleContinues()
{
  for(list<ContLoc*>::iterator i = continueLocs.begin();i != continueLocs.end();++i) {
    (*i)->SetSucc(initLoc);
  }
  continueLocs.clear();
}

/*********************************************************************/
//end of ContLocSet.cpp
/*********************************************************************/
