/******************************** CPPFile *****************************

* FileName [ContLocExtractor.cpp]

* PackageName [main]

* Synopsis [Method definitions of ContLocExtractor class.]

* SeeAlso [ContLocExtractor.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <gmp.h>
#include <string>
#include <list>
#include <map>
#include <vector>
#include <set>
using namespace std;

#include "BigInt.h"
#include "Util.h"
#include "Node.h"
#include "ProcAbs.h"
#include "Action.h"
#include "DFSAdapter.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "ContLoc.h"
#include "ContLocSet.h"
#include "ProcInfo.h"
#include "ContLocExtractor.h"
#define YYSTYPE int
#include "StdcParser.h"
using namespace magic;

/*********************************************************************/
//constructors
/*********************************************************************/
ContLocExtractor::ContLocExtractor() { assert(false); }

ContLocExtractor::ContLocExtractor(const string &n) : procName(n) {}

ContLocExtractor::ContLocExtractor(const ContLocExtractor &rhs)
{
  *this = rhs;
}

ContLocExtractor::~ContLocExtractor() {}

/*********************************************************************/
//operators
/*********************************************************************/
const ContLocExtractor &ContLocExtractor::operator = (const ContLocExtractor &rhs)
{
  procName = rhs.procName;
  return *this;
}

/*********************************************************************/
//manipulation the cache
/*********************************************************************/
ContLocSet &ContLocExtractor::GetOut(const Node *n)
{
  return cache[n];
}

void ContLocExtractor::SetOut(const Node *n,const ContLocSet &c) 
{
  assert(cache.count(n) == 0);
  cache[n] = c;
}

/*********************************************************************/
//overloaded methods for expressions
/*********************************************************************/
void ContLocExtractor::outIdExpr(const IdExpr &node)
{
  SetOut(&node,ContLocSet());
}

/*********************************************************************/
void ContLocExtractor::outIntConstExpr(const IntConstExpr &node)
{
  SetOut(&node,ContLocSet());
}

/*********************************************************************/
void ContLocExtractor::outConstExpr(const ConstExpr &node)
{
  SetOut(&node,ContLocSet());
}

/*********************************************************************/
void ContLocExtractor::outStrExpr(const StrExpr &node)
{
  SetOut(&node,ContLocSet());
}

/*********************************************************************/
void ContLocExtractor::outStrStrExpr(const StrStrExpr &node)
{
  SetOut(&node,ContLocSet());
}

/*********************************************************************/
void ContLocExtractor::outStmtExpr(const StmtExpr &node)
{
  Util::Error("compound statements not handled : " + 
	      node.ToString() + "\n");
}

/*********************************************************************/
void ContLocExtractor::outBrackExpr(const BrackExpr &node)
{
  ContLocSet x = GetOut(node.array);
  ContLocSet y = GetOut(node.index);
  ContLocSet res = x;
  res.Concat(GetOut(node.index));

  if(!res.IsEmpty()) {
    Expr a = x.IsEmpty() ? Expr(node.array) : x.GetFinalExpr();
    Expr b = y.IsEmpty() ? Expr(node.index) : y.GetFinalExpr();
    Expr c = ExprManager::GetBrackExpr(a,b);
    res.SetFinalExpr(c);
  }
  SetOut(&node,res);
}

/*********************************************************************/
void ContLocExtractor::outParExpr(const ParExpr &node)
{
  ContLocSet res = GetOut(node.proc);
  
  Expr b = res.IsEmpty() ? Expr(node.proc) : res.GetFinalExpr();
  
  list<Expr> args;
  list<BasicExpr*> &c = node.args->data;
  for(list<BasicExpr*>::iterator i = c.begin();i != c.end();++i) {
    ContLocSet d = GetOut(*i);
    Expr e;
    if(d.IsEmpty()) {
      e = Expr(*i);
    } else {
      res.Concat(d);
      e = d.GetFinalExpr();
    }
    args.push_back(e);
  }
  if(!res.IsEmpty()) {
    Expr l = ExprManager::GetParExpr(b,args);
    res.SetFinalExpr(l);
  }

  SetOut(&node,res);
}

/*********************************************************************/
void ContLocExtractor::outDotExpr(const DotExpr &node) 
{
  ContLocSet res = GetOut(node.expr);
  if(!res.IsEmpty()) {
    Expr b = ExprManager::GetDotExpr(res.GetFinalExpr(),node.id);
    res.SetFinalExpr(b);
  }
  SetOut(&node,res);
}

/*********************************************************************/
void ContLocExtractor::outArrowExpr(const ArrowExpr &node) 
{
  ContLocSet res = GetOut(node.expr);
  if(!res.IsEmpty()) {
    Expr b = ExprManager::GetArrowExpr(res.GetFinalExpr(),node.id);
    res.SetFinalExpr(b);
  }
  SetOut(&node,res);
}

/*********************************************************************/
//we handle both pre and post increments and decrements identically
//because CIL ensures that these expressions are not embedded in other
//expressions. if we decide not to use CIL then the pre and post
//operators have to be treated differently.
/*********************************************************************/
void ContLocExtractor::outIncExpr(const IncExpr &node)
{
  //the following statement modified because we treat pre and post
  //operators identically
  //pre increment or decrement
  //if((node.type == Database::PRE_INC) || (node.type == Database::PRE_DEC)) {
  if((node.type == Database::PRE_INC) || (node.type == Database::PRE_DEC) ||
     (node.type == Database::POST_INC) || (node.type == Database::POST_DEC)) {
    ContLocSet x = GetOut(node.expr);
    
    //create the assignment that does the increment or decrement
    Expr d = ExprManager::GetIntConstExpr(1);
    //the following statement modified because we treat pre and post
    //operators identically
    //Expr *e = new BinaryExpr(node.expr,d,(node.type == Database::PRE_INC) ? '+' : '-');
    Expr e = ExprManager::GetBinaryExpr(node.expr,d,((node.type == Database::PRE_INC) || (node.type == Database::POST_INC)) ? '+' : '-');

    //create the control locations for the assignment statement
    ContLoc *k = CreateAssignContLoc(node.expr,e.GetExpr());
    ContLocSet l = CreateSingletonContLocSet(k,Expr(node.expr));
    
    //concatenate the assignment before the evaluation of the
    //expression
    l.Concat(x);
    SetOut(&node,l);
  }
  //this should not happen
  else assert(false);
}

/*********************************************************************/
void ContLocExtractor::outUnaryExpr(const UnaryExpr &node) 
{
  ContLocSet res = GetOut(node.expr);
  if(!res.IsEmpty()) {
    Expr a = res.GetFinalExpr();
    Expr b = ExprManager::GetUnaryExpr(a,node.op);
    res.SetFinalExpr(b);
  }
  SetOut(&node,res);
}

/*********************************************************************/
void ContLocExtractor::outEmptyExpr(const EmptyExpr &node) 
{
  SetOut(&node,ContLocSet());
}

/*********************************************************************/
void ContLocExtractor::outBinaryExpr(const BinaryExpr &node) 
{
  //logical and operator : according to C semantics if the first
  //sub-expression is false then the whole expression is false and the
  //second sub-expression is not even evaluated
  if(node.op == MAGIC_AND_OP) {
    //create a false expression
    Expr z = ExprManager::GetIntConstExpr(0);
    
    //create the locations for the conditional expression
    ContLocSet res = CreateCondExprContLocSet(node.lhs,node.rhs,z.GetExpr());
    SetOut(&node,res);
  }
  //logical or operator : according to C semantics if the first
  //sub-expression is true then the whole expression is true and the
  //second sub-expression is not even evaluated
  else if(node.op == MAGIC_OR_OP) {
    //create a true expression
    Expr y = ExprManager::GetIntConstExpr(1);
    
    //create the locations for the conditional expression
    ContLocSet res = CreateCondExprContLocSet(node.lhs,y.GetExpr(),node.rhs);
    SetOut(&node,res);
  }
  //all other operators
  else {
    ContLocSet x = GetOut(node.lhs);
    ContLocSet y = GetOut(node.rhs);
    ContLocSet res = x;
    res.Concat(y);
    
    if(!res.IsEmpty()) {
      Expr a = (x.IsEmpty()) ? Expr(node.lhs) : x.GetFinalExpr();
      Expr b = (y.IsEmpty()) ? Expr(node.rhs) : y.GetFinalExpr();
      Expr c = (node.op == ',') ? b : ExprManager::GetBinaryExpr(a,b,node.op);
      res.SetFinalExpr(c);
    }
    SetOut(&node,res);
  }
}

/*********************************************************************/
void ContLocExtractor::outQuestExpr(const QuestExpr &node) 
{
  SetOut(&node,CreateCondExprContLocSet(node.cond,node.tcase,node.ecase));
}

/*********************************************************************/
void ContLocExtractor::outAssignExpr(const AssignExpr &node)
{  
  //get the locations and final expression for the lhs of the assignment
  ContLocSet lset = GetOut(node.lhs);
  ContLocSet res = lset;
  const BasicExpr *lhs = (lset.IsEmpty()) ? node.lhs : lset.GetFinalExpr().GetExpr();

  //get the locations and final expression for the rhs of the assignment
  ContLocSet rset = GetOut(node.rhs);
  const BasicExpr *rhs = (rset.IsEmpty()) ? node.rhs : rset.GetFinalExpr().GetExpr();
  res.Concat(rset);

   //create the control location
  ContLoc *x = CreateAssignContLoc(lhs,rhs);
  
  //create the control location set
  ContLocSet d = CreateSingletonContLocSet(x,lhs);
  res.Concat(d);
  res.SetFinalExpr(lhs);

  SetOut(&node,res);
}

/*********************************************************************/
//overloaded methods for statements
/*********************************************************************/
void ContLocExtractor::outLabelStmt(const LabelStmt &node) 
{
  ContLoc *x = new ContLoc(&Database::procInfos[procName],ContLoc::LABEL_ID);
  allLocs.insert(x);
  x->SetLabel(node.id);
  ContLocSet y = CreateSingletonContLocSet(x,Expr());
  y.Concat(GetOut(node.stmt));
  SetOut(&node,y);
}

/*********************************************************************/
void ContLocExtractor::outCaseStmt(const CaseStmt &node)
{
  ContLoc *x = new ContLoc(&Database::procInfos[procName],ContLoc::LABEL_CASE);
  allLocs.insert(x);
  x->SetLocExpr(node.expr);
  ContLocSet y = CreateSingletonContLocSet(x,Expr());
  y.Concat(GetOut(node.stmt));
  SetOut(&node,y);
}

/*********************************************************************/
void ContLocExtractor::outDefaultStmt(const DefaultStmt &node)
{
  ContLoc *x = new ContLoc(&Database::procInfos[procName],ContLoc::LABEL_DEFAULT);
  allLocs.insert(x);
  ContLocSet y = CreateSingletonContLocSet(x,Expr());
  y.Concat(GetOut(node.stmt));
  SetOut(&node,y);
}

/*********************************************************************/
void ContLocExtractor::outCompStmt(const CompStmt &node)
{
  ContLocSet res;
  
  //add the locations for the statements
  list<Stmt*> &stmts = node.data->data;
  for(list<Stmt*>::iterator i = stmts.begin();i != stmts.end();++i) {
    res.Concat(GetOut(*i));
  }

  //associate the final result with the node
  SetOut(&node,res);
  
  //if this is the procedure body then we have to update the
  //procedure info
  if(&node == &(Database::procInfos[procName].GetAst())) {
    ProcInfo *procInfo = &Database::procInfos[procName];
    ContLocSet cls = GetOut(&node);
    if(cls.IsEmpty()) {
      //this means that the procedure body has no control
      //locations.  create a return location.
      ContLoc *x = new ContLoc(procInfo,ContLoc::RETURN);
      allLocs.insert(x);
      Expr fx = ExprManager::GetEmptyExpr();
      x->SetLocExpr(fx);
      
      //make a final location
      ContLoc *a = new ContLoc(procInfo,ContLoc::FINAL);
      allLocs.insert(a);
      x->SetTSucc(a);
      procInfo->SetFinalLoc(a);
      
      //make the return location the initial location
      procInfo->SetInitLoc(x);
      
      //create the set of locations
      list<ContLoc*> y; 
      y.push_back(x);
      y.push_back(a);
      procInfo->SetContLocs(y);
    } else {
      //handle goto statements
      HandleGotos(cls);
      
      //copy the stuff over from the procedure body and create a
      //special final control location
      ContLoc *x = new ContLoc(procInfo,ContLoc::FINAL);
      allLocs.insert(x);
      cls.GetContLocs().push_back(x);
      
      //make the new final location a successor of all final
      //locations of the procedure. add a dummy return
      //statement in between.
      for(list<ContLoc*>::iterator i = cls.GetFinalLocs().begin();i != cls.GetFinalLocs().end();++i) {
	int c = (*i)->GetLocType();
	if((c != ContLoc::ASSIGN_EXPR) && (c != ContLoc::ASSIGN_CALL) &&
	   (c != ContLoc::BRANCH)) {
	  Util::Error("special final control location : " + (*i)->ToString() + "\n");
	} else {
	  ContLoc *b = new ContLoc(procInfo,ContLoc::RETURN);
	  allLocs.insert(b);
	  Expr fx = ExprManager::GetEmptyExpr();
	  b->SetLocExpr(fx);
	  (*i)->SetSucc(b);
	  b->SetTSucc(x);
	  cls.GetContLocs().push_back(b);
	}
      }
      //make the new final location a successor of all return
      //locations of the procedure.
      for(list<ContLoc*>::iterator i = cls.GetReturnLocs().begin();i != cls.GetReturnLocs().end();++i) {
	if((*i)->GetLocType() == ContLoc::RETURN) {
	  (*i)->SetTSucc(x);
	} else {
	  Util::Error("return location with non-return type : " + (*i)->ToString() + "\n");
	}
      }
      
      procInfo->SetContLocs(cls.GetContLocs());
      procInfo->SetInitLoc(cls.GetInitLoc());
      procInfo->SetFinalLoc(x);
    }

    //sanity check - make sure no control locations have been left
    //behind
    assert(procInfo->GetContLocs().size() == allLocs.size());
  }
}

/*********************************************************************/
void ContLocExtractor::outExprStmt(const ExprStmt &node) 
{
  ContLocSet res = GetOut(node.expr);
  pair<bool,int> a = Util::IsProcCall(node.expr);
  if(a.first) {
    ContLoc *x = new ContLoc(&Database::procInfos[procName],ContLoc::ASSIGN_CALL);
    allLocs.insert(x);
    string tv = Util::NewTempVar();
    Database::procInfos[procName].AddLocal(tv);
    Expr y = ExprManager::GetIdExpr(tv);
    x->SetLocLhs(y);
    x->SetLocRhs(node.expr);
    x->SetIndCallKey(a.second);
    ContLocSet z = CreateSingletonContLocSet(x,y);
    res.Concat(z);
  } else if(res.IsEmpty()) {
    Util::Message(2,"warning : ignored statement : " + node.ToString() + "\n");
  }
  SetOut(&node,res);
}

/*********************************************************************/
void ContLocExtractor::outEmptyStmt(const EmptyStmt &node) 
{
  SetOut(&node,ContLocSet());
}

/*********************************************************************/
void ContLocExtractor::outIfStmt(const IfStmt &node) 
{
  //create the control locations for the branch condition
  ContLoc *x = new ContLoc(&Database::procInfos[procName],ContLoc::BRANCH);
  allLocs.insert(x);
  ContLocSet brSet = GetBranchContLocSet(x,node.expr,true);
  
  //set the branch expression
  if(brSet.GetContLocs().size() > 1) {
    x->SetLocExpr(brSet.GetFinalExpr());
  } else {
    x->SetLocExpr(node.expr);
  }
  
  ContLocSet y = GetOut(node.tstmt);
  ContLocSet z = GetOut(node.estmt);
  ContLocSet res;
  
  if((y.IsEmpty()) && (z.IsEmpty())) {
    res = brSet;
  } else if(y.IsEmpty()) {
    z.GetFinalLocs().push_back(x);
    x->SetESucc(z.GetInitLoc());
    res = brSet;
    res.Append(z);
  } else if(z.IsEmpty()) {
    y.GetFinalLocs().push_back(x);
    x->SetTSucc(y.GetInitLoc());
    res = brSet;
    res.Append(y);
  } else {
    x->SetTSucc(y.GetInitLoc());
    x->SetESucc(z.GetInitLoc());
    y.Merge(z);
    res = brSet;
    res.Append(y);
  }

  //associate this set of locations with the node
  SetOut(&node,res);
}

/*********************************************************************/
void ContLocExtractor::outSwitchStmt(const SwitchStmt &node)
{
  ContLocSet x = GetOut(node.expr);
  ContLocSet y = GetOut(node.stmt);
  ContLocSet res;
  Expr switchExpr;
  if(x.IsEmpty()) {
    switchExpr = Expr(node.expr);
  } else {
    switchExpr = x.GetFinalExpr();
    res = x;
  }

  if(!y.IsEmpty()) {
    //get the case and default locations
    list<ContLoc*> a = y.GetCaseLocs();
    list<ContLoc*> b = y.GetDefaultLocs();
    if(b.size() > 1) {
      Util::Message(2,"warning : multiple default cases in switch statement in procedure " + 
		   procName + "\n");
    }
    if(a.size() == 0) {
      Util::Error("no switch cases in " + node.ToString() + "\n");
    }
    //create the if-then blocks corresponding to the different
    //cases
    for(list<ContLoc*>::iterator i = a.begin();i != a.end();++i) {
      //create the equality expression
      Expr f = ExprManager::GetBinaryExpr(switchExpr,(*i)->GetLocExpr(),MAGIC_EQ_OP);

      //create the branch control location
      ContLoc *h = new ContLoc(&Database::procInfos[procName],ContLoc::BRANCH);
      allLocs.insert(h);
      h->SetLocExpr(f);
      //this is important because we might have multiple
      //case statements leading to the same block of
      //code. we have to set the true successor if this
      //branch condition to be the first non-case or
      //non-default successor of this case label.
      ContLoc *c = *i;
      while((c != NULL) && 
	    ((c->GetLocType() == ContLoc::LABEL_CASE) ||
	     (c->GetLocType() == ContLoc::LABEL_DEFAULT))) {
	c = c->GetTSucc();
      }
      if(c == NULL) Util::Error("ERROR: case statement without any body ...\n");
      h->SetTSucc(c);
      ContLocSet l = CreateSingletonContLocSet(h,Expr());
      res.Concat(l);
      
      //if the branch condition is an important predicate
      //specified by the user then make this branch location
      //useful
      CheckBranchUseful(l,f);
    }
    //create the link to the default block if any
    if(b.size() != 0) {
      ContLoc *m = b.front();
      for(list<ContLoc*>::iterator j = res.GetFinalLocs().begin();j != res.GetFinalLocs().end();++j) {
	(*j)->SetSucc(m);
      }
      //clear the final statements
      res.GetFinalLocs().clear();
    }
    //eliminate the case statements
    y.HandleCases();
    //merge the sets of control locations for the if-then
    //block and the body of the switch statement
    res.Merge(y);
  }
  
  //handle the break statements
  res.HandleBreaks();
  
  SetOut(&node,res);
}

/*********************************************************************/
void ContLocExtractor::outWhileStmt(const WhileStmt &node)
{
  //create the control location for the loop condition
  ContLoc *x = new ContLoc(&Database::procInfos[procName],ContLoc::BRANCH);
  allLocs.insert(x);
  ContLocSet brSet = GetBranchContLocSet(x,node.expr,true);
  
  //set the branch expression
  if(brSet.GetContLocs().size() > 1) {
    x->SetLocExpr(brSet.GetFinalExpr());
  } else {
    x->SetLocExpr(node.expr);
  }
  
  //get the locations for the body of the loop
  ContLocSet y = GetOut(node.stmt);
  ContLocSet res;
  if(y.IsEmpty()) {
    res = brSet;
    x->SetTSucc(brSet.GetInitLoc());
  } else {
    //make the initial state of the condition the successor of the body
    for(list<ContLoc*>::iterator i = y.GetFinalLocs().begin();i != y.GetFinalLocs().end();++i) {
      (*i)->SetSucc(brSet.GetInitLoc());
    }
    y.GetFinalLocs().clear();
    y.GetFinalLocs().push_back(x);
    x->SetTSucc(y.GetInitLoc());
    res = brSet;
    res.Append(y);
  }
  
  //handle the break and continue statements
  res.HandleBreaks();
  res.HandleContinues();
  
  //associate this set of locations with the node
  SetOut(&node,res);
}

/*********************************************************************/
void ContLocExtractor::outDoStmt(const DoStmt &node) 
{
  //create the control location for the loop condition
  ContLoc *x = new ContLoc(&Database::procInfos[procName],ContLoc::BRANCH);
  allLocs.insert(x);
  ContLocSet brSet = GetBranchContLocSet(x,node.expr,true);
  
  //set the branch expression
  if(brSet.GetContLocs().size() > 1) {
    x->SetLocExpr(brSet.GetFinalExpr());
  } else {
    x->SetLocExpr(node.expr);
  }
  
  //add the locations for the branch after the body
  ContLocSet res = GetOut(node.stmt);
  res.Concat(brSet);
  
  //set the true successor of the branch condition to be the
  //initial location of the entire statement
  x->SetTSucc(res.GetInitLoc());
  
  //handle the break and continue statements
  res.HandleBreaks();
  res.HandleContinues();
  
  //associate this set of locations with the node
  SetOut(&node,res);
}

/*********************************************************************/
void ContLocExtractor::outForStmt(const ForStmt &node) { assert(false); }

/*********************************************************************/
void ContLocExtractor::outGotoStmt(const GotoStmt &node) 
{
  ContLoc *x = new ContLoc(&Database::procInfos[procName],ContLoc::GOTO);
  allLocs.insert(x);
  x->SetLabel(node.id);
  ContLocSet y = CreateSingletonContLocSet(x,Expr());
  SetOut(&node,y);
}

/*********************************************************************/
void ContLocExtractor::outContinueStmt(const ContinueStmt &node) 
{
  ContLoc *x = new ContLoc(&Database::procInfos[procName],ContLoc::CONTINUE);
  allLocs.insert(x);
  ContLocSet y = CreateSingletonContLocSet(x,Expr());
  SetOut(&node,y);
}

/*********************************************************************/
void ContLocExtractor::outBreakStmt(const BreakStmt &node) 
{
  ContLoc *x = new ContLoc(&Database::procInfos[procName],ContLoc::BREAK);
  allLocs.insert(x);
  ContLocSet y = CreateSingletonContLocSet(x,Expr());
  SetOut(&node,y);
}

/*********************************************************************/
void ContLocExtractor::outReturnStmt(const ReturnStmt &node) 
{
  //create the set for the return statement
  ContLoc *x = new ContLoc(&Database::procInfos[procName],ContLoc::RETURN);
  allLocs.insert(x);
  ContLocSet c = CreateSingletonContLocSet(x,Expr()); 
  
  //create the set for the expression
  ContLocSet d = GetOut(node.expr);
  
  //add the locations for the return after the ones for the
  //expression
  ContLocSet res = d;
  res.Concat(c);
  
  //set the return expression appropriately
  if(d.IsEmpty()) {
    x->SetLocExpr(node.expr);
  } else {
    x->SetLocExpr(d.GetFinalExpr());
  }
  SetOut(&node,res);
}

/*********************************************************************/
//create a singleton control location set. the arguments are the
//single control locations in the set and the final expression.
/*********************************************************************/
ContLocSet ContLocExtractor::CreateSingletonContLocSet(ContLoc *cloc,const Expr &fexpr)
{
  list<ContLoc*> all,ret,lab,br,cnt,gt,fin;
  
  all.push_back(cloc);

  if(cloc->GetLocType() == ContLoc::RETURN) {
    ret.push_back(cloc);
  } else if((cloc->GetLocType() == ContLoc::LABEL_ID) ||
	    (cloc->GetLocType() == ContLoc::LABEL_CASE) ||
	    (cloc->GetLocType() == ContLoc::LABEL_DEFAULT)) {
    lab.push_back(cloc);
    fin.push_back(cloc);
  } else if(cloc->GetLocType() == ContLoc::BREAK) {
    br.push_back(cloc);
  } else if(cloc->GetLocType() == ContLoc::CONTINUE) {
    cnt.push_back(cloc);
  } else if(cloc->GetLocType() == ContLoc::GOTO) {
    gt.push_back(cloc);
  } else {
    fin.push_back(cloc);
  }
  
  return ContLocSet(all,cloc,fin,ret,lab,br,cnt,gt,fexpr);
}

/*********************************************************************/
//create an assignment control location. the arguments are the lhs and
//rhs of the assignment.
/*********************************************************************/
ContLoc *ContLocExtractor::CreateAssignContLoc(const BasicExpr *lhs,const BasicExpr *rhs)
{
  //get the type of the rhs i.e. if it is a normal expression or
  //a function call
  pair<bool,int> a = Util::IsProcCall(rhs);
  int type = (a.first) ? ContLoc::ASSIGN_CALL : ContLoc::ASSIGN_EXPR;

  //create the location
  ContLoc *res = new ContLoc(&Database::procInfos[procName],type);
  res->SetIndCallKey(a.second);
  allLocs.insert(res);
  
  //set the lhs and rhs
  res->SetLocLhs(lhs);
  res->SetLhsAliasKey(lhs->GetPamKey());
  res->SetLocRhs(rhs);
  
  return res;
}

/*********************************************************************/
//create the control locations for a conditional expression. the
//arguments are the choice expression, the final expression if the
//choice expression is true and the final expression if the choice
//expression is false.
/*********************************************************************/
ContLocSet ContLocExtractor::CreateCondExprContLocSet(const BasicExpr *ifExpr,
						      const BasicExpr *thExpr,const BasicExpr *elExpr)
{
  ContLocSet icls = GetOut(ifExpr);
  ContLocSet tcls = GetOut(thExpr);
  ContLocSet ecls = GetOut(elExpr);
  
  //create the if-then-else location set
  ContLoc *c = new ContLoc(&Database::procInfos[procName],ContLoc::BRANCH);
  allLocs.insert(c);
  c->SetLocExpr((icls.IsEmpty()) ? ifExpr : icls.GetFinalExpr());
  ContLocSet d = icls;
  d.Concat(CreateSingletonContLocSet(c,Expr()));

  //get the expression where the final value will be stored
  string tv = Util::NewTempVar();
  Database::procInfos[procName].AddLocal(tv);
  Expr f = ExprManager::GetIdExpr(tv);
  
  //create the locations for the then branch
  ContLoc *g = CreateAssignContLoc(f.GetExpr(),(tcls.IsEmpty()) ? thExpr : tcls.GetFinalExpr().GetExpr());
  ContLocSet h = tcls;
  h.Concat(CreateSingletonContLocSet(g,f));
  
  //create the locations for the else branch
  ContLoc *k = CreateAssignContLoc(f.GetExpr(),(ecls.IsEmpty()) ? elExpr : ecls.GetFinalExpr().GetExpr());
  ContLocSet l = ecls;
  l.Concat(CreateSingletonContLocSet(k,f));

  //link up the branch location with the two successors
  c->SetTSucc(h.GetInitLoc());
  c->SetESucc(l.GetInitLoc());
  d.GetFinalLocs().clear();
  d.Merge(h);
  d.Merge(l);
  
  //set the final expression
  d.SetFinalExpr(f);

  return d;
}

/*********************************************************************/
//handle the goto statements. set their successors appropriately.
/*********************************************************************/
void ContLocExtractor::HandleGotos(ContLocSet &res)
{
  list<ContLoc*> &a = res.GetGotoLocs();
  for(list<ContLoc*>::iterator i = a.begin();i != a.end();++i) {
    string c = (*i)->GetLabel();
    list<ContLoc*> d = res.GetIdLabelLocs();
    bool succFound = false;
    for(list<ContLoc*>::iterator j = d.begin();j != d.end();++j) {
      string f = (*j)->GetLabel();
      if(c == f) {
	(*i)->SetSucc(*j);
	succFound = true;
	break;
      }
    }
    if(!succFound) {
      Util::Error("no successor for goto " + c + "\n");
    }
  }
}

/*********************************************************************/
//create the set of control locations for a branch condition. also
//return the final expression string in the result. the final argument
//indicates whether this branch corresponds to the guard of a loop
//statement.
/*********************************************************************/
ContLocSet ContLocExtractor::GetBranchContLocSet(ContLoc *loc,const BasicExpr *expr,const bool loopBranch)
{
  //get the locations for the branch condition expression
  ContLocSet x = GetOut(expr);
  
  //create the cont loc set for the branch statement
  ContLocSet c = CreateSingletonContLocSet(loc,Expr());
  
  //concatenate the two sets
  ContLocSet res = x;
  res.Concat(c);
  
  //store the final expression if necessary
  if(!x.IsEmpty()) {
    res.SetFinalExpr(x.GetFinalExpr());
  }
  
  //check if this branch could be useful
  CheckBranchUseful(res,expr);
  
  //check if this branch could be fair
  if(loopBranch) CheckBranchFair(res,expr);

  //return the final result
  return res;
}

/*********************************************************************/
//given a set of control locations and an expression make all the
//branch statements in the location useful if the expression is a
//predicate specified by the user. this is only done if automatic
//predicate discovery is not done.
/*********************************************************************/
void ContLocExtractor::CheckBranchUseful(ContLocSet &locs,const Expr &expr)
{
  //check if the expression is important
  set<string> aids = Util::ComputeStrLvalues(expr.GetExpr());
  
  bool flag = false;
  const list<BasicExpr*> &x = Database::procInfos[procName].GetPreds().data;
  for(list<BasicExpr*>::const_iterator i = x.begin();i != x.end();++i) {
    set<string> bids = Util::ComputeStrLvalues(*i);    
    //first we check if the two predicates have the same set
    //of ids. if not they cannot be related.
    if(aids != bids) continue;
    //then we check for semantics relation
    if(ExprManager::EquivalentTo(expr,Expr(*i)) || ExprManager::NegationOf(expr,Expr(*i))) {
      flag = true;
      break;
    }
  }
  
  //if so make all the branches useful
  if(flag) locs.MakeBranchesUseful();
}

/*********************************************************************/
//given a set of control locations and an expression make all the
//branch statements in the location fair if the expression is a fair
//loop specified by the user.
/*********************************************************************/
void ContLocExtractor::CheckBranchFair(ContLocSet &locs,const Expr &expr)
{
  //check if the expression is important
  set<string> aids = Util::ComputeStrLvalues(expr.GetExpr());
  
  bool flag = false;
  const list<BasicExpr*> &x = Database::procInfos[procName].GetFairLoops().data;
  for(list<BasicExpr*>::const_iterator i = x.begin();i != x.end();++i) {
    set<string> bids = Util::ComputeStrLvalues(*i);    
    //first we check if the two predicates have the same set
    //of ids. if not they cannot be related.
    if(aids != bids) continue;
    //then we check for semantics relation
    if(ExprManager::EquivalentTo(expr,Expr(*i)) || ExprManager::NegationOf(expr,Expr(*i))) {
      flag = true;
      break;
    }
  }
  
  //if so make all the branches useful
  if(flag) locs.MakeBranchesFair();
}

/*********************************************************************/
//end of ContLocExtractor.cpp
/*********************************************************************/
