/**************************** CPPHeaderFile ***************************

* FileName [ContLoc.h]

* PackageName [main]

* Synopsis [Header file for ContLoc class.]

* Description [This class encapsulates a control location in a C
* procedure.]

* SeeAlso [ContLoc.cpp]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University.  All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#ifndef __CONTLOC_H__
#define __CONTLOC_H__

namespace magic {

//other classes needed
class ContLoc;
class ImplState;
class LtsInfo;
class ProcInfo;
class Action;
class ProcManager;

/*********************************************************************/
//this class is used to backup important fields of a control location
//that can be restored later. these fields are modified during every
//iteration of the abstract-verify-refine loop and restored at the
//beginning of the next iteration
/*********************************************************************/
class ContLocBackup
{
 private:
  //the successor control locations - there could be atmost two
  //successors normally.
  ContLoc *tSucc;
  ContLoc *eSucc;

  //the successor if this location is a call to a procedure. the
  //successor is the first control location of the called procedure
  ContLoc *callSucc;
  PredSet preds;
  
 public:
  ContLocBackup();
  ~ContLocBackup();
  ContLocBackup(const ContLocBackup &rhs);
  const ContLocBackup &operator = (const ContLocBackup &rhs);

  friend class ContLoc;
};

/*********************************************************************/
//the ContLoc class
/*********************************************************************/
class ContLoc
{
 private:
  //the id of the location is the index of the location in the global
  //vector of currently useful locations. locations are strictly
  //ordered by this id.
  short locId;

  //the key assigned to PAM if this is an indirect function call site
  int indCallKey;

  //the alias key assigned to the lhs of this location
  int lhsAliasKey;
  
  //the set of predicates associated with the control location
  PredSet preds;
  
  //the flag indicating whether this location is useful for the
  //simulation check. all useless locations are removed before
  //actual simulation check begins.
  bool useful;
  
  //flag indicating whether this location corresponds to the branch
  //for a fair loop statement
  bool fairLoop;

  //the pair of associated fairness actions if this is a fair loop
  //locations
  pair<Action,Action> fairLoopActs;

  //the info about the procedure to which the control location belongs
  ProcInfo *procInfo;

  //if this is a call-site then the following field is the name of the
  //procedure whose abstraction is to be inlined here
  string absProc;
  
  //the successor control locations - there could be atmost two
  //successors normally.
  ContLoc *tSucc;
  ContLoc *eSucc;

  //the list of skipped locations between this location and its
  //successors. these locations are skipped because they are useless
  //for the particular predicates being considered. however they have
  //to be taken into account while generating concrete traces through
  //the program.
  list<ContLoc*> tSkipped;
  list<ContLoc*> eSkipped;
  
  //the type of this location
  int locType;
  
  //the expression returned in case this is a return location or the
  //branching expression if this is a branch location or the
  //procedure call expression in case this is a procedure call with
  //no return value
  Expr locExpr;
  
  //the expressions for lhs and rhs in case this is an assignment
  //location
  Expr locLhs;
  Expr locRhs;

  //possible aliases of the lhs in case this is an assignment statement
  set<Expr> lhsAliases;

  //the pointer expression being currently tracked for invalid pointer
  //dereference
  Expr ipdExpr;
  
  //the label string in case this is a labeled control location
  string label;
  
  //the successor if this location is a call to a procedure. the
  //successor is the first control location of the called procedure
  ContLoc *callSucc;

  //the list of skipped successors between this location and its call
  //successor
  list<ContLoc*> callSkipped;
  
  //the set of predecessors required for various dataflow analysis
  //like live variable analysis and relevant predicate computation.
  set<ContLoc*> predecessors;

  //the set of all reachable impl states
  set<const ImplState*> implStates;

  //map from keys to impl states for quick lookup
  map<string,const ImplState*> keyToImpl;

  //map from impl states to true propositions
  map< const ImplState*,set<Expr> > propMap;

  //the map of predicate valuations and spec LTS states to successors
  //and actions. if A is mapped to B then it means that B is the set
  //of successors and actions of the implementation state obtained by
  //pairing this control location with A. also the map to
  //predecessors.
  map< string,map< Action,set<const ImplState*> > > succsAndActions;
  map< string,map< Action,set<const ImplState*> > > predsAndActions;

  //the set of live variables at this location
  set<string> liveVars;
  
  //the kill and gen sets for live variable analysis
  set<string> killSet;
  set<string> genSet;

  //the backup
  ContLocBackup backup;

  void Cleanup();
  void CreateGenSetReturn();
  void CreateGenSetAssignExpr();
  void CreateGenSetAssignCall();
  void CreateGenSetBranch();
  list<ContLoc*> ComputeSuccLocs();

  void UpdatePredsReturn(const pair< vector<Predicate>,vector<Predicate> > &arg,vector<Predicate> &res);
  void UpdatePredsFinal(const pair< vector<Predicate>,vector<Predicate> > &arg,vector<Predicate> &res);
  void UpdatePredsAssignExpr(const pair< vector<Predicate>,vector<Predicate> > &arg,vector<Predicate> &res);
  void UpdatePredsAssignCall(const pair< vector<Predicate>,vector<Predicate> > &arg,vector<Predicate> &res);  
  void UpdatePredsBranch(const pair< vector<Predicate>,vector<Predicate> > &arg,vector<Predicate> &res);
  void MergePredsConjunct(const vector<Predicate> &arg,const Expr &expr,vector<Predicate> &res);

  bool IsAbsCallSite() const;
  
 public:
  //the types of control locations
  static const int BREAK;
  static const int RETURN;
  static const int FINAL;
  static const int LABEL_ID;
  static const int LABEL_CASE;
  static const int LABEL_DEFAULT;
  static const int CONTINUE;
  static const int GOTO;
  static const int ASSIGN_EXPR;
  static const int ASSIGN_CALL;
  static const int BRANCH;

  ContLoc();
  ContLoc(ProcInfo *p,const int t);
  ContLoc(const ContLoc &rhs);
  virtual ~ContLoc();
  const ContLoc &operator = (const ContLoc &rhs);

  string ToString() const;

  void SetLocType(const int l) { locType = l; }
  void SetTSucc(ContLoc *s) { tSucc = s; }
  void SetESucc(ContLoc *s) { eSucc = s; }
  void SetSucc(ContLoc *s) {
    tSucc = (tSucc == NULL) ? s : tSucc;
    eSucc = (eSucc == NULL) ? s : eSucc;
  }
  void SetLocExpr(const Expr &e);
  void SetLocLhs(const Expr &e);
  void SetLocRhs(const Expr &e);
  void SetLhsAliases(const set<Expr> &s);
  void SetIpdExpr(const Expr &i);
  void SetLabel(const string &l) { label = l; }
  void SetUseful(bool u) { useful = u; }
  void SetFairLoop(bool f) { fairLoop = f; }
  void SetFairLoopActs(const pair<Action,Action> &a) { fairLoopActs = a; }
  void SetCallSucc(ContLoc *s) { callSucc = s; }
  void SetLocId(short l) { locId = l; }
  void SetIndCallKey(int i) { indCallKey = i; }
  void SetLhsAliasKey(int l) { lhsAliasKey = l; }
  void SetProcInfo(ProcInfo *p) { procInfo = p; }
  void SetAbsProc(const string &a) { absProc = a; }
  void AddLhsAlias(const Expr &e) { lhsAliases.insert(e); }

  int GetLocType() const { return locType; }
  string GetLabel() const { return label; }
  const Expr &GetLocExpr() const { return locExpr; }
  const Expr &GetLocLhs() const { return locLhs; }
  const Expr &GetLocRhs() const { return locRhs; }
  const set<Expr> &GetLhsAliases() const { return lhsAliases; }
  const Expr &GetIpdExpr() const { return ipdExpr; }
  ContLoc *GetTSucc() { return tSucc; }
  const ContLoc *GetTSucc() const { return tSucc; }
  ContLoc *GetESucc() { return eSucc; }
  const ContLoc *GetESucc() const { return eSucc; }
  list<ContLoc*> &GetTSkipped() { return tSkipped; }
  const list<ContLoc*> &GetTSkipped() const { return tSkipped; }
  list<ContLoc*> &GetESkipped() { return eSkipped; }
  const list<ContLoc*> &GetESkipped() const { return eSkipped; }
  ContLoc *GetCallSucc() { return callSucc; }
  const ContLoc *GetCallSucc() const { return callSucc; }
  list<ContLoc*> &GetCallSkipped() { return callSkipped; }
  const list<ContLoc*> &GetCallSkipped() const { return callSkipped; }
  const set<ContLoc*> &GetPredecessors() const { return predecessors; }
  const set<const ImplState*> &GetImplStates() const { return implStates; }
  PredSet &GetPreds() { return preds; }
  const PredSet &GetPreds() const { return preds; }
  ProcInfo *GetProcInfo() const { return procInfo; }
  const string &GetAbsProc() const { return absProc; }
  short GetLocId() const { return locId; }
  int GetIndCallKey() const { return indCallKey; }
  int GetLhsAliasKey() const { return lhsAliasKey; }
  bool IsUseful() const { return useful; }
  bool IsFairLoop() const { return fairLoop; }
  const pair<Action,Action> &GetFairLoopActs() const { return fairLoopActs; }

  bool IsCallSite(pair< Expr,list<Expr> > &comps) const;
  bool IsSkipLoc() const;
  void AddNamespaceTag(const ProcManager &procManager);
  void InitLiveVarAnalysis(const ProcManager &procManager);
  void MakePred();
  bool UpdateLiveVars();
  set<Expr> GetDerefPtrExprs() const;
  void AddAssignActPreds(const set< pair<Expr,Expr> > &actPreds,const ProcManager &procManager);
  void AddCallActPreds(const set< pair<Expr,Expr> > &actPreds,const ProcManager &procManager);
  void AddRetActPreds(const set<Expr> &actPreds,const ProcManager &procManager);
  bool AddPred(const Predicate &pred,vector<Predicate> &res);
  void UpdatePreds(const pair< vector<Predicate>,vector<Predicate> > &arg,vector<Predicate> &res);
  void ComputeUseful(const ProcManager &procManager);
  void ReplaceSucc(ContLoc *prev,ContLoc *next);
  void ComputeSpecInfos();
  void RestoreSuccsAndActions(FILE *in);
  void BackupSuccsAndActions(FILE *out);
  void SetPropositions(const ImplState *is,const set<Expr> &props);
  void GetPropositions(const string &key,set<Expr> &res);
  bool ContainsSuccsAndActions(const string &key);  
  const map< Action,set<const ImplState*> > &GetSuccsAndActions(const string &key);
  void AddSuccsAndActions(const string &key,const map< Action,set<const ImplState*> > &val);
  void RemSuccsAndActions(const string &key,const map< Action,set<const ImplState*> > &val);
  map< Action,set<const ImplState*> > GetPredsAndActions(const string &key);
  set<const ImplState*> GetImplStates(const PVal &pv);
  set<const ImplState*> GetImplStates(const Expr &pred);
  const ImplState *GetImplState(const PVal &pv,short st,short sp);
  void Backup();
  void Restore();
};

} //namespace magic

#endif //__CONTLOC_H__

/*********************************************************************/
//end of ContLoc.h
/*********************************************************************/
