/******************************** CPPFile *****************************

* FileName [ContLoc.cpp]

* PackageName [main]

* Synopsis [Method definitions of ContLoc class.]

* SeeAlso [ContLoc.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <gmp.h>
#include <string>
#include <list>
#include <set>
#include <map>
#include <vector>
using namespace std;

#include "BigInt.h"
#include "Util.h"
#include "Node.h"
#include "ProcAbs.h"
#include "Action.h"
#include "LtsTrans.h"
#include "FspInfo.h"
#include "ProcInfo.h"
#include "LtsInfo.h"
#include "LtlFormula.h"
#include "LtlManager.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "ImplState.h"
#include "LocalAbsLts.h"
#include "ContLoc.h"
#ifdef MAGIC_FULL
#include "SAT.h"
#endif //MAGIC_FULL
#include "SatSimChecker.h"
#include "ConcCEDag.h"
#include "Component.h"
#include "ProcManager.h"
#define YYSTYPE int
#include "StdcParser.h"
using namespace magic;

/*********************************************************************/
//methods for ContLocBackup class
/*********************************************************************/
ContLocBackup::ContLocBackup() { tSucc = eSucc = callSucc = NULL; }

ContLocBackup::~ContLocBackup() {}

ContLocBackup::ContLocBackup(const ContLocBackup &rhs) { *this = rhs; }

const ContLocBackup &ContLocBackup::operator = (const ContLocBackup &rhs)
{
  tSucc = rhs.tSucc;
  eSucc = rhs.eSucc;
  callSucc = rhs.callSucc;
  preds = rhs.preds;
  return *this;
}

/*********************************************************************/
//the types of control locations
/*********************************************************************/
const int ContLoc::BREAK = 10000;
const int ContLoc::RETURN = 10010;
const int ContLoc::FINAL = 10020;
const int ContLoc::LABEL_ID = 10030;
const int ContLoc::LABEL_CASE = 10040;
const int ContLoc::LABEL_DEFAULT = 10050;
const int ContLoc::CONTINUE = 10060;
const int ContLoc::GOTO = 10070;
const int ContLoc::ASSIGN_EXPR = 10080;
const int ContLoc::ASSIGN_CALL = 10090;
const int ContLoc::BRANCH = 10110;

/*********************************************************************/
//constructors
/*********************************************************************/
ContLoc::ContLoc()
{
  locId = 0;
  indCallKey = -1;
  lhsAliasKey = -1;
  useful = false;
  fairLoop = false;
  procInfo = NULL;
  tSucc = eSucc = NULL;
  locType = 0;
  callSucc = NULL;
}

ContLoc::ContLoc(ProcInfo *p,const int t)
{
  locId = 0;
  indCallKey = -1;
  lhsAliasKey = -1;
  useful = false;
  fairLoop = false;
  procInfo = p;
  tSucc = eSucc = NULL;
  locType = t;
  callSucc = NULL;
}

ContLoc::ContLoc(const ContLoc &rhs)
{
  *this = rhs;
}

ContLoc::~ContLoc()
{
  Cleanup();
}

const ContLoc &ContLoc::operator = (const ContLoc &rhs)
{
  Cleanup();
  locId = rhs.locId;
  indCallKey = rhs.indCallKey;
  lhsAliasKey = rhs.lhsAliasKey;
  preds = rhs.preds;
  useful = rhs.useful;
  fairLoop = rhs.fairLoop;
  fairLoopActs = rhs.fairLoopActs;
  procInfo = rhs.procInfo;
  absProc = rhs.absProc;
  tSucc = rhs.tSucc;
  eSucc = rhs.eSucc;
  tSkipped = rhs.tSkipped;
  eSkipped = rhs.eSkipped;
  locType = rhs.locType;
  locExpr = rhs.locExpr;
  locLhs = rhs.locLhs;
  locRhs = rhs.locRhs;
  lhsAliases = rhs.lhsAliases;
  ipdExpr = rhs.ipdExpr;
  label = rhs.label;
  callSucc = rhs.callSucc;
  callSkipped = rhs.callSkipped;
  predecessors = rhs.predecessors;
  implStates = rhs.implStates;
  keyToImpl = rhs.keyToImpl;
  propMap = rhs.propMap;
  succsAndActions = rhs.succsAndActions;
  predsAndActions = rhs.predsAndActions;
  liveVars = rhs.liveVars;
  killSet = rhs.killSet;
  genSet = rhs.genSet;
  backup = rhs.backup;
  return *this;
}

void ContLoc::Cleanup()
{
  for(set<const ImplState*>::const_iterator i = implStates.begin();i != implStates.end();++i) {
    delete *i;
  }
  implStates.clear();
}

/*********************************************************************/
//get string representation
/*********************************************************************/
string ContLoc::ToString() const
{
  string a;
  
  if(locType == BREAK) a = "break location";
  else if(locType == RETURN) {
    a = "return ( " + locExpr.ToString() + ")";
  }
  else if(locType == FINAL) a = "final location";
  else if(locType == LABEL_ID) a = "label location";
  else if(locType == LABEL_CASE) a = "case location";
  else if(locType == LABEL_DEFAULT) a = "default location";
  else if(locType == CONTINUE) a = "continue location";
  else if(locType == GOTO) a = "goto location";
  else if((locType == ASSIGN_EXPR) || (locType == ASSIGN_CALL)) {
    assert(!locLhs.IsNull() && !locRhs.IsNull());
    a = locLhs.ToString() + "= " + Util::TrimString(locRhs.ToString());
  } else if(locType == BRANCH) {
    assert(!locExpr.IsNull());
    a = "branch ( " + locExpr.ToString() + ")";
  } else {
    assert(false);
    a = "unknown location";
  }

  return a;
}

/*********************************************************************/
//accessors and modifiers
/*********************************************************************/
void ContLoc::SetLocExpr(const Expr &e) { locExpr = e; }

void ContLoc::SetLocLhs(const Expr &e) { locLhs = e; }

void ContLoc::SetLocRhs(const Expr &e) { locRhs = e; }

void ContLoc::SetLhsAliases(const set<Expr> &s) { lhsAliases = s; }

void ContLoc::SetIpdExpr(const Expr &i) { ipdExpr = i; }

/*********************************************************************/
//if this control location is a call-site then return true and also
//the components of the call i.e. the procedure name and the arguments
//to the called procedure in the array supplied as argument. otherwise
//return false.
/*********************************************************************/
bool ContLoc::IsCallSite(pair< Expr,list<Expr> > &comps) const
{
  if(locType == ASSIGN_CALL) {
    ExprManager::GetCalledProcDetails(locRhs,comps);
    return true;
  } else {
    return false;
  }
}

/*********************************************************************/
//returns true iff this is a skip location - break, label, case,
//default, continue, goto or a call location abstracted only by
//useless spec LTSs.
/*********************************************************************/
bool ContLoc::IsSkipLoc() const
{
  if ((locType == BREAK) || (locType == LABEL_ID) ||
      (locType == LABEL_CASE) || (locType == LABEL_DEFAULT) ||
      (locType == CONTINUE) || (locType == GOTO)) {
    return true;
  } else if(locType == ASSIGN_CALL) {
    //get the called procedure expression and name
    pair< Expr,list<Expr> > comps; ExprManager::GetCalledProcDetails(locRhs,comps);
    //if this is a C procedure with special semantics then don't skip
    string procName = Util::TrimString(comps.first.ToString());
    if((procName == Database::EXIT_C_PROC) || (procName == Database::MALLOC_C_PROC) || (procName == Database::FREE_C_PROC)) return false;
    //if this is a call to "assert" and we are checking for assertion
    //failures then don't skip
    if(Database::CHECK_ASSERTION_FAILURE && (procName == Database::ASSERT_C_PROC)) return false;
    //if this involves a pointer dereference and we are checking for
    //invalid pointer dereferences then don't skip
    if(Database::CHECK_INVALID_PTR_DEREF) {
      set<Expr> x = GetDerefPtrExprs();
      if(!x.empty()) return false;
    }
    //if this is an abstracted call-site
    pair< set<string>,set<string> > csi = procInfo->GetCallSiteInfo(comps.first,indCallKey);
    if(csi.first.empty()) {
      //get the set of names of LTSs that abstract this call-site
      set<string> snames;
      for(set<string>::const_iterator i = csi.second.begin();i != csi.second.end();++i) {
	if(Database::procInfos.count(*i) != 0) {
	  const list<ProcAbs> &pabs = Database::procInfos[*i].GetCallerAbs();
	  for(list<ProcAbs>::const_iterator j = pabs.begin();j != pabs.end();++j) {
	    snames.insert(j->GetLtsName());
	  }
	}
      }
      //check if any of the specs are useful - this location is skip
      //iff every spec LTS is useless
      for(set<string>::iterator i = snames.begin();i != snames.end();++i) {
	const LtsInfo &b = Database::GetLtsInfo(*i);
	if(b.IsUseful()) return false;
      }
      return true;
    }
    //otherwise this location is not skip
    else return false;
  } else return false;
}

/*********************************************************************/
//add namespace to this location by prepending a namespace tag to
//every identifier lvalue appearing in any expression in this
//location. the tag is of the form "N::" where "N" is the index of the
//proc manager to which this control location belongs.
/*********************************************************************/
void ContLoc::AddNamespaceTag(const ProcManager &procManager)
{
  list<string> parLocs = procManager.GetParams();
  parLocs.insert(parLocs.end(),procManager.GetLocals().begin(),procManager.GetLocals().end());
  list<string> repList;
  for(list<string>::const_iterator i = parLocs.begin();i != parLocs.end();++i) {
    size_t pos = i->find(':');
    assert(pos < (i->size() - 2));
    repList.push_back(i->substr(pos + 2,i->size() - pos - 2));
  }
  list<Expr> oldList; ExprManager::StringListToExprList(repList,oldList);
  list<Expr> newList; ExprManager::StringListToExprList(parLocs,newList);
  if(locExpr.GetExpr() != NULL) locExpr = ExprManager::Replace(oldList,newList,locExpr);
  if(locLhs.GetExpr() != NULL) locLhs = ExprManager::Replace(oldList,newList,locLhs);
  if(locRhs.GetExpr() != NULL) locRhs = ExprManager::Replace(oldList,newList,locRhs);
  set<Expr> la = lhsAliases;
  lhsAliases.clear();
  for(set<Expr>::const_iterator i = la.begin();i != la.end();++i) {
    lhsAliases.insert(ExprManager::Replace(oldList,newList,*i));
  }
}

/*********************************************************************/
//initialise the live variable analysis at this location
/*********************************************************************/
void ContLoc::InitLiveVarAnalysis(const ProcManager &procManager)
{
  //initialise the live variable set to an empty set
  liveVars.clear();
  
  //create the kill set - all kill sets are empty
  killSet.clear();
  
  if(locType == RETURN) {
    CreateGenSetReturn();
  } else if(locType == ASSIGN_EXPR) {
    CreateGenSetAssignExpr();
  } else if(locType == ASSIGN_CALL) {
    CreateGenSetAssignCall();
  } else if(locType == BRANCH) {
    CreateGenSetBranch();
  } else genSet.clear();

  //for ltl model checking add the variables appearing in propositions
  if(Database::CONF_TYPE == Database::CONF_LTL) {
    const set<Expr> &ltlProps = procManager.GetProps();
    for(set<Expr>::const_iterator i = ltlProps.begin();i != ltlProps.end();++i) {
      set<string> x = Util::ComputeIdLvalues(i->GetExpr());
      genSet.insert(x.begin(),x.end());
    }
  }
}

/*********************************************************************/
//create the genset for a retun control location
/*********************************************************************/
void ContLoc::CreateGenSetReturn()
{
  genSet.clear();
  genSet = Util::ComputeIdLvalues(locExpr.GetExpr());
}

/*********************************************************************/
//create the genset for an assignment location where the rhs is not a
//procedure call
/*********************************************************************/
void ContLoc::CreateGenSetAssignExpr()
{
  //add the lvalues in the lhs and the rhs
  genSet = Util::ComputeIdLvalues(locLhs.GetExpr());
  set<string> lv = Util::ComputeIdLvalues(locRhs.GetExpr());
  genSet.insert(lv.begin(),lv.end());
}

/*********************************************************************/
//create the genset for an assignment location where the rhs is a
//procedure call
/*********************************************************************/
void ContLoc::CreateGenSetAssignCall()
{
  //add the lvalues in the lhs 
  genSet = Util::ComputeIdLvalues(locLhs.GetExpr());
  //get the call details
  pair< Expr,list<Expr> > comps; ExprManager::GetCalledProcDetails(locRhs,comps);
  //add the variables in the arguments and the procedure name in the
  //genset
  list<Expr> &args = comps.second;
  for(list<Expr>::const_iterator i = args.begin();i != args.end();++i) {
    set<string> x = Util::ComputeIdLvalues(i->GetExpr());
    genSet.insert(x.begin(),x.end());
  }
  set<string> x = Util::ComputeIdLvalues(comps.first.GetExpr());
  genSet.insert(x.begin(),x.end());
  //add the variables in the guard predicates
  set< pair<Expr,string> > callAbsInfo = procInfo->GetCallAbsInfo(comps.first,absProc);
  for(set< pair<Expr,string> >::const_iterator i = callAbsInfo.begin();i != callAbsInfo.end();++i) {
    //replace "$1", "$2" etc. with the actual arguments
    set<string> x = Util::ComputeIdLvalues(ExprManager::ReplaceDummyVarsOne(i->first,args).GetExpr());
    genSet.insert(x.begin(),x.end());
  }
}

/*********************************************************************/
//create the genset for a branch location
/*********************************************************************/
void ContLoc::CreateGenSetBranch()
{
  genSet = Util::ComputeIds(locExpr.GetExpr());
}

/*********************************************************************/
//make this control location a predecessor of each of its successor
/*********************************************************************/
void ContLoc::MakePred()
{
  //get the set of successor locations
  list<ContLoc*> x = ComputeSuccLocs();
  
  //make this location a predecessor of each successor
  for(list<ContLoc*>::iterator i = x.begin();i != x.end();++i) {
    (*i)->predecessors.insert(this);
  }
}

/*********************************************************************/
//compute the set of successor locations appropriate for live variable
//analysis
/*********************************************************************/
list<ContLoc*> ContLoc::ComputeSuccLocs()
{
  list<ContLoc*> x;
  if((locType == RETURN) || (locType == FINAL)) {}
  else if((locType == ASSIGN_EXPR) || (locType == ASSIGN_CALL)) {
    x.push_back(tSucc);
  } else if(locType == BRANCH) {
    x.push_back(tSucc);
    x.push_back(eSucc);
  } else assert(false);
  return x;
}

/*********************************************************************/
//update the set of live variables at this location. return true if
//the set is changed by the update and false otherwise.
/*********************************************************************/
bool ContLoc::UpdateLiveVars()
{
  //get the set of successor locations
  list<ContLoc*> x = ComputeSuccLocs();
  
  //collect the live variables at the successors
  set<string> y;
  for(list<ContLoc*>::iterator i = x.begin();i != x.end();++i) {
    set<string> &a = (*i)->liveVars;
    y.insert(a.begin(),a.end());
  }

  //remove the kill set and add the gen set
  for(set<string>::iterator i = killSet.begin();i != killSet.end();++i) {
    y.erase(*i);
  }
  y.insert(genSet.begin(),genSet.end());

  //compute if the set of live variables will be modified
  bool res = false;
  for(set<string>::iterator i = y.begin();i != y.end();++i) {
    if(liveVars.count(*i) == 0) {
      res = true;
      break;
    }
  }
  
  //update the set of live variables
  liveVars = y;
  
  return res;
}

/*********************************************************************/
//get the set of pointer expressions dereferenced at this location
/*********************************************************************/
set<Expr> ContLoc::GetDerefPtrExprs() const
{
  set<Expr> eset;
  if(locType == ContLoc::ASSIGN_EXPR) {
    eset.insert(locLhs);
    eset.insert(locRhs);
  } else if(locType == ContLoc::ASSIGN_CALL) {
    //get the components to the call i.e. the procedure name and the arguments
    pair< Expr,list<Expr> > comps; ExprManager::GetCalledProcDetails(locRhs,comps);
    //get the set of predicates associated with this call
    set< pair<Expr,string> > callAbsInfo =  procInfo->GetCallAbsInfo(comps.first,absProc);
    //do the replacement and add the predicate
    list<Expr> &args = comps.second;
    for(set< pair<Expr,string> >::const_iterator i = callAbsInfo.begin();i != callAbsInfo.end();++i) {
      const LtsInfo &absInfo = Database::GetLtsInfo(i->second);
      const set<Action> &absActs = absInfo.GetActions();
      for(set<Action>::const_iterator j = absActs.begin();j != absActs.end();++j) {      
	if(j->IsAssign()) {
	  eset.insert(ExprManager::ReplaceDummyVarsOne(j->GetLhsExpr(),args));
	}
      }
    }
    eset.insert(locLhs);
    eset.insert(locRhs);
  } else if((locType == ContLoc::BRANCH) || (locType == ContLoc::RETURN)) {
    eset.insert(locExpr);
  } else assert(locType == ContLoc::FINAL);
  set<Expr> res;
  for(set<Expr>::const_iterator i = eset.begin();i != eset.end();++i) {
    set<Expr> x = Util::ComputeExprDerefLvalues(i->GetExpr());
    res.insert(x.begin(),x.end());
  }
  return res;
}

/*********************************************************************/
//add the set of predicates associated with assign actions to an
//ASSIGN_EXPR location. this set of predicates is supplied as the
//first argument.
/*********************************************************************/
void ContLoc::AddAssignActPreds(const set< pair<Expr,Expr> > &actPreds,const ProcManager &procManager)
{
  list<Expr> repList; ExprManager::StringListToExprList(procManager.GetParams(),repList);
  for(set< pair<Expr,Expr> >::const_iterator i = actPreds.begin();i != actPreds.end();++i) {
    Expr newLhs = ExprManager::ReplaceDummyVarsOne(i->first,repList);
    if(ExprManager::LvalueCompatible(locLhs,newLhs) || ExprManager::LvalueCompatible(newLhs,locLhs)) {
      //create the lvalue predicate and add it to this location
      Expr a = ExprManager::GetUnaryExpr(locLhs,'&');
      Expr b = ExprManager::GetUnaryExpr(newLhs,'&');
      Expr lpred = ExprManager::GetBinaryExpr(a,b,MAGIC_EQ_OP);
      vector<Predicate> pvec;
      if(AddPred(Predicate(lpred,Predicate::REQUIRED,pair<const ContLoc*,int>(this,1)),pvec)) {
	Util::Message(3,"predicate " + lpred.ToString() + " added to control location " + ToString() + "\n");
      }
      //create the rvalue predicate and add it to this location
      repList.push_front(locRhs);
      Expr rpred = ExprManager::ReplaceDummyVarsZero(i->second,repList);
      repList.pop_front();
      if(AddPred(Predicate(rpred,Predicate::REQUIRED,pair<const ContLoc*,int>(this,1)),pvec)) {
	Util::Message(3,"predicate " + rpred.ToString() + " added to control location " + ToString() + "\n");
      } 
    }
  }
}

/*********************************************************************/
//add the set of predicates associated with call actions. the argument
//is a map from procedure names to sets of predicates associated with
//calls to those procedures
/*********************************************************************/
void ContLoc::AddCallActPreds(const set< pair<Expr,Expr> > &actPreds,const ProcManager &procManager)
{
  //get the components to the call i.e. the procedure name and the arguments
  pair< Expr,list<Expr> > comps;
  if(locType == ASSIGN_CALL) ExprManager::GetCalledProcDetails(locRhs,comps);
  else Util::Error("trying to get components of non-procedure call : " + ToString() + "\n");  
  //get the set of predicates associated with this call
  set< pair<Expr,string> > callAbsInfo =  procInfo->GetCallAbsInfo(comps.first,absProc);
  //do the replacement and add the predicate
  list<Expr> &args = comps.second;
  for(set< pair<Expr,string> >::const_iterator i = callAbsInfo.begin();i != callAbsInfo.end();++i) {
    //replace "$1", "$2" etc. with the actual arguments
    Expr b = ExprManager::ReplaceDummyVarsOne(i->first,args);
    vector<Predicate> pvec;
    if(AddPred(Predicate(b,Predicate::REQUIRED,pair<const ContLoc*,int>(this,1)),pvec)) {
      Util::Message(3,"predicate " + b.ToString() + " added to control location " + ToString() + "\n");
    }
    //handle assignment actions in the inlined abstract LTSs
    const LtsInfo &absInfo = Database::GetLtsInfo(i->second);
    const set<Action> &absActs = absInfo.GetActions();
    for(set<Action>::const_iterator j = absActs.begin();j != absActs.end();++j) {      
      if(j->IsAssign()) {
	Expr newLhs1 = ExprManager::ReplaceDummyVarsOne(j->GetLhsExpr(),args);
	//add predicates to check for lvalue equality
	list<Expr> repList; ExprManager::StringListToExprList(procManager.GetParams(),repList);
	for(set< pair<Expr,Expr> >::const_iterator k = actPreds.begin();k != actPreds.end();++k) {
	  Expr newLhs2 = ExprManager::ReplaceDummyVarsOne(k->first,repList);
	  if(ExprManager::LvalueCompatible(newLhs1,newLhs2) || ExprManager::LvalueCompatible(newLhs2,newLhs1)) {
	    //create the lvalue predicate and add it to this location
	    Expr a = ExprManager::GetUnaryExpr(newLhs1,'&');
	    Expr b = ExprManager::GetUnaryExpr(newLhs2,'&');
	    Expr lpred = ExprManager::GetBinaryExpr(a,b,MAGIC_EQ_OP);
	    if(AddPred(Predicate(lpred,Predicate::REQUIRED,pair<const ContLoc*,int>(this,1)),pvec)) {
	      Util::Message(3,"predicate " + lpred.ToString() + " added to control location " + ToString() + "\n");
	    } 
	  }
	}
      }
    }
  }

  //add predicates to check for lvalue equality
  list<Expr> repList; ExprManager::StringListToExprList(procManager.GetParams(),repList);
  for(set< pair<Expr,Expr> >::const_iterator i = actPreds.begin();i != actPreds.end();++i) {
    Expr newLhs = ExprManager::ReplaceDummyVarsOne(i->first,repList);
    if(ExprManager::LvalueCompatible(locLhs,newLhs) || ExprManager::LvalueCompatible(newLhs,locLhs)) {
      //create the lvalue predicate and add it to this location
      Expr a = ExprManager::GetUnaryExpr(locLhs,'&');
      Expr b = ExprManager::GetUnaryExpr(newLhs,'&');
      Expr lpred = ExprManager::GetBinaryExpr(a,b,MAGIC_EQ_OP);
      vector<Predicate> pvec;
      if(AddPred(Predicate(lpred,Predicate::REQUIRED,pair<const ContLoc*,int>(this,1)),pvec)) {
	Util::Message(3,"predicate " + lpred.ToString() + " added to control location " + ToString() + "\n");
      } 
    }
  }
}

/*********************************************************************/
//add the set of predicates associated with return actions. this set
//of predicates is supplied as the first argument.
/*********************************************************************/
void ContLoc::AddRetActPreds(const set<Expr> &actPreds,const ProcManager &procManager)
{
  //if this function returns void then no predicates to add
  if(locExpr.IsEmptyExpr()) return;
  //add the predicates after replacing all occurrences of "$0" with
  //the return expression and "$1", "$2" etc. with the parameters of
  //the procedure
  list<Expr> repList; ExprManager::StringListToExprList(procManager.GetParams(),repList);
  repList.push_front(locExpr);
  for(set<Expr>::const_iterator i = actPreds.begin();i != actPreds.end();++i) {
    Expr res = ExprManager::ReplaceDummyVarsZero(*i,repList);
    //create the predicate and add it to this location
    vector<Predicate> pvec;
    if(AddPred(Predicate(res,Predicate::REQUIRED,pair<const ContLoc*,int>(this,1)),pvec)) {
      Util::Message(3,"predicate " + res.ToString() + " added to control location " + ToString() + "\n");
    }
  }
}

/*********************************************************************/
//add a predicate to this control location. return true if the set of
//predicates was updated and false otherwise.
/*********************************************************************/
bool ContLoc::AddPred(const Predicate &pred,vector<Predicate> &res)
{
  //sanitize the predicate
  Predicate spred; pred.Sanitize(this,spred);
  //if using live variables
  if(Database::USE_LIVE_VARS) {
    //first check if the predicate has any live variables as a lvalue
    if(spred.ContainsLvalue(liveVars)) {
      if(preds.AddPred(spred)) {
	res.push_back(spred);
	return true;
      } else return false;
    } else return false;
  } else {
    if(preds.AddPred(spred)) {
      res.push_back(spred);
      return true;
    } else return false;
  }
}

/*********************************************************************/
//update the predicates at this control location. return true if the
//predicates were updated and false otherwise.
/*********************************************************************/
void ContLoc::UpdatePreds(const pair< vector<Predicate>,vector<Predicate> > &arg,vector<Predicate> &res)
{
  //return location
  if(locType == RETURN) UpdatePredsReturn(arg,res);
  //final location
  else if(locType == FINAL) UpdatePredsFinal(arg,res);
  //assign with rhs being not a procedure call
  else if(locType == ASSIGN_EXPR) UpdatePredsAssignExpr(arg,res);
  //assign with rhs being a procedure call
  else if(locType == ASSIGN_CALL) UpdatePredsAssignCall(arg,res);
  //branch statement
  else if(locType == BRANCH) UpdatePredsBranch(arg,res);
  //unknown location
  else assert(false);
}

/*********************************************************************/
//no update required at return control locations. the seeding
//predicates are enough.
/*********************************************************************/
void ContLoc::UpdatePredsReturn(const pair< vector<Predicate>,vector<Predicate> > &arg,vector<Predicate> &res)
{
  assert(arg.first.empty() && arg.second.empty() && res.empty());
}

/*********************************************************************/
//update the predicates at a final control location. return true if
//the predicates were updated and false otherwise.
/*********************************************************************/
void ContLoc::UpdatePredsFinal(const pair< vector<Predicate>,vector<Predicate> > &arg,vector<Predicate> &res)
{
  //the final location does not need any predicates
  assert(arg.first.empty() && arg.second.empty() && res.empty());
}

/*********************************************************************/
//update the predicates at a assign control location with the rhs
//being not a procedure call. return true if the predicates were
//updated and false otherwise.
/*********************************************************************/
void ContLoc::UpdatePredsAssignExpr(const pair< vector<Predicate>,vector<Predicate> > &arg,vector<Predicate> &res)
{
  //add the weakest preconditions of the predicates at the
  //successor location under the assignment
  list<Expr> lhsList,rhsList;
  lhsList.push_back(locLhs);
  rhsList.push_back(locRhs);

  for(vector<Predicate>::const_iterator i = arg.first.begin();i != arg.first.end();++i) {
    const Predicate &succPred = *i;
    //create the weakest precondition. if the predicate is
    //being inferred from a predicate of non-REQUIRED type
    //then add one to the type.
    int newType = (succPred.GetType() == Predicate::REQUIRED) ? Predicate::REQUIRED : (succPred.GetType() + 1);
    Predicate predWP;    
    bool newPred = succPred.ComputeWP(lhsList,rhsList,predWP);
    predWP.SetType(newType);
    //if the computed WP is different then this location is useful
    if(newPred) useful = true;
    if(AddPred(predWP,res)) {
      Util::Message(3,"predicate " + predWP.ToString() + " added to control location " + ToString() + "\n");
    }
  }
}

/*********************************************************************/
//update the predicates at a assign control location with the rhs
//being a procedure call. return true if the predicates were updated
//and false otherwise.
/*********************************************************************/
void ContLoc::UpdatePredsAssignCall(const pair< vector<Predicate>,vector<Predicate> > &arg,vector<Predicate> &res)
{
  //we just inherit the predicates from the successor 
  for(vector<Predicate>::const_iterator i = arg.first.begin();i != arg.first.end();++i) {
    if(AddPred(*i,res)) {
      Util::Message(3,"predicate " + i->ToString() + " added to control location " + ToString() + "\n");
    }
  }
}

/*********************************************************************/
//update the predicates at a branch control location. return true if
//the predicates were updated and false otherwise.
/*********************************************************************/
void ContLoc::UpdatePredsBranch(const pair< vector<Predicate>,vector<Predicate> > &arg,vector<Predicate> &res)
{
  //if no predicates have been added just add the branch
  //condition
  if(Database::INIT_SEEDS_FROM_SPEC && 
     ((Database::CONF_TYPE == Database::CONF_LTL) || 
      (Database::CONF_TYPE == Database::CONF_SEAW) || preds.GetPreds().empty())) {
    //if the spec files contain the initial predicates then
    //check if this branch condition is one of them or the
    //negation of one of them. add the branch condition to
    //this location only if that condition is satisfied.
    Predicate condPred(locExpr,Predicate::TENTATIVE,pair<const ContLoc*,int>(this,1));	    
    if(Database::PREDS_FROM_SPEC) {
      if(useful) {
	if(AddPred(condPred,res)) {
	  Util::Message(3,"predicate " + condPred.ToString() + " added to control location " + ToString() + "\n");
	}
      }
    } else {
      if(AddPred(condPred,res)) {
	Util::Message(3,"predicate " + condPred.ToString() + " added to control location " + ToString() + "\n");
      }
    }
  }
  //merge predicates inferred at successors taking into account branch
  //condition
  MergePredsConjunct(arg.first,locExpr,res);
  Expr nexpr = ExprManager::NegateExpr(locExpr);
  MergePredsConjunct(arg.second,nexpr,res);
}

/*********************************************************************/
//add the conjunction of the predicates of a successor location with
//the given expression.
/*********************************************************************/
void ContLoc::MergePredsConjunct(const vector<Predicate> &arg,const Expr &expr,vector<Predicate> &res)
{
  for(vector<Predicate>::const_iterator i = arg.begin();i != arg.end();++i) {
    Predicate filtered; i->Conjunct(expr,filtered);
    if((filtered.GetSize() != 0) && AddPred(filtered,res)) {
      Util::Message(3,"predicate " + filtered.ToString() + " added to control location " + ToString() + "\n");
    }
  }
}

/*********************************************************************/
//compute if this location is useful or not. this is only applicable
//for ASSIGN_EXPR locations. the useful bit has been partially
//computed during the previous relevant predicate computation
//phase. now we make certain useless locations useful if they can
//cause assign actions.
/*********************************************************************/
void ContLoc::ComputeUseful(const ProcManager &procManager)
{
  if(locType != ASSIGN_EXPR) return;
  else if(Database::NO_USELESS_LOCS) {
    //if we are checking for null-pointer dereferences and this
    //assignment involves a pointer dereference then it is useful
    if(Database::CHECK_INVALID_PTR_DEREF) {
      set<Expr> x = GetDerefPtrExprs();
      if(!x.empty()) {
	useful = true;
	return;
      }
    }
    //check if this assignment can result in an assignment action of
    //interest
    for(PredVal i = preds.GetInitPredValuation();preds.IsValidPredValuation(i);i = preds.GetNextPredValuation(i)) {
      set<Expr> state; preds.ToExpr(i,state);
      set<Action> acts; Database::GetSpecAssignActions(locLhs,locRhs,procManager.GetParams(),state,acts);
      if(!acts.empty()) {
	useful = true;
	return;
      }
    }
  } else {
    useful = true;
  }
}

/*********************************************************************/
//if the first argument is a successor of this locations then replace
//it with the second argument.
/*********************************************************************/
void ContLoc::ReplaceSucc(ContLoc *prev,ContLoc *next)
{
  assert(prev != next);
  if((locType == RETURN) || (locType == FINAL)) {}
  else if(locType == ASSIGN_EXPR) {
    if(tSucc == prev) {
      tSkipped.push_back(tSucc);
      tSucc = next;
    }
  } else if(locType == ASSIGN_CALL) {
    if(tSucc == prev) {
      tSkipped.push_back(tSucc);
      tSucc = next;
    }
    if(callSucc == prev) {
      callSkipped.push_back(callSucc);
      callSucc = next;
    }
  } else if(locType == BRANCH) {
    if(tSucc == prev) {
      tSkipped.push_back(tSucc);
      tSucc = next;
    }
    if(eSucc == prev) {
      eSkipped.push_back(eSucc);
      eSucc = next;
    }
  } else {
    assert(false);
  }
}

/*********************************************************************/
//returns true if this is an abstracted call site
/*********************************************************************/
bool ContLoc::IsAbsCallSite() const
{
  pair< Expr,list<Expr> > comps;
  if(IsCallSite(comps)) {
    assert(!procInfo->GetCallSiteInfo(comps.first,indCallKey).second.empty());
    return true;
  } else return false;
}

/*********************************************************************/
//compute the spec infos corresponding to this control location
/*********************************************************************/
void ContLoc::ComputeSpecInfos()
{
  //if this is an abstracted call-site then each implementation
  //state will also have a LTS state component
  if(IsAbsCallSite()) {
    //get the relevant procedure abstractions
    set<ProcAbs> pabs;  
    if(Database::procInfos.count(absProc) != 0) {
      pabs.insert(Database::procInfos[absProc].GetCallerAbs().begin(),Database::procInfos[absProc].GetCallerAbs().end());
    } else pabs.insert(ProcAbs::GetEpsilonAbstraction());
    //get the relevant spec lts names
    set<string> allSpecs;
    for(set<ProcAbs>::const_iterator i = pabs.begin();i != pabs.end();++i) {
      allSpecs.insert(i->GetLtsName());
    }
    //create the lts infos
    for(set<string>::iterator i = allSpecs.begin();i != allSpecs.end();++i) {
      Database::GetLtsInfo(*i);
    }
  }
}

/*********************************************************************/
//restore the successors and actions of this control location from the
//file whose pointer is passed as argument
/*********************************************************************/
void ContLoc::RestoreSuccsAndActions(FILE *in)
{
  while(true) {
    ContLoc *newLoc = NULL;
    PVal oldPV,newPV;
    short oldSS,oldSI,newSS,newSI,actId;
    oldPV.Read(in);
    fscanf(in,"%hd",&oldSS);
    fscanf(in,"%hd",&oldSI);
    fscanf(in,"%hd",&actId);
    fscanf(in,"%p",&newLoc);
    newPV.Read(in);
    fscanf(in,"%hd",&newSS);
    fscanf(in,"%hd",&newSI);
    PVal dummy = PVal::MaxPVal();
    if((oldPV == dummy) && (oldSS == -1) && (oldSI == -1) && (actId == Action::ACTION_NONE) && 
       (newLoc == 0x0) && (newPV == dummy) && (newSS == -1) && (newSI == -1)) break;
    const ImplState *oldState = GetImplState(oldPV,oldSS,oldSI);
    string oldKey = oldState->GetKey();
    const ImplState *newState = newLoc->GetImplState(newPV,newSS,newSI);
    succsAndActions[oldKey][Action(actId)].insert(newState);
    string newKey = newState->GetKey();
    newLoc->predsAndActions[newKey][Action(actId)].insert(oldState);
  }
}

/*********************************************************************/
//backup the successors and actions of this control location to the
//file whose pointer is passed as argument
/*********************************************************************/
void ContLoc::BackupSuccsAndActions(FILE *out)
{
  for(set<const ImplState*>::const_iterator i = implStates.begin();i != implStates.end();++i) {
    const PVal &oldPV = (*i)->GetPredVal();
    short oldSS = (*i)->GetSpecStateId();
    short oldSI = (*i)->GetSpecInfoId();
    map< Action,set<const ImplState*> > succs;
    (*i)->GetSuccsAndActions(set<Action>(),set<Action>(),succs);
    for(map< Action,set<const ImplState*> >::const_iterator j = succs.begin();j != succs.end();++j) {
      for(set<const ImplState*>::const_iterator k = j->second.begin();k != j->second.end();++k) {
	ContLoc *newLoc = (*k)->GetContLoc();
	const PVal &newPV = (*k)->GetPredVal();
	short newSS = (*k)->GetSpecStateId();
	short newSI = (*k)->GetSpecInfoId();
	short actId = j->first.GetIndex();
	fprintf(out,"<");
	oldPV.Write(out);
	fprintf(out,"> %hd %hd %hd %p <",oldSS,oldSI,actId,newLoc);
	newPV.Write(out);
	fprintf(out,"> %hd %hd ",newSS,newSI);
      }  
    }
  }
  fprintf(out,"<> -1 -1 () 0x0 <> -1 -1 ");
}

/*********************************************************************/
//returns true iff the succs and actions map contains an entry for
//this key
/*********************************************************************/
bool ContLoc::ContainsSuccsAndActions(const string &key)
{
  return (succsAndActions.count(key) != 0);
}

/*********************************************************************/
//set the propositions for a given impl state
/*********************************************************************/
void ContLoc::SetPropositions(const ImplState *is,const set<Expr> &props)
{
  assert(propMap.count(is) == 0);
  propMap[is] = props;
}

/*********************************************************************/
//return the set of propositions true in the implementation state with
//given key
/*********************************************************************/
void ContLoc::GetPropositions(const string &key,set<Expr> &res)
{
  map<string,const ImplState*>::const_iterator i = keyToImpl.find(key);
  assert(i != keyToImpl.end());
  map< const ImplState*,set<Expr> >::const_iterator j = propMap.find(i->second);
  assert(j != propMap.end());
  res = j->second;
}

/*********************************************************************/
//returns the entry for the key in the succs and actions map
/*********************************************************************/
const map< Action,set<const ImplState*> > &ContLoc::GetSuccsAndActions(const string &key)
{
  assert(succsAndActions.count(key) != 0);
  return succsAndActions[key];
}

/*********************************************************************/
//update the successors and actions map by adding new transitions
/*********************************************************************/
void ContLoc::AddSuccsAndActions(const string &key,const map< Action,set<const ImplState*> > &val)
{
  map<string,const ImplState*>::const_iterator i = keyToImpl.find(key);
  assert(i != keyToImpl.end());
  for(map< Action,set<const ImplState*> >::const_iterator j = val.begin();j != val.end();++j) {
    succsAndActions[key][j->first].insert(j->second.begin(),j->second.end());
    for(set<const ImplState*>::const_iterator k = j->second.begin();k != j->second.end();++k) {
      ContLoc *newLoc = (*k)->GetContLoc();
      string newKey = (*k)->GetKey();
      newLoc->predsAndActions[newKey][j->first].insert(i->second);
    }
  }
}

/*********************************************************************/
//update the successors and actions map by removing transitions
/*********************************************************************/
void ContLoc::RemSuccsAndActions(const string &key,const map< Action,set<const ImplState*> > &val)
{
  map<string,const ImplState*>::const_iterator i = keyToImpl.find(key);
  assert(i != keyToImpl.end());
  for(map< Action,set<const ImplState*> >::const_iterator j = val.begin();j != val.end();++j) {
    for(set<const ImplState*>::const_iterator k = j->second.begin();k != j->second.end();++k) {
      succsAndActions[key][j->first].erase(*k);
      if(succsAndActions[key][j->first].empty()) {
	succsAndActions[key].erase(j->first);
      }
      ContLoc *newLoc = (*k)->GetContLoc();
      string newKey = (*k)->GetKey();
      newLoc->predsAndActions[newKey][j->first].erase(i->second);
      if(newLoc->predsAndActions[newKey][j->first].empty()) {
	newLoc->predsAndActions[newKey].erase(j->first);
      }
    }
  }
}

/*********************************************************************/
//returns the entry for the key in the preds and actions map
/*********************************************************************/
map< Action,set<const ImplState*> > ContLoc::GetPredsAndActions(const string &key)
{
  if(predsAndActions.count(key) == 0) return map< Action,set<const ImplState*> >();
  else return predsAndActions[key];
}

/*********************************************************************/
//return the implementation states given a predicate valuation
/*********************************************************************/
set<const ImplState*> ContLoc::GetImplStates(const PVal &pv)
{
  set<const ImplState*> results;
  string key = ImplState(locId,pv,-1,-1).GetKey();
  if(keyToImpl.count(key) == 0) {
    const ImplState *is = new ImplState(locId,pv,-1,-1);
    implStates.insert(is);
    keyToImpl[key] = is;
    results.insert(is);
  } else {
    results.insert(keyToImpl[key]);
  }
  return results;
}

/*********************************************************************/
//returns the initial implementation states matching a predicate
/*********************************************************************/
set<const ImplState*> ContLoc::GetImplStates(const Expr &pred)
{
  set<const ImplState*> results;
  assert(!pred.IsNull());

  //get the ast for the negation of the predicate
  Expr neg = ExprManager::NegateExpr(pred);
  set<Expr> eset; eset.insert(neg);

  vector<PVal> pvals; preds.GetConsistentValuations(pred,pvals);
  for(vector<PVal>::const_iterator i = pvals.begin();i != pvals.end();++i) {
    PredVal pval = preds.PValToPredVal(*i);
    set<Expr> conc; preds.ToExpr(pval,conc);
    if(!ExprManager::ProveImplies(conc,eset)) {
      string key = ImplState(locId,pval.first,-1,-1).GetKey();
      if(keyToImpl.count(key) == 0) {
	const ImplState *is = new ImplState(locId,pval.first,-1,-1);
	implStates.insert(is);
	keyToImpl[key] = is;
	results.insert(is);
      } else {
	results.insert(keyToImpl[key]);
      }
    }
  }
  //all done
  return results;
}

/*********************************************************************/
//return the implementation state given a predicate valuation and a
//LTS state id.
/*********************************************************************/
const ImplState *ContLoc::GetImplState(const PVal &pv,short st,short sp)
{
  string key = ImplState(locId,pv,st,sp).GetKey();
  if(keyToImpl.count(key) == 0) {
    const ImplState *is = new ImplState(locId,pv,st,sp);
    implStates.insert(is);
    keyToImpl[key] = is;
    return is;
  } else {
    return keyToImpl[key];
  }
}

/*********************************************************************/
//backup important fields
/*********************************************************************/
void ContLoc::Backup()
{
  backup.tSucc = tSucc;
  backup.eSucc = eSucc;
  backup.callSucc = callSucc;
}

/*********************************************************************/ 
//restore all important fields of the control locations,cleanup
//implementation states, specification state names, successor
//implementation states and simulation relation from the previous
//iteration and set the useful fields to false
/*********************************************************************/
void ContLoc::Restore()
{
  //restore fields
  tSucc = backup.tSucc;
  eSucc = backup.eSucc;
  callSucc = backup.callSucc;
  for(set<const ImplState*>::const_iterator i = implStates.begin();i != implStates.end();++i) {
    delete *i;
  }
  implStates.clear();
  keyToImpl.clear();
  propMap.clear();
  succsAndActions.clear();
  predsAndActions.clear();
}

/*********************************************************************/
//end of ContLoc.cpp
/*********************************************************************/
