/**************************** CPPHeaderFile ***************************

* FileName [ConcCEDag.h]

* PackageName [main]

* Synopsis [Header file for ConcCEDag class.]

* Description [This class encapsulates a concrete counterexample dag.]

* SeeAlso [ConcCEDag.cpp]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University.  All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#ifndef __CONCCEDAG_H__
#define __CONCCEDAG_H__

namespace magic {

//other classes needed
class L1AbsState;

/*********************************************************************/
//a concrete counterexample dag
/*********************************************************************/
class ConcCEDag
{
 private:
  //label from level-2 to level-1 states
  vector<L1AbsState*> stateLabel;

  //the labeling of nodes with actions
  vector<Action> actLabel;

  //the transition relation
  vector< set<int> > trans;

  //the beginning index of the loop at the end of the
  //counterexample. -1 indicates no loop.
  int loopBegin;

  //the portion before the loop in reverse order. does not include the
  //beginning index of the loop.
  list<size_t> handle;

  //the looped portion. starts with the beginning index of the loop
  //and ends just before the loop beginning index reappears.
  list<size_t> loop;

  //the number of times you have to go through the loop to prove the
  //counterexample invalid. -1 indicates no loop or infinity.
  int loopCount;

  void ComputeTopoWithRoot(const int node,set<int> &visited,vector<int> &topo) const;
  void ComputeStateLabelNoLtsAbs(const Component &component); 
  void DisplayWithRoot(const int node,list<const L1AbsState*> &stateStack,list<Action> &actStack,const Component &component) const;
  void DisplayPath(const list<const L1AbsState*> &stateStack,const list<Action> &actStack,const Component &component) const;

 public:
  static const int STATE_LABEL_FOUND;
  static const int STATE_LABEL_NOT_FOUND;
  static const int STATE_LABEL_UNDECIDED;

  ConcCEDag() { loopBegin = loopCount = -1; }
  ConcCEDag(const ConcCEDag &rhs) { *this = rhs; }
  ~ConcCEDag() { Cleanup(); }

  const ConcCEDag &operator = (const ConcCEDag &rhs);
  bool operator == (const ConcCEDag &rhs) const;
  bool operator < (const ConcCEDag &rhs) const;

  void Cleanup();
  void ComputeTopo(vector<int> &topo) const;
  bool ComputeStateLabelFsp(Component &component);
  int ComputeStateLabelLtl(Component &component);
  bool ComputeStateLabelDeadlock(Component &component);
  void DagToPaths(set<ConcCEDag> &concPaths,const Component &component) const;
  bool NoLoop() const { return trans.back().empty(); }
  void Display(const Component &component) const;

  friend class ProcManager;
  friend class CEDagVerifier;
  friend class PredAbsRefiner;
  friend class Explain;
};

} //namespace magic

#endif //__CONCCEDAG_H__

/*********************************************************************/
//end of ConcCEDag.h
/*********************************************************************/
