/******************************** CPPFile *****************************

* FileName [ConcCEDag.cpp]

* PackageName [main]

* Synopsis [Method definitions of ConcCEDag class.]

* SeeAlso [ConcCEDag.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cassert>
#include <gmp.h>
#include <string>
#include <list>
#include <set>
#include <map>
#include <vector>
#include <fstream>
using namespace std;

#include "BigInt.h"
#include "Util.h"
#include "Node.h"
#include "Action.h"
#include "ActionManager.h"
#include "Database.h"
#include "GlobalAbsLts.h"
#include "GlobalCEDag.h"
#include "Component.h"
#include "Buchi.h"
#include "ConcCEDag.h"
using namespace magic;

/*********************************************************************/
//static members
/*********************************************************************/
const int ConcCEDag::STATE_LABEL_FOUND = 11800;
const int ConcCEDag::STATE_LABEL_NOT_FOUND = 11810;
const int ConcCEDag::STATE_LABEL_UNDECIDED = 11820;

/*********************************************************************/
//operators
/*********************************************************************/
const ConcCEDag &ConcCEDag::operator = (const ConcCEDag &rhs)
{
  Cleanup();
  for(vector<L1AbsState*>::const_iterator i = rhs.stateLabel.begin();i != rhs.stateLabel.end();++i) {
    stateLabel.push_back((*i)->Clone());
  }
  actLabel = rhs.actLabel;
  trans = rhs.trans;
  loopBegin = rhs.loopBegin;
  handle = rhs.handle;
  loop = rhs.loop;
  loopCount = rhs.loopCount;
  return *this;
}

bool ConcCEDag::operator == (const ConcCEDag &rhs) const
{
  if(stateLabel.size() != rhs.stateLabel.size()) return false;
  if(stateLabel.empty()) {
    return (*(rhs.stateLabel.begin()))->VecEq(stateLabel,rhs.stateLabel);
  } else {
    return (*(stateLabel.begin()))->VecEq(stateLabel,rhs.stateLabel);
  }
}

bool ConcCEDag::operator < (const ConcCEDag &rhs) const
{
  if(stateLabel.size() < rhs.stateLabel.size()) return true;
  if(stateLabel.size() > rhs.stateLabel.size()) return false;
  if(stateLabel.empty()) {
    return (*(rhs.stateLabel.begin()))->VecLt(stateLabel,rhs.stateLabel);
  } else {
    return (*(stateLabel.begin()))->VecLt(stateLabel,rhs.stateLabel);
  }
}

/*********************************************************************/
//cleanup
/*********************************************************************/
void ConcCEDag::Cleanup()
{
  for(vector<L1AbsState*>::const_iterator i = stateLabel.begin();i != stateLabel.end();++i) delete *i;
  stateLabel.clear();
  actLabel.clear();
  trans.clear();
}

/*********************************************************************/
//compute reverse topological order
/*********************************************************************/
void ConcCEDag::ComputeTopo(vector<int> &topo) const
{
  set<int> visited;
  ComputeTopoWithRoot(0,visited,topo);
  assert(topo.size() == stateLabel.size());
}

/*********************************************************************/
//compute reverse topological order of a sub-dag
/*********************************************************************/
void ConcCEDag::ComputeTopoWithRoot(const int node,set<int> &visited,vector<int> &topo) const
{
  visited.insert(node);
  for(set<int>::const_iterator i = trans[node].begin();i != trans[node].end();++i) {
    if(visited.count(*i) == 0) ComputeTopoWithRoot(*i,visited,topo);
  }
  topo.push_back(node);
}

/*********************************************************************/
//compute the loc label when no lts abstraction is used
/*********************************************************************/
void ConcCEDag::ComputeStateLabelNoLtsAbs(const Component &component)
{
  //compute reverse topological order
  vector<int> topo; ComputeTopo(topo);
  map<L2AbsState,const L1AbsState*> stateMap;
  component.CreateL2ToL1AbsStateMap(stateMap);
  int ssn = Database::GetSpecStateNum();
  for(vector<int>::const_iterator i = topo.begin();i != topo.end();++i) {
    L2AbsState locState = GlobalAbsLts::CompToAbsStates(GlobalCEDag::stateLabel[*i] / ssn)[component.GetIndex()];
    stateLabel[*i] = stateMap[locState]->Clone();
  }
  //cleanup
  for(map<L2AbsState,const L1AbsState*>::const_iterator i = stateMap.begin();i != stateMap.end();++i) delete i->second;
}

/*********************************************************************/
//compute the control location label
/*********************************************************************/
bool ConcCEDag::ComputeStateLabelFsp(Component &component)
{
  stateLabel.resize(GlobalCEDag::stateLabel.size());  
  trans = GlobalCEDag::trans;
  int ssn = Database::GetSpecStateNum();
  actLabel.resize(GlobalCEDag::actLabel.size());
  for(size_t i = 0;i < GlobalCEDag::actLabel.size();++i) {
    BigInt source = GlobalCEDag::stateLabel[i] / ssn;
    if(trans[i].empty()) {
      set<BigInt> y; GlobalAbsLts::GetSuccsOnAction(source,GlobalCEDag::actLabel[i],y);
      assert(!y.empty());
      BigInt sink = *(y.begin());
      actLabel[i] = GlobalAbsLts::GetCompAct(GlobalCEDag::actLabel[i],source,sink,component.GetIndex());
    } else {
      int x = *(trans[i].begin());
      BigInt sink = GlobalCEDag::stateLabel[x] / ssn;
      actLabel[i] = GlobalAbsLts::GetCompAct(GlobalCEDag::actLabel[i],source,sink,component.GetIndex());
    }
  }
  //if not using LTS abstraction
  if(!Database::USE_LTS_ABSTRACTION) {
    ComputeStateLabelNoLtsAbs(component);
    return true;
  }
  //compute the reverse topological order
  vector<int> topo; ComputeTopo(topo);
  //compute set of possible impl states
  vector<L1StSet> dagToL1Set(GlobalCEDag::stateLabel.size());
  set<Action> absActs; component.GetL2AbsActs(absActs);
  for(vector<int>::const_iterator i = topo.begin();i != topo.end();++i) {
    //intersect successors
    component.GetCompleteL1StSet(dagToL1Set[*i]);
    for(set<int>::const_iterator j = trans[*i].begin();j != trans[*i].end();++j) {
      component.IntersectL1StSets(dagToL1Set[*i],dagToL1Set[*j]);
    }
    //take pre-image if necessary
    const Action &act = actLabel[*i];
    if(absActs.count(act) != 0) {
      dagToL1Set[*i] = component.GetL1AbsPredsOnAction(dagToL1Set[*i],act);
    }
    //project on global ce dag
    L2AbsState locState = GlobalAbsLts::CompToAbsStates(GlobalCEDag::stateLabel[*i] / ssn)[component.GetIndex()];
    dagToL1Set[*i] = component.ProjectL1StSetOnL2AbsState(dagToL1Set[*i],locState);
    //check if no possibilities left
    if(dagToL1Set[*i].Empty()) {
      for(set<int>::const_iterator j = trans[*i].begin();j != trans[*i].end();++j) {
	L2AbsState l2s = GlobalAbsLts::CompToAbsStates(GlobalCEDag::stateLabel[*j] / ssn)[component.GetIndex()];
	component.UpdateL2AbsRefineInfo(l2s,dagToL1Set[*j]);
      }
      return false;
    }
  }  
  //check if some initial impl state can represent the root
  L1StSet initImpls; component.GetL1AbsInit(initImpls);
  component.IntersectL1StSets(initImpls,dagToL1Set[0]);
  if(initImpls.Empty()) {
    L2AbsState l2s = GlobalAbsLts::CompToAbsStates(GlobalCEDag::stateLabel[0] / ssn)[component.GetIndex()];
    component.UpdateL2AbsRefineInfo(l2s,dagToL1Set[0]);
    return false;
  }
  //select a impl state and then cont loc for each node
  vector<L1AbsState*> dagToL1(GlobalCEDag::stateLabel.size());
  dagToL1[0] = component.PickL1AbsState(initImpls);
  for(vector<int>::reverse_iterator i = topo.rbegin();i != topo.rend();++i) {
    stateLabel[*i] = dagToL1[*i]->Clone();
    set<L1AbsState*> iss; iss.insert(dagToL1[*i]);
    L1StSet succs; component.CreateL1StSet(iss,succs);
    const Action &act = actLabel[*i];
    if(absActs.count(act) != 0) {
      succs = component.GetL1AbsSuccsOnAction(succs,act);
    }
    for(set<int>::const_iterator j = trans[*i].begin();j != trans[*i].end();++j) {
      component.IntersectL1StSets(succs,dagToL1Set[*j]);
    }
    L1AbsState *x = component.PickL1AbsState(succs);
    for(set<int>::const_iterator j = trans[*i].begin();j != trans[*i].end();++j) {
      dagToL1[*j] = x->Clone();
    }
    delete x;
  }
  //cleanup
  for(vector<L1AbsState*>::const_iterator i = dagToL1.begin();i != dagToL1.end();++i) delete *i;
  //success
  return true;
}

/*********************************************************************/
//compute the control location label for ltl model checking. if no
//labeling exists, generate the info for refining lts abstraction.
/*********************************************************************/
int ConcCEDag::ComputeStateLabelLtl(Component &component)
{
  //for finite counterexamples
  if(GlobalCEDag::NoLoop()) {
    return ComputeStateLabelFsp(component) ? STATE_LABEL_FOUND : STATE_LABEL_NOT_FOUND;
  }
  //if not using LTS abstraction
  if(!Database::USE_LTS_ABSTRACTION) {
    stateLabel.resize(GlobalCEDag::stateLabel.size());
    trans = GlobalCEDag::trans;
    int ssn = Database::GetSpecStateNum();
    actLabel.resize(GlobalCEDag::actLabel.size());
    for(size_t i = 0;i < GlobalCEDag::actLabel.size();++i) {
      BigInt source = GlobalCEDag::stateLabel[i] / ssn;
      if(trans[i].empty()) {
	set<BigInt> y; GlobalAbsLts::GetSuccsOnAction(source,GlobalCEDag::actLabel[i],y);
	assert(!y.empty());
	BigInt sink = *(y.begin());
	actLabel[i] = GlobalAbsLts::GetCompAct(GlobalCEDag::actLabel[i],source,sink,component.GetIndex());
      } else {
	int x = *(trans[i].begin());
	BigInt sink = GlobalCEDag::stateLabel[x] / ssn;
	actLabel[i] = GlobalAbsLts::GetCompAct(GlobalCEDag::actLabel[i],source,sink,component.GetIndex());
      }
    }
    ComputeStateLabelNoLtsAbs(component);
    return STATE_LABEL_FOUND;
  }
  //map from local states to nodes
  int ssn = Database::GetSpecStateNum();
  map< L2AbsState,set<size_t> > stateToDag;
  for(size_t i = 0;i < GlobalCEDag::stateLabel.size();++i) {
    stateToDag[GlobalAbsLts::CompToAbsStates(GlobalCEDag::stateLabel[i] / ssn)[component.GetIndex()]].insert(i);
  }
  //map from nodes to candidate level-1 abstract state sets
  vector< set<L1AbsState*> > dagToIS(GlobalCEDag::stateLabel.size());
  set<L1AbsState*> ri; component.GetAllL1AbsStates(ri);
  for(set<L1AbsState*>::const_iterator i = ri.begin();i != ri.end();++i) {
    L2AbsState x; component.L1ToL2AbsState(**i,x);
    const set<size_t> &y = stateToDag[x];
    for(set<size_t>::const_iterator j = y.begin();j != y.end();++j) {
      dagToIS[*j].insert((*i)->Clone());
    }
  }
  //cleanup
  for(set<L1AbsState*>::const_iterator i = ri.begin();i != ri.end();++i) delete *i;
  ri.clear();
  //try each candidate loop begin point
  bool refInfo = false;
  loopBegin = *(GlobalCEDag::trans[GlobalCEDag::stateLabel.size() - 1].begin());
  for(set<L1AbsState*>::const_iterator it = dagToIS[loopBegin].begin();it != dagToIS[loopBegin].end();++it) {
    //first simulate the loop backward
    set<Action> absActs; component.GetL2AbsActs(absActs);
    vector<L1StSet> dagToImpls1(GlobalCEDag::stateLabel.size());
    for(size_t j = 0;j < dagToImpls1.size();++j) {
      component.CreateL1StSet(dagToIS[j],dagToImpls1[j]);
    }
    set<L1AbsState*> succImpls;
    succImpls.insert((*it)->Clone());
    component.CreateL1StSet(succImpls,dagToImpls1[loopBegin]);
    for(set<L1AbsState*>::const_iterator i = succImpls.begin();i != succImpls.end();++i) delete *i;
    bool looped = true;    
    for(int currNode = GlobalCEDag::stateLabel.size() - 1;currNode >= loopBegin;--currNode) {
      int nextNode = *(GlobalCEDag::trans[currNode].begin());
      const Action &act = GlobalCEDag::actLabel[currNode];
      L1StSet currSt = dagToImpls1[nextNode];
      if(absActs.count(act) != 0) {
	currSt = component.GetL1AbsPredsOnAction(currSt,act);
      }
      component.IntersectL1StSets(currSt,dagToImpls1[currNode]);
      if(currSt.Empty()) {	
	L2AbsState l2s = GlobalAbsLts::CompToAbsStates(GlobalCEDag::stateLabel[currNode] / ssn)[component.GetIndex()];
	if(component.UpdateL2AbsRefineInfo(l2s,dagToImpls1[currNode])) refInfo = true;
	else {
	  l2s = GlobalAbsLts::CompToAbsStates(GlobalCEDag::stateLabel[nextNode] / ssn)[component.GetIndex()];
	  refInfo = component.UpdateL2AbsRefineInfo(l2s,dagToImpls1[nextNode]) ? true : refInfo;
	}
	looped = false;
	break;
      }
      dagToImpls1[currNode] = currSt;
    }
    if(!looped) continue;
    //if loop is valid check the handle
    vector<L1StSet> dagToImpls2(GlobalCEDag::stateLabel.size());
    for(size_t i = 0;i < dagToImpls2.size();++i) {
      component.CreateL1StSet(dagToIS[i],dagToImpls2[i]);
    }
    for(size_t i = loopBegin;i < dagToImpls1.size();++i) {
      dagToImpls2[i] = dagToImpls1[i];
    }
    bool handled = true;
    for(int currNode = loopBegin - 1;currNode >= 0;--currNode) {
      int nextNode = *(GlobalCEDag::trans[currNode].begin());
      const Action &act = GlobalCEDag::actLabel[currNode];
      L1StSet currSt = dagToImpls2[nextNode];
      if(absActs.count(act) != 0) {
	currSt = component.GetL1AbsPredsOnAction(currSt,act);
      }
      component.IntersectL1StSets(currSt,dagToImpls2[currNode]);
      if(currSt.Empty()) {
	L2AbsState l2s = GlobalAbsLts::CompToAbsStates(GlobalCEDag::stateLabel[currNode] / ssn)[component.GetIndex()];
	if(component.UpdateL2AbsRefineInfo(l2s,dagToImpls2[currNode])) refInfo = true;
	else {
	  l2s = GlobalAbsLts::CompToAbsStates(GlobalCEDag::stateLabel[nextNode] / ssn)[component.GetIndex()];
	  refInfo = component.UpdateL2AbsRefineInfo(l2s,dagToImpls1[nextNode]) ? true : refInfo;
	}
	handled = false;
	break;
      }
      dagToImpls2[currNode] = currSt;
    }
    if(!handled) continue;
    //check if some representative initial state exists
    L1StSet oldInitReps = dagToImpls2[0];
    L1StSet ist; component.GetL1AbsInit(ist);
    component.IntersectL1StSets(dagToImpls2[0],ist);
    if(dagToImpls2[0].Empty()) {
      refInfo = component.UpdateL2AbsRefineInfo(0,oldInitReps) ? true : refInfo;
      continue;
    }
    //create level-1 state labeling of the CE dag
    stateLabel.clear();
    actLabel.clear();
    trans.clear();
    vector<L1AbsState*> ivec;
    ivec.push_back(component.PickL1AbsState(dagToImpls2[0]));
    actLabel.push_back(GlobalCEDag::actLabel[0]);
    trans.push_back(set<int>());
    for(size_t i = 1;i < GlobalCEDag::stateLabel.size();++i) {
      const Action &act = actLabel.back();
      if(absActs.count(act) == 0) {
	ivec.push_back(ivec.back()->Clone());
      } else {
	L1StSet x; component.GetL1AbsSuccsOnAction(*(ivec.back()),act,x);
	component.IntersectL1StSets(x,dagToImpls2[i]);
	ivec.push_back(component.PickL1AbsState(x));
      }
      actLabel.push_back(GlobalCEDag::actLabel[i]);
      trans.back().insert(i);
      trans.push_back(set<int>());
    }
    size_t loopLength = GlobalCEDag::stateLabel.size() - loopBegin;
    int currNode = loopBegin;
    while(true) {
      const Action &act = actLabel.back();
      L1StSet succSet;
      if(absActs.count(act) == 0) {
	set<L1AbsState*> succImpls;
	succImpls.insert(ivec.back()->Clone());
	component.CreateL1StSet(succImpls,succSet);
	for(set<L1AbsState*>::const_iterator i = succImpls.begin();i != succImpls.end();++i) delete *i;
      } else {
	component.GetL1AbsSuccsOnAction(*(ivec.back()),act,succSet);
      }
      if(currNode == loopBegin) {
	looped = false;
	for(size_t i = loopBegin;i < ivec.size();i += loopLength) {
	  if(component.L1AbsContains(succSet,*(ivec[i]))) {
	    trans.back().clear();
	    trans.back().insert(i);
	    looped = true;
	  }
	}
	if(looped) break;
      }
      ivec.push_back(component.PickL1AbsState(succSet));
      actLabel.push_back(GlobalCEDag::actLabel[currNode]);
      trans.back().insert(trans.size());
      trans.push_back(set<int>());
      currNode = (static_cast<size_t>(currNode) == (GlobalCEDag::stateLabel.size() - 1)) ? loopBegin : (currNode + 1);
    }
    for(size_t i = 0;i < ivec.size();++i) stateLabel.push_back(ivec[i]->Clone());
    //cleanup and return
    for(vector<L1AbsState*>::const_iterator i = ivec.begin();i != ivec.end();++i) delete *i;
    for(vector< set<L1AbsState*> >::const_iterator i = dagToIS.begin();i != dagToIS.end();++i) {
      for(set<L1AbsState*>::const_iterator j = i->begin();j != i->end();++j) delete *j;
    }
    component.ClearL2AbsRefineInfo();
    return STATE_LABEL_FOUND;
  }
  //if we reached this point it means no concrete CE was found
  for(vector< set<L1AbsState*> >::const_iterator i = dagToIS.begin();i != dagToIS.end();++i) {
    for(set<L1AbsState*>::const_iterator j = i->begin();j != i->end();++j) delete *j;
  }
  return refInfo ? STATE_LABEL_NOT_FOUND : STATE_LABEL_UNDECIDED;
}

/*********************************************************************/
//compute the control location label
/*********************************************************************/
bool ConcCEDag::ComputeStateLabelDeadlock(Component &component)
{
  stateLabel.resize(GlobalCEDag::stateLabel.size());  
  trans = GlobalCEDag::trans;
  int ssn = Database::GetSpecStateNum();
  actLabel.resize(GlobalCEDag::actLabel.size());
  for(size_t i = 0;i < GlobalCEDag::actLabel.size();++i) {
    BigInt source = GlobalCEDag::stateLabel[i] / ssn;
    assert(!trans[i].empty());
    int x = *(trans[i].begin());
    BigInt sink = GlobalCEDag::stateLabel[x] / ssn;
    actLabel[i] = GlobalAbsLts::GetCompAct(GlobalCEDag::actLabel[i],source,sink,component.GetIndex());
  }
  actLabel.push_back(ActionManager::GetImplEpsilonAction(component.GetIndex()));
  //if not using LTS abstraction
  if(!Database::USE_LTS_ABSTRACTION) {
    ComputeStateLabelNoLtsAbs(component);
    return true;
  }
  //compute the reverse topological order
  vector<int> topo; ComputeTopo(topo);
  //compute set of possible impl states
  vector<L1StSet> dagToL1Set(GlobalCEDag::stateLabel.size());
  set<Action> absActs; component.GetL2AbsActs(absActs);
  vector<int>::const_iterator it = topo.begin();
  L2AbsState lastState = GlobalAbsLts::CompToAbsStates(GlobalCEDag::stateLabel[*it] / ssn)[component.GetIndex()];
  set<Action> refusal; component.GetL2AbsRefusal(lastState,refusal);
  //create level-1 abstract states corresponding to the last level-2
  //abstract state and with the same refusal set
  set<L1AbsState*> allStates,lastStates,sameRef;
  component.GetAllL1AbsStates(allStates);
  for(set<L1AbsState*>::const_iterator i = allStates.begin();i != allStates.end();++i) {
    L2AbsState x; component.L1ToL2AbsState(**i,x);
    if(x == lastState) lastStates.insert((*i)->Clone());
  } 
  for(set<L1AbsState*>::const_iterator i = allStates.begin();i != allStates.end();++i) delete *i;
  assert(!lastStates.empty());
  for(set<L1AbsState*>::const_iterator i = lastStates.begin();i != lastStates.end();++i) {
    if(component.IsL1AbsRefusal(**i,refusal)) sameRef.insert((*i)->Clone());
  }
  //if no such state then split, otherwise continue
  if(sameRef.empty()) {
    for(set<Action>::const_iterator i = refusal.begin();i != refusal.end();++i) {
      set<L1AbsState*> separate;      
      for(set<L1AbsState*>::const_iterator j = lastStates.begin();j != lastStates.end();++j) {
	L1StSet x; component.GetL1AbsSuccsOnAction(**j,*i,x);
	if(!x.Empty()) separate.insert(*j);
      }
      if(!separate.empty()) {
	L1StSet sepSet; component.CreateL1StSet(separate,sepSet);
	component.UpdateL2AbsRefineInfo(lastState,sepSet);
	for(set<L1AbsState*>::const_iterator i = lastStates.begin();i != lastStates.end();++i) delete *i;
	return false;
      }
    }
    assert(false);
  }
  for(set<L1AbsState*>::const_iterator i = lastStates.begin();i != lastStates.end();++i) delete *i;
  component.CreateL1StSet(sameRef,dagToL1Set[*it]);  
  for(set<L1AbsState*>::const_iterator i = sameRef.begin();i != sameRef.end();++i) delete *i;
  for(++it;it != topo.end();++it) {
    //intersect successors
    component.GetCompleteL1StSet(dagToL1Set[*it]);
    for(set<int>::const_iterator j = trans[*it].begin();j != trans[*it].end();++j) {
      component.IntersectL1StSets(dagToL1Set[*it],dagToL1Set[*j]);
    }
    //take pre-image if necessary
    const Action &act = actLabel[*it];
    if(absActs.count(act) != 0) {
      dagToL1Set[*it] = component.GetL1AbsPredsOnAction(dagToL1Set[*it],act);
    }
    //project on global ce dag
    L2AbsState locState = GlobalAbsLts::CompToAbsStates(GlobalCEDag::stateLabel[*it] / ssn)[component.GetIndex()];
    dagToL1Set[*it] = component.ProjectL1StSetOnL2AbsState(dagToL1Set[*it],locState);
    //check if no possibilities left
    if(dagToL1Set[*it].Empty()) {
      for(set<int>::const_iterator j = trans[*it].begin();j != trans[*it].end();++j) {
	L2AbsState l2s = GlobalAbsLts::CompToAbsStates(GlobalCEDag::stateLabel[*j] / ssn)[component.GetIndex()];
	component.UpdateL2AbsRefineInfo(l2s,dagToL1Set[*j]);
      }
      return false;
    }

  }  
  //check if some initial impl state can represent the root
  L1StSet initImpls; component.GetL1AbsInit(initImpls);
  component.IntersectL1StSets(initImpls,dagToL1Set[0]);
  if(initImpls.Empty()) {
    L2AbsState l2s = GlobalAbsLts::CompToAbsStates(GlobalCEDag::stateLabel[0] / ssn)[component.GetIndex()];
    component.UpdateL2AbsRefineInfo(l2s,dagToL1Set[0]);
    return false;
  }
  //select a impl state and then cont loc for each node
  vector<L1AbsState*> dagToL1(GlobalCEDag::stateLabel.size());
  dagToL1[0] = component.PickL1AbsState(initImpls);
  for(vector<int>::reverse_iterator i = topo.rbegin();i != topo.rend();++i) {
    stateLabel[*i] = dagToL1[*i]->Clone();
    if(trans[*i].empty()) continue;
    set<L1AbsState*> iss; iss.insert(dagToL1[*i]);
    L1StSet succs; component.CreateL1StSet(iss,succs);
    const Action &act = actLabel[*i];    
    if(absActs.count(act) != 0) {
      succs = component.GetL1AbsSuccsOnAction(succs,act);
    }
    for(set<int>::const_iterator j = trans[*i].begin();j != trans[*i].end();++j) {
      component.IntersectL1StSets(succs,dagToL1Set[*j]);
    }
    if(!succs.Empty()) {
      L1AbsState *x = component.PickL1AbsState(succs);
      for(set<int>::const_iterator j = trans[*i].begin();j != trans[*i].end();++j) {
	dagToL1[*j] = x->Clone();
      }
      delete x;
    }
  }
  //cleanup
  for(vector<L1AbsState*>::const_iterator i = dagToL1.begin();i != dagToL1.end();++i) delete *i;
  //success
  return true;
}

/*********************************************************************/
//compute the set of concrete paths in the ce dag
/*********************************************************************/
void ConcCEDag::DagToPaths(set<ConcCEDag> &concPaths,const Component &component) const
{
  vector<int> topo; ComputeTopo(topo);
  set<Action> absActs; component.GetL2AbsActs(absActs);
  set<ConcPath> dagToPaths;
  component.L1AbsDagToPaths(stateLabel,actLabel,trans,topo,absActs,dagToPaths);
  for(set<ConcPath>::const_iterator i = dagToPaths.begin();i != dagToPaths.end();++i) {
    //create a dag corresponding to the path
    ConcCEDag x;
    x.stateLabel.resize(i->size());
    x.actLabel.resize(i->size());
    x.trans.resize(i->size());
    int count = 0;
    for(list<ConcEle>::const_iterator j = i->begin();j != i->end();++j,++count) {
      x.stateLabel[count] = j->first->Clone();
      x.actLabel[count] = j->second;
      if(count > 0) x.trans[count - 1].insert(count);
    }    
    concPaths.insert(x);
    //cleanup
    for(list<ConcEle>::const_iterator j = i->begin();j != i->end();++j,++count) {
      delete j->first;
    }
  }
}

/*********************************************************************/
//display to standard output
/*********************************************************************/
void ConcCEDag::Display(const Component &component) const
{
  list<const L1AbsState*> stateStack;
  stateStack.push_back(stateLabel[0]);
  list<Action> actStack;
  if((Database::CONF_TYPE == Database::CONF_SIMUL) || (Database::CONF_TYPE == Database::CONF_REACH) || 
     (Database::CONF_TYPE == Database::CONF_DLOCK)) {
    DisplayWithRoot(0,stateStack,actStack,component);
  } else if(Database::CONF_TYPE == Database::CONF_LTL) {
    if(NoLoop()) DisplayWithRoot(0,stateStack,actStack,component);
    else {
      int i = 0;
      while(actStack.size() <= stateLabel.size()) {
	actStack.push_back(actLabel[i]);
	assert(trans[i].size() == 1);
	i = *(trans[i].begin());
	stateStack.push_back(stateLabel[i]);
      }
      DisplayPath(stateStack,actStack,component);
    }
  } else assert(false);
}

/*********************************************************************/
//display the subtree with given root
/*********************************************************************/
void ConcCEDag::DisplayWithRoot(const int node,list<const L1AbsState*> &stateStack,list<Action> &actStack,const Component &component) const
{
  const Action &act = actLabel[node];
  set<Action> absActs; component.GetL2AbsActs(absActs);
  if(absActs.count(act) == 0) {
    if(trans[node].empty()) {
      DisplayPath(stateStack,actStack,component);
    } else {
      for(set<int>::const_iterator i = trans[node].begin();i != trans[node].end();++i) {
	DisplayWithRoot(*i,stateStack,actStack,component);
      }
    }
  } else {
    actStack.push_back(act);
    if(trans[node].empty()) {
      DisplayPath(stateStack,actStack,component);
    } else {
      for(set<int>::const_iterator i = trans[node].begin();i != trans[node].end();++i) {
	stateStack.push_back(stateLabel[*i]);
	DisplayWithRoot(*i,stateStack,actStack,component);
	stateStack.pop_back();
      }
    }
    actStack.pop_back();
  }
}

/*********************************************************************/
//display the current path
/*********************************************************************/
void ConcCEDag::DisplayPath(const list<const L1AbsState*> &stateStack,const list<Action> &actStack,const Component &component) const
{	
  list<const L1AbsState*>::const_iterator i = stateStack.begin();
  set<Action> absActs; component.GetL2AbsActs(absActs);
  list<Action>::const_iterator j = actStack.begin();  
  ofstream ceFile;
  if(!Database::CE_FILE.empty()) ceFile.open(Database::CE_FILE.c_str());
  bool stutter = false;
  Util::Message(2,"*******************************************\n");
  while((i != stateStack.end()) || (j != actStack.end())) {
    stutter = (j != actStack.end()) && (j->IsComplement() || (absActs.count(*j) == 0));
    if(i != stateStack.end()) {
      list<const L1AbsState*>::const_iterator k = i; ++k;
      const L1AbsState *next = (k == stateStack.end()) ? NULL : (*k);
      string stStr; component.L1AbsStateToString(**i,next,stutter,stStr);
      Util::Message(2,stStr + "\n");    
      ++i;
    }
    if(j != actStack.end()) {		
      if((Database::CE_DISPLAY_TYPE == Database::CE_LOC_ACT) || (Database::CE_DISPLAY_TYPE == Database::CE_LOC_CONS_ACT)) {
	Util::Message(2,"############ " + j->ToString() + " ############\n");
	if(!Database::CE_FILE.empty()) ceFile<< j->ToString() <<endl;
      } else Util::Message(2,"########################\n");
      stutter = (absActs.count(*j) == 0);
      ++j;
    }
  }
  if(!Database::CE_FILE.empty()) ceFile.close();
  Util::Message(2,"*******************************************\n");
}

/*********************************************************************/
//end of ConcCEDag.cpp
/*********************************************************************/
