/**************************** CPPHeaderFile ***************************

* FileName [Component.h]

* PackageName [main]

* Synopsis [Header file for Component class.]

* Description [This class encapsulates a component of the
* implementation we are trying to analyse. It is an abstract class
* that gets instantiated differently depending on the nature of the
* implementation. Currently the types of implementations supported are
* C programs and PACC processes.]

* SeeAlso [ProcManager.h PaccAPI.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University.  All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#ifndef __COMPONENT_H__
#define __COMPONENT_H__

namespace magic {

/*********************************************************************/
//an abstract class representing a level-1 abstract state
/*********************************************************************/
class L1AbsState
{
 public:
  virtual ~L1AbsState() {}
  virtual L1AbsState *Clone() const = 0;
  virtual void Cleanup() = 0;
  virtual bool SetEq(const set<L1AbsState*> &lhs,const set<L1AbsState*> &rhs) const = 0;
  virtual bool SetLt(const set<L1AbsState*> &lhs,const set<L1AbsState*> &rhs) const = 0;
  virtual bool SetContains(const set<L1AbsState*> &lhs,const L1AbsState &rhs) const = 0;
  virtual void SetIntersect(set<L1AbsState*> &lhs,const set<L1AbsState*> &rhs) const = 0;
  virtual bool VecEq(const vector<L1AbsState*> &lhs,const vector<L1AbsState*> &rhs) const = 0;
  virtual bool VecLt(const vector<L1AbsState*> &lhs,const vector<L1AbsState*> &rhs) const = 0;
};

//we use an integer to represent both level-1 and level-2 abstract
//states
typedef pair<L1AbsState*,Action> ConcEle;
typedef list<ConcEle> ConcPath;
typedef int L2AbsState;
class L1StSet;

/*********************************************************************/
//an abstract class representing a component. any implementation is
//assumed to be composed of a set of components executing in parallel.
/*********************************************************************/
class Component
{
 protected:
  //the index of the component
  size_t index;

  //the name of the component
  string name;

 public:
  virtual ~Component() {}

  size_t GetIndex() const { return index; }
  const string &GetName() const { return name; }

  //initialise
  virtual bool Initialise() = 0;

  //cleanup
  virtual void Cleanup() = 0;

  //the level-1 abstract initial states
  virtual void GetL1AbsInit(L1StSet &res) const = 0;

  //the level-2 abstract initial states
  virtual void GetL2AbsInit(set<L2AbsState> &res) = 0;

  //level-2 abstract initial states from previous iteration. needed
  //for action-guided refinement.
  virtual void GetPrevL2AbsInit(set<L2AbsState> &res) = 0;
  
  //number of level-1 abstract states
  virtual int GetL1AbsStateNum() = 0;

  //get the number of level-2 abstract states
  virtual int GetL2AbsStateNum() = 0;

  //get the successors of a level-2 abstract state on an action
  virtual void GetL2AbsSuccsOnAction(const L2AbsState &state,const Action &act,set<L2AbsState> &res) = 0;

  //check if the given transition is a valid level-2 abstract transition
  virtual bool IsL2AbsTrans(const L2AbsState &from,const Action &act,const L2AbsState &to) = 0;

  //get the predecessors of a level-2 abstract state on an action
  virtual void GetL2AbsPredsOnAction(const L2AbsState &state,const Action &act,set<L2AbsState> &res) = 0;

  //check if the argument action is a level-2 abstract action
  virtual bool IsL2AbsAct(const Action &act) const = 0;

  //get the set of level-2 abstract actions - either all actions or
  //only actions enabled from a given state or only incoming actions
  //into a given state
  virtual void GetL2AbsActs(set<Action> &res) const = 0;
  virtual void GetL2AbsActsOut(const L2AbsState &state,set<Action> &res) const = 0;
  virtual void GetL2AbsActsIn(const L2AbsState &state,set<Action> &res) const = 0;

  //get the number of level-2 abstract states in the previous
  //iteration
  virtual int GetPrevL2AbsStateNum() = 0;

  //get the set of level-2 abstract states whose successors might have
  //changed due to action-guided abstraction refinement
  virtual void GetL2AbsSuccsChanged(set<L2AbsState> &res) = 0;

  //get the successors of a level-2 abstract state on an action from
  //previous iteration
  virtual void GetPrevL2AbsSuccsOnAction(const L2AbsState &state,const Action &act,set<L2AbsState> &res) = 0;

  //get the set of propositions labeling a level-2 abstract state
  virtual void GetL2AbsProps(const L2AbsState &state,set<Expr> &res) = 0;

  //return true if the two sets of level-2 abstract states correspond
  //to different sets of level-1 abstract states
  virtual bool DistinctL2AbsStates(const set<L2AbsState> &arg1,const set<L2AbsState> &arg2) = 0;

  //return the complete set of all level-1 abstract states
  virtual void GetCompleteL1StSet(L1StSet &res) const = 0;

  //return the intersection of two sets of level-1 abstract
  //states. change the first set to return the result.
  virtual void IntersectL1StSets(L1StSet &lhs,const L1StSet &rhs) const = 0;

  //return the set of predecessors of a set of level-1 abstract states
  virtual L1StSet GetL1AbsPredsOnAction(const L1StSet &states,const Action &act) const = 0;

  //return the successors of a level-1 abstract state
  virtual void GetL1AbsSuccsOnAction(const L1AbsState &state,const Action &act,L1StSet &res) const = 0;
  virtual L1StSet GetL1AbsSuccsOnAction(const L1StSet &states,const Action &act) const = 0;

  //project a set of level-1 abstract states on a level-2 abstract state
  virtual L1StSet ProjectL1StSetOnL2AbsState(const L1StSet &one,const L2AbsState &two) const = 0;

  //compute the map from level-2 to level-1 abstract states. this only
  //makes sense when we are using no action-guided abstraction. the
  //pointers in the result are to be dallocated by the caller
  virtual void CreateL2ToL1AbsStateMap(map<L2AbsState,const L1AbsState*> &res) const = 0;

  //pick an arbitrary level-1 abstract state from a set
  virtual L1AbsState *PickL1AbsState(const L1StSet &arg) const = 0;
  
  //check if a set of level-1 abstract states contains a level-1
  //abstract state
  virtual bool L1AbsContains(const L1StSet &arg1,const L1AbsState &arg2) const = 0;

  //get the set of all level-1 abstract states
  virtual void GetAllL1AbsStates(set<L1AbsState*> &res) const = 0;

  //get the level-2 abstract state corresponding to a level-1 abstract state
  virtual void L1ToL2AbsState(const L1AbsState &arg,L2AbsState &res) const = 0;

  //create an L1StSet object corresponding to a set of level-1
  //abstract states
  virtual void CreateL1StSet(const set<L1AbsState*> &arg,L1StSet &res) const = 0;

  //reset all info for doing action-guided abstraction refinement
  virtual void ClearL2AbsRefineInfo() = 0;

  //update info for action-guided abstraction refinement given a
  //level-2 abstract state to be split and the action on which it has
  //to be split
  virtual bool UpdateL2AbsRefineInfo(const L2AbsState &l2s,const L1StSet &l1s) = 0;

  //get a string representation of a level-1 abstract state and a set
  //of level-1 abstract states
  virtual void L1AbsStateToString(const L1AbsState &arg,string &res) const = 0;
  virtual void L1StSetToString(const L1StSet &arg,string &res) const = 0;

  //get a string representation of a level-1 abstract state given the
  //next state in a path and whether there is stuttering
  virtual void L1AbsStateToString(const L1AbsState &curr,const L1AbsState *next,const bool stutter,string &res) const = 0;

  //convert a dag to the corresponding set of paths
  virtual void L1AbsDagToPaths(const vector<L1AbsState*> &stateLabel,const vector<Action> &actLabel,const vector< set<int> > &trans,const vector<int> &topo,const set<Action> &absActs,set<ConcPath> &res) const = 0;

  //get the set of fair loop actions
  virtual void GetFairLoopActs(set< pair<Action,Action> > &res) const = 0;

  //return true if the expressions represents a concrete global
  //variable
  virtual bool IsConcGlobalVar(const Expr &arg) const = 0;

  //return the refusal set of a level-2 abstract state
  virtual void GetL2AbsRefusal(const L2AbsState &arg,set<Action> &res) const = 0;

  //check if the second argument is the refusal set of the first argument
  virtual bool IsL1AbsRefusal(const L1AbsState &state,const set<Action> &refusal) const = 0;

  //output an FSP process equivalent to this model
  virtual void OutputFsp() const = 0;
};

/*********************************************************************/
//a set of level-1 abstract states
/*********************************************************************/
class L1StSet
{
 private:
  bool all;
  set<L1AbsState*> ele;

 public:
  L1StSet() { all = false; }  
  L1StSet(const bool a,const set<L1AbsState*> &e) {
    all = a;
    for(set<L1AbsState*>::const_iterator i = e.begin();i != e.end();++i) {
      ele.insert((*i)->Clone());
    }
  }
  ~L1StSet() { Cleanup(); }
  L1StSet(const L1StSet &rhs) { *this = rhs; }
  const L1StSet &operator = (const L1StSet &rhs) {
    Cleanup();
    all = rhs.all;
    for(set<L1AbsState*>::const_iterator i = rhs.ele.begin();i != rhs.ele.end();++i) {
      ele.insert((*i)->Clone());
    }
    return *this;
  }
  bool operator == (const L1StSet &rhs) const {
    if(all != rhs.all) return false;
    if((ele.empty() && rhs.ele.empty())) return true;
    if(ele.size() != rhs.ele.size()) return false;
    if(ele.empty()) {
      return (*(rhs.ele.begin()))->SetEq(ele,rhs.ele);
    } else {
      return (*(ele.begin()))->SetEq(ele,rhs.ele);
    }
  }
  bool operator < (const L1StSet &rhs) const {
    if(all < rhs.all) return true;
    if(all > rhs.all) return false;
    if((ele.empty() && rhs.ele.empty())) return false;
    if(ele.size() < rhs.ele.size()) return true;
    if(ele.size() > rhs.ele.size()) return false;
    if(ele.empty()) {
      return (*(rhs.ele.begin()))->SetLt(ele,rhs.ele);
    } else {
      return (*(ele.begin()))->SetLt(ele,rhs.ele);
    }
  }
  void Cleanup() {
    for(set<L1AbsState*>::const_iterator i = ele.begin();i != ele.end();++i) delete *i;
    ele.clear();
  }
  bool GetAll() const { return all; }
  const set<L1AbsState*> &GetEle() const { return ele; }
  static L1StSet GetComplete() {
    L1StSet res;
    res.all = true;
    return res;
  }
  void Intersect(const L1StSet &rhs) {
    if(all) *this = rhs;
    else {
      if(!ele.empty() && !rhs.all) {
	(*(ele.begin()))->SetIntersect(ele,rhs.ele);
      }
    }
  }
  bool Empty() const { return ((!all) && (ele.empty())); }
};

} //namespace magic

#endif //__COMPONENT_H__

/*********************************************************************/
//end of Component.h
/*********************************************************************/
