/******************************** CPPFile *****************************

* FileName [CEGen.cpp]

* PackageName [main]

* Synopsis [Method definitions of CEGen class.]

* SeeAlso [CEGen.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cassert>
#include <gmp.h>
#include <string>
#include <list>
#include <set>
#include <vector>
#include <map>
using namespace std;

#include "BigInt.h"
#include "Util.h"
#include "Timer.h"
#include "Statistics.h"
#include "Node.h"
#include "Action.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "ContLoc.h"
#include "LocalAbsLts.h"
#include "GlobalCEDag.h"
#include "ConcCEDag.h"
#include "Component.h"
#include "ProcManager.h"
#include "CEDagVerifier.h"
#include "CEGen.h"
using namespace magic;

/*********************************************************************/
//do all the stuff
/*********************************************************************/
void CEGen::Run()
{
  //we cannot generate counterexample for simulation or error state
  //reachability unless we are using the horn sat solver
  if(((Database::CONF_TYPE == Database::CONF_SIMUL) || (Database::CONF_TYPE == Database::CONF_REACH)) && 
     (Database::SAT_SOLVER_USED != Database::SAT_SOLVER_HORN_AI)) {
    return;
  }

  Statistics::cegTimer.Start();
  GlobalCEDag::Initialize();
  GlobalCEDag::Compute();    
  Statistics::cegTimer.Stop();
  
  //if checking simulation of ERROR state reachability
  if((Database::CONF_TYPE == Database::CONF_SIMUL) || (Database::CONF_TYPE == Database::CONF_REACH)) {
    RunFsp();
  }
  //if checking LTL formula
  else if(Database::CONF_TYPE == Database::CONF_LTL) RunLtl();
  //if checking deadlock
  else if(Database::CONF_TYPE == Database::CONF_DLOCK) RunDeadlock();
  //if checking SE-AW formula
  else if(Database::CONF_TYPE == Database::CONF_SEAW) RunSeaw();
  //impossible
  else assert(false);
}

/*********************************************************************/
//the top-level routine when we are checking for simulation or ERROR
//state reachability
/*********************************************************************/
void CEGen::RunFsp()
{
  //generate CE for each component
  for(size_t i = 0;i < Database::components.size();++i) {
    if(Database::CHECK_CE_VALIDITY) {
      int res = CEDagVerifier(*(Database::components[i]),true).Run();
      if(res == CEDagVerifier::CE_VALID) Util::Message(2,"NOTE: this is a valid counterexample ...\n");
      else Util::Message(2,"NOTE: this is a spurious counterexample ...\n");
    } else {
      ConcCEDag concDag;
      Statistics::cevTimer.Start();
      assert(concDag.ComputeStateLabelFsp(*(Database::components[i])));
      ProcManager &procManager = static_cast<ProcManager&>(*(Database::components[i]));
      procManager.RestoreSilentTrans(concDag);
      Statistics::cevTimer.Stop();
      concDag.Display(*(Database::components[i]));
    }
  }
}

/*********************************************************************/
//the top-level routine when we are checking for LTL property
/*********************************************************************/
void CEGen::RunLtl()
{
  //generate CE for each component
  for(size_t i = 0;i < Database::components.size();++i) {
    if(Database::CHECK_CE_VALIDITY) {
      int res = CEDagVerifier(*(Database::components[i]),true).Run();
      if(res == CEDagVerifier::CE_VALID) Util::Message(2,"NOTE: this is a valid counterexample ...\n");
      else Util::Message(2,"NOTE: this is a spurious counterexample ...\n");
    } else {
      ConcCEDag concDag;
      Statistics::cevTimer.Start();
      assert(concDag.ComputeStateLabelLtl(*(Database::components[i])));
      ProcManager &procManager = static_cast<ProcManager&>(*(Database::components[i]));
      procManager.RestoreSilentTrans(concDag);
      Statistics::cevTimer.Stop();
      concDag.Display(*(Database::components[i]));
    } 
  }
}

/*********************************************************************/
//the top-level routine when we are checking for deadlock
/*********************************************************************/
void CEGen::RunDeadlock()
{
  //generate CE for each component
  for(size_t i = 0;i < Database::components.size();++i) {
    if(Database::CHECK_CE_VALIDITY) {
      int res = CEDagVerifier(*(Database::components[i]),true).Run();
      if(res == CEDagVerifier::CE_VALID) Util::Message(2,"NOTE: this is a valid counterexample ...\n");
      else Util::Message(2,"NOTE: this is a spurious counterexample ...\n");
    } else {
      ConcCEDag concDag;
      Statistics::cevTimer.Start();
      assert(concDag.ComputeStateLabelDeadlock(*(Database::components[i])));
      ProcManager &procManager = static_cast<ProcManager&>(*(Database::components[i]));
      procManager.RestoreSilentTrans(concDag);
      Statistics::cevTimer.Stop();
      concDag.Display(*(Database::components[i]));
    }
  }
}

/*********************************************************************/
//the top-level routine when we are checking for SE-AW formula
/*********************************************************************/
void CEGen::RunSeaw()
{
}

/*********************************************************************/
//end of CEGen.cpp
/*********************************************************************/
