/******************************** CPPFile *****************************

* FileName [CEDagVerifier.cpp]

* PackageName [main]

* Synopsis [Method definitions of CEDagVerifier class.]

* SeeAlso [CEDagVerifier.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <gmp.h>
#include <string>
#include <map>
#include <set>
#include <list>
#include <vector>
#include <stack>
#include <typeinfo>
using namespace std;

#include "BigInt.h"
#include "Util.h"
#include "Node.h"
#include "LtsInfo.h"
#include "Action.h"
#include "LtsTrans.h"
#include "ActionManager.h"
#include "ProcInfo.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "ContLoc.h"
#include "ImplState.h"
#include "LocalAbsLts.h"
#include "GlobalAbsLts.h"
#include "GlobalCEDag.h"
#include "Component.h"
#include "ProcManager.h"
#include "ConcCEDag.h"
#include "Explain.h"
#include "CEDagVerifier.h"
#define YYSTYPE int
#include "StdcParser.h"
using namespace magic;

/*********************************************************************/
//define static fields
/*********************************************************************/
const int CEDagVerifier::CE_VALID = 11600;
const int CEDagVerifier::CE_INVALID_DAG = 11610;
const int CEDagVerifier::CE_UNDECIDED_DAG = 11620;
const int CEDagVerifier::CE_INVALID_PATH = 11630;
const int CEDagVerifier::CE_UNDECIDED_PATH = 11640;

/*********************************************************************/
//verify the projection of the global CE Dag on this procedure. if the
//projection is invalid refine it and return CE_INVALID_DAG. otherwise
//check the paths in the projection. if some path is invalid add it to
//the set of spurious paths and return CE_INVALID_PATH. if every path
//is valid return CE_VALID.
/*********************************************************************/
int CEDagVerifier::Run()
{
  //if checking simulation of ERROR state reachability
  if((Database::CONF_TYPE == Database::CONF_SIMUL) || (Database::CONF_TYPE == Database::CONF_REACH)) {
    return RunFsp();
  }
  //if checking LTL formula
  else if(Database::CONF_TYPE == Database::CONF_LTL) {
    return RunLtl();
  }
  //if checking deadlock
  else if(Database::CONF_TYPE == Database::CONF_DLOCK) {
    return RunDeadlock();
  }
  //impossible
  else assert(false);
}

/*********************************************************************/
//the top-level routine when we are checking for simulation or ERROR
//state reachability
/*********************************************************************/
int CEDagVerifier::RunFsp()
{
  //if the implementation is a C program
  //initialise the concrete ce dag
  ConcCEDag concDag;
  if(!concDag.ComputeStateLabelFsp(component)) {
    assert(Database::IMPL_TYPE == Database::IMPL_C);
    return CE_INVALID_DAG;
  }
  //reintroduce skipped silent transitions into the counterexample
  ProcManager &procManager = static_cast<ProcManager&>(component);
  procManager.RestoreSilentTrans(concDag);
  //check out the concrete dag
  spuriousCE = false;
  if(Database::CHECK_CE_BRANCHING) VerifyConcreteDag(concDag);
  else {
    set<ConcCEDag> concPaths; concDag.DagToPaths(concPaths,component);
    for(set<ConcCEDag>::const_iterator i = concPaths.begin();i != concPaths.end();++i) {
      VerifyConcreteDag(*i);
    }
    if(!spuriousCE) VerifyConcreteDag(concDag);
  }
  if(!spuriousCE || displayCE) concDag.Display(component);
  //if this is a real counterexample, explain it
  if (Database::EXPLAIN_CE && (!spuriousCE)) {
    Explain(concDag,component).Run ();
  }
  return spuriousCE ? CE_INVALID_PATH : CE_VALID;
}

/*********************************************************************/
//update the pre or postcondition of a predicate set after a statement
/*********************************************************************/
void CEDagVerifier::UpdatePrePost(const set<ContLoc*> &seeds,PredSet &preds,const int node,
				  const ConcCEDag &concDag,
				  map< int,pair<Action,Expr> > &inlinedAssignMap,
				  const bool pre) const
{
  set<Expr> expSet;
  const vector<Predicate> &pvec = preds.GetPreds();
  for(vector<Predicate>::const_iterator i = pvec.begin();i != pvec.end();++i) {
    const vector<Expr> &x = i->GetValToExpr();
    expSet.insert(x.begin(),x.end());
  }
  set<ContLoc*> dummyExp;
  UpdatePrePost(seeds,dummyExp,expSet,node,concDag,inlinedAssignMap,pre,2);
  preds.Clear();
  const ContLoc *loc = static_cast<const CL1AbsState*>(concDag.stateLabel[node])->GetContLoc();
  for(set<Expr>::const_iterator i = expSet.begin();i != expSet.end();++i) {
    Predicate p(*i,Predicate::REQUIRED,pair<const ContLoc*,int>(loc,1));
    preds.AddPred(p);
  }
}

/*********************************************************************/
//update the pre or postcondition after a statement
/*********************************************************************/
void CEDagVerifier::UpdatePrePost(const set<ContLoc*> &seeds,set<ContLoc*> &expLocs,
				  set<Expr> &expSet,const int node,const ConcCEDag &concDag,
				  map< int,pair<Action,Expr> > &inlinedAssignMap,
				  const bool pre,const int type) const
{  
  assert(Database::IMPL_TYPE == Database::IMPL_C);
  ProcManager &procManager = static_cast<ProcManager&>(component);
  const Action &act = concDag.actLabel[node];
  if(procManager.absLts.GetActs().count(act) != 0) {
    //if this is the last node corresponding to the deadlocked state
    if((Database::CONF_TYPE == Database::CONF_DLOCK) && concDag.trans[node].empty()) return;
    const ContLoc *loc = static_cast<const CL1AbsState*>(concDag.stateLabel[node])->GetContLoc();
    int locType = loc->GetLocType();
    //for assignment locations
    if(locType == ContLoc::ASSIGN_EXPR) {
      list<Expr> elist; ExprManager::StringListToExprList(procManager.GetParams(),elist);
      elist.push_front(loc->GetLocRhs());
      pair<Expr,Expr> comps = act.GetConcreteAssignExprs(elist);
      if(pre) {
	expSet = UpdateWP(expSet,loc->GetLocLhs(),loc->GetLocRhs());
	expSet.insert(comps.second);
      } else {
	expSet.insert(comps.second);
	expSet = UpdateSP(expSet,loc->GetLocLhs(),loc->GetLocRhs());
      }
      /*
	expSet = pre ? UpdateWP(expSet,loc->GetLocLhs(),loc->GetLocRhs()) : 
	UpdateSP(expSet,loc->GetLocLhs(),loc->GetLocRhs());
	expSet.insert(actExp.begin(),actExp.end());
      */
      //check if the assignment corresponds to a predicate inferred as
      //a result of some branch X. in that case add X as a possible
      //explanation. update the cache if required.
      if(type == 0) {
	Expr cond = ExprManager::GetBinaryExpr(loc->GetLocLhs(),loc->GetLocRhs(),MAGIC_EQ_OP);
	if(procManager.assignToBranch.count(cond) == 0) {      
	  for(map< ContLoc*,map<ContLoc*,PredSet> >::const_iterator j = procManager.locToPreds.begin();j != procManager.locToPreds.end();++j) {
	    bool locFlag = true;
	    for(map<ContLoc*,PredSet>::const_iterator k = j->second.begin();locFlag && (k != j->second.end());++k) {
	      const vector<Predicate> &preds = k->second.GetPreds();
	      for(vector<Predicate>::const_iterator l = preds.begin();locFlag && (l != preds.end());++l) {
		const vector<Expr> &vte = l->GetValToExpr();
		for(vector<Expr>::const_iterator m = vte.begin();m != vte.end();++m) {
		  if(ExprManager::EquivalentTo(cond,*m) || ExprManager::NegationOf(cond,*m)) { 
		    procManager.assignToBranch[cond].insert(j->first);
		    locFlag = false;
		    break;
		  }
		}
	      }
	    }
	  }
	}
	for(set<ContLoc*>::const_iterator j = procManager.assignToBranch[cond].begin();j != procManager.assignToBranch[cond].end();++j) {
	  expLocs.insert(*j);
	}
      }
    }
    //for branch locations
    else if(locType == ContLoc::BRANCH) {
      ContLoc *cloc = (ContLoc*)(loc);
      if(type == 0) {
	if(!concDag.trans[node].empty()) {
	  int succNode = *(concDag.trans[node].begin());
	  const ContLoc *succLoc = static_cast<const CL1AbsState*>(concDag.stateLabel[succNode])->GetContLoc();
	  if((loc->GetTSkipped().empty() && (succLoc == loc->GetTSucc())) ||
	     (!loc->GetTSkipped().empty() && (succLoc == loc->GetTSkipped().front()))) {
	    expSet.insert(loc->GetLocExpr());
	  } else if((loc->GetESkipped().empty() && (succLoc == loc->GetESucc())) ||
		    (!loc->GetESkipped().empty() && (succLoc == loc->GetESkipped().front()))) {
	    expSet.insert(ExprManager::NegateExpr(loc->GetLocExpr()));
	  } else assert(false);
	  expLocs.insert(cloc);
	}
      } else if(type == 1) {
	if(seeds.count(cloc) != 0) {
	  if(!concDag.trans[node].empty()) {
	    int succNode = *(concDag.trans[node].begin());
	    const ContLoc *succLoc = static_cast<const CL1AbsState*>(concDag.stateLabel[succNode])->GetContLoc();
	    if((loc->GetTSkipped().empty() && (succLoc == loc->GetTSucc())) ||
	       (!loc->GetTSkipped().empty() && (succLoc == loc->GetTSkipped().front()))) {
	      expSet.insert(loc->GetLocExpr());
	    } else if((loc->GetESkipped().empty() && (succLoc == loc->GetESucc())) ||
		      (!loc->GetESkipped().empty() && (succLoc == loc->GetESkipped().front()))) {
	      expSet.insert(ExprManager::NegateExpr(loc->GetLocExpr()));
	    } else assert(false);
	  }
	}
      } else {
	assert(type == 2);
	if(seeds.count((ContLoc*)(loc)) != 0) {
	  expSet.insert(loc->GetLocExpr());
	}
      }
    }
    //for call locations
    else if(locType == ContLoc::ASSIGN_CALL) {
      assert(loc->GetCallSucc() == NULL);
      char sid = static_cast<const CL1AbsState*>(concDag.stateLabel[node])->GetSpecInfoId();
      if((sid == -1) && (!concDag.trans[node].empty())) {
	int succNode = *(concDag.trans[node].begin());
	char succSid = static_cast<const CL1AbsState*>(concDag.stateLabel[succNode])->GetSpecInfoId();
	//assert(succSid != -1);
	if(succSid != -1) {
	  const LtsInfo &li = Database::ltsInfos[succSid];
	  expSet.insert(loc->GetProcInfo()->GetCallAbsLtsGuard(loc->GetLocRhs(),loc->GetAbsProc(),li.GetName()));
	}
      }
      if(act.IsRecv()) {
	if(!act.GetStore().empty()) {	  
	  pair< Expr,list<Expr> > comps; ExprManager::GetCalledProcDetails(loc->GetLocRhs(),comps);
	  list<Expr> lhsList,dummyList;
	  for(list<Expr>::const_iterator i = act.GetStore().begin();i != act.GetStore().end();++i) {
	    lhsList.push_back(ExprManager::ReplaceDummyVarsOne(*i,comps.second));
	    dummyList.push_back(ExprManager::GetIdExpr(Util::NewTempVar()));
	  }
	  if(pre) {
	    expSet = UpdateWP(expSet,lhsList,dummyList);
	  } else {
	    expSet = UpdateSP(expSet,lhsList,dummyList);
	  }
	}
      }
      if(act.IsComplement()) {
	if(!act.GetMessage().empty()) {
	  Action compAct; bool foundComp = false;
	  const CL1AbsState *las = static_cast<const CL1AbsState*>(concDag.stateLabel[node]);
	  LtsInfo &specInfo = Database::ltsInfos[las->GetSpecInfoId()];
	  list<LtsTrans> outTrans; specInfo.GetOutTrans(specInfo.GetIdState(las->GetSpecStateId()),outTrans);
	  for(list<LtsTrans>::const_iterator i = outTrans.begin();i != outTrans.end();++i) {
	    if(i->GetAction().IsRecv() && (i->GetAction().GetChannel() == act.GetChannel())) {
	      compAct = i->GetAction();
	      foundComp = true;
	      break;
	    }
	  }
	  assert(foundComp);
	  pair< Expr,list<Expr> > comps; ExprManager::GetCalledProcDetails(loc->GetLocRhs(),comps);
	  list<Expr> lhsList;
	  for(list<Expr>::const_iterator i = compAct.GetStore().begin();i != compAct.GetStore().end();++i) {
	    lhsList.push_back(ExprManager::ReplaceDummyVarsOne(*i,comps.second));
	  }
	  list<Expr> rhsList;
	  for(list<Expr>::const_iterator i = act.GetMessage().begin();i != act.GetMessage().end();++i) {
	    const BasicExpr *rexpr = i->GetExpr();
	    assert(typeid(*rexpr) == typeid(BinaryExpr));
	    const BinaryExpr *rbexpr = static_cast<const BinaryExpr*>(rexpr);
	    assert(rbexpr->op == MAGIC_EQ_OP);
	    string tv = Util::NewTempVar();
	    Expr tvExpr = ExprManager::GetIdExpr(tv);
	    comps.second.push_front(tvExpr);
	    rhsList.push_back(ExprManager::ReplaceDummyVarsZero(rbexpr->rhs,comps.second));
	    comps.second.pop_front();
	  }
	  if(pre) {
	    expSet = UpdateWP(expSet,lhsList,rhsList);
	  } else {
	    expSet = UpdateSP(expSet,lhsList,rhsList);
	  }
	}
      }
      if(act.IsAssign()) {
	bool done = false;
	const BasicExpr *rexpr = act.GetRhsExpr().GetExpr();
	if(typeid(*rexpr) == typeid(BinaryExpr)) {
	  const BinaryExpr *rbexpr = static_cast<const BinaryExpr*>(rexpr);
	  if(rbexpr->op == MAGIC_EQ_OP) {
	    list<Expr> elist; ExprManager::StringListToExprList(procManager.GetParams(),elist);
	    string tv = Util::NewTempVar();
	    Expr tvExpr = ExprManager::GetIdExpr(tv);
	    elist.push_front(tvExpr);
	    pair<Expr,Expr> comps = act.GetConcreteAssignExprs(elist);
	    if(pre) {
	      expSet = UpdateWP(expSet,comps.first,rbexpr->rhs);
	    } else {
	      expSet = UpdateSP(expSet,comps.first,rbexpr->rhs);
	    }
	    done = true;
	  }
	}
	if(!done) {	  
	  if(inlinedAssignMap.count(node) == 0) {
	    string tv = Util::NewTempVar();
	    Expr tvExpr = ExprManager::GetIdExpr(tv);
	    list<Expr> elist; ExprManager::StringListToExprList(procManager.GetParams(),elist);
	    elist.push_front(tvExpr);
	    pair<Expr,Expr> comps = act.GetConcreteAssignExprs(elist);
	    inlinedAssignMap[node].first = Action(comps.first,comps.second);
	    inlinedAssignMap[node].second = tvExpr;
	  }
	  const Action &newAct = inlinedAssignMap[node].first;
	  const Expr &dummy = inlinedAssignMap[node].second;
	  if(pre) {
	    expSet = UpdateWP(expSet,newAct.GetLhsExpr(),dummy);
	    expSet.insert(newAct.GetRhsExpr());
	  } else {
	    expSet.insert(newAct.GetRhsExpr());
	    expSet = UpdateSP(expSet,newAct.GetLhsExpr(),dummy);
	  }
	}
      }
    }
    //for return locations
    else if(locType == ContLoc::RETURN) {      
      if(act.IsDefaultRet()) {
	set<Expr> actExprs; Database::GetSpecRetActExprs(actExprs);
	for(set<Expr>::const_iterator i = actExprs.begin();i != actExprs.end();++i) {
	  if(!i->IsEmptyExpr()) {
	    list<Expr> x; x.push_back(loc->GetLocExpr());
	    expSet.insert(ExprManager::NegateExpr(ExprManager::ReplaceDummyVarsZero(*i,x)));
	  }
	}
      } else {
	const Expr &retExpr = act.GetRetExpr();
	if(!retExpr.IsEmptyExpr()) {
	  list<Expr> x; x.push_back(loc->GetLocExpr());
	  expSet.insert(ExprManager::ReplaceDummyVarsZero(retExpr,x));
	}
      }
    }
    //impossible
    else assert(locType == ContLoc::FINAL);
  }
}

/*********************************************************************/
//check the validity of a concrete ce dag. if the dag is spurious add
//it to the global collection.
/*********************************************************************/
void CEDagVerifier::VerifyConcreteDag(const ConcCEDag &concDag)
{
  assert(Database::IMPL_TYPE == Database::IMPL_C);
  ProcManager &procManager = static_cast<ProcManager&>(component);
  //the set of branch locations that can explain this path's
  //infeasibility
  set<ContLoc*> expLocs;
  
  //compute the weakest precondition at each node of the graph in
  //reverse topological order
  vector< set<Expr> > dagToWP(concDag.stateLabel.size());
  vector<int> topo; concDag.ComputeTopo(topo);
  for(vector<int>::const_iterator i = topo.begin();i != topo.end();++i) {
    for(set<int>::const_iterator j = concDag.trans[*i].begin();j != concDag.trans[*i].end();++j) {
      dagToWP[*i].insert(dagToWP[*j].begin(),dagToWP[*j].end());
    }
    map< int,pair<Action,Expr> > inlinedAssignMap;
    set<ContLoc*> dummySeeds;
    UpdatePrePost(dummySeeds,expLocs,dagToWP[*i],*i,concDag,inlinedAssignMap,true,0);
    //set<Expr> cons = static_cast<const CL1AbsState*>(concDag.stateLabel[*i])->GetCons();
    //dagToWP[*i].insert(cons.begin(),cons.end());
  }
  
  //add the constraint for the initial state
  dagToWP[0].insert(procManager.guard);
  
  //check for spuriousness
  if(ExprManager::ProveImplies(dagToWP[0],set<Expr>())) {
    Util::Message(3,"This is a spurious path ....\n");
    spuriousCE = true;
    //add the path to the set of spurious paths and update the set of
    //maps if necessary
    if(procManager.dagToLocs.count(concDag) == 0) {
      //procManager.dagToLocs[dag] = procManager.branchLocs;
      procManager.dagToLocs[concDag] = expLocs;
    }
  } 
}

/*********************************************************************/
//given a set of expressions and an assignment return the set of
//strongest postconditions of the first argument w.r.t. to the second
/*********************************************************************/
set<Expr> CEDagVerifier::UpdateSP(const set<Expr> &old,const Expr &lhs,const Expr &rhs) const
{
  list<Expr>lhsList; lhsList.push_back(lhs);
  list<Expr>rhsList; rhsList.push_back(rhs);
  return UpdateSP(old,lhsList,rhsList);
}

/*********************************************************************/
//given a set of expressions and a parallel assignment return the set
//of strongest postconditions of the first argument w.r.t. to the
//second
/*********************************************************************/
set<Expr> CEDagVerifier::UpdateSP(const set<Expr> &old,const list<Expr> &lhsList,const list<Expr> &rhsList) const
{
  //the list of skolem constant
  list<Expr> skolemList;
  for(size_t i = 0;i < lhsList.size();++i) skolemList.push_back(ExprManager::GetIdExpr(Util::NewTempVar()));
  //update the expressions
  set<Expr> res;
  for(set<Expr>::const_iterator i = old.begin();i != old.end();++i) {
    res.insert(ExprManager::ComputeWP(lhsList,skolemList,*i));
  }
  //add skolem constraints
  list<Expr>::const_iterator lit = lhsList.begin();
  list<Expr>::const_iterator rit = rhsList.begin();
  list<Expr>::const_iterator sit = skolemList.begin();
  for(;lit != lhsList.end();++lit,++rit,++sit) {
    Expr x = ExprManager::ComputeWP(*lit,*sit,*rit);
    Expr y = ExprManager::GetBinaryExpr(*lit,x,MAGIC_EQ_OP);
    res.insert(y);
  }
  //all done
  return res;
}

/*********************************************************************/
//given a set of expressions and an assignment return the set of
//weakest preconditions of the first argument w.r.t. to the second
/*********************************************************************/
set<Expr> CEDagVerifier::UpdateWP(const set<Expr> &old,const Expr &lhs,const Expr &rhs) const
{
  list<Expr>lhsList; lhsList.push_back(lhs);
  list<Expr>rhsList; rhsList.push_back(rhs);
  return UpdateWP(old,lhsList,rhsList);
}

/*********************************************************************/
//given a set of expressions and a parallel assignment return the set
//of weakest preconditions of the first argument w.r.t. to the second
/*********************************************************************/
set<Expr> CEDagVerifier::UpdateWP(const set<Expr> &old,const list<Expr> &lhsList,const list<Expr> &rhsList) const
{
  set<Expr> res;
  for(set<Expr>::const_iterator i = old.begin();i != old.end();++i) {
    res.insert(ExprManager::ComputeWP(lhsList,rhsList,*i));
  }
  return res;
}

/*********************************************************************/
//the top-level routine when we are checking for LTL property
/*********************************************************************/
int CEDagVerifier::RunLtl()
{
  ConcCEDag concDag;
  int res = concDag.ComputeStateLabelLtl(component);
  if(res == ConcCEDag::STATE_LABEL_NOT_FOUND) {
    assert(Database::IMPL_TYPE == Database::IMPL_C);
    return CE_INVALID_DAG;
  } else if(res == ConcCEDag::STATE_LABEL_UNDECIDED) {
    assert(Database::IMPL_TYPE == Database::IMPL_C);
    return CE_UNDECIDED_DAG;
  }
  if(Database::IMPL_TYPE == Database::IMPL_C) {
    ProcManager &procManager = static_cast<ProcManager&>(component);
    procManager.RestoreSilentTrans(concDag);
    //for finite counterexamples
    if(concDag.NoLoop()) {
      spuriousCE = false;
      VerifyConcreteDag(concDag);
      if(!spuriousCE || displayCE) concDag.Display(component);
      if (Database::EXPLAIN_CE && (!spuriousCE)) {
	Explain(concDag,component).Run ();
      }
      return spuriousCE ? CE_INVALID_PATH : CE_VALID;
    }    
    //for infinite counterexamples - initialise the simulation relation
    map< size_t,set<Expr> > simRel;
    list<size_t> order;
    concDag.loopBegin = 0;
    while(true) {
      order.push_back(concDag.loopBegin);
      simRel[concDag.loopBegin] = set<Expr>();
      if(concDag.loopBegin == 0) simRel[concDag.loopBegin].insert(procManager.guard);
      int old = concDag.loopBegin;
      concDag.loopBegin = *(concDag.trans[concDag.loopBegin].begin());
      if(concDag.loopBegin <= old) break;
    }
    concDag.handle.clear();
    list<size_t>::const_iterator it = order.begin();
    while((*it) != static_cast<size_t>(concDag.loopBegin)) {
      concDag.handle.push_front(*it);
      ++it;
    }
    concDag.loop.clear();
    while(it != order.end()) {
      concDag.loop.push_front(*it);
      ++it;
    }
    //compute the maximal simulation relation
    set<ContLoc*> expLocs;
    int loopCount = 0;
    map< int,pair<Action,Expr> > inlinedAssignMap;
    while(true) {
      ++loopCount;
      bool changed = false;
      for(list<size_t>::const_iterator i = concDag.loop.begin();i != concDag.loop.end();++i) {
	//update simulation relation
	size_t succ = *(concDag.trans[*i].begin());
	set<Expr> succSim = simRel[succ];
	set<ContLoc*> dummySeeds;
	UpdatePrePost(dummySeeds,expLocs,succSim,*i,concDag,inlinedAssignMap,true,0);
	set<Expr> cons = static_cast<const CL1AbsState*>(concDag.stateLabel[*i])->GetCons();
	succSim.insert(cons.begin(),cons.end());
	if((*i) == static_cast<size_t>(concDag.loopBegin)) {
	  //check if simulation relation has been updated
	  set<Expr> x; x.insert(ExprManager::ConjunctExprSet(succSim));
	  if(!ExprManager::ProveImplies(simRel[*i],x)) {
	    if(loopCount > (Database::MAX_PRED_INFER_LOOP + 1)) {
	      Util::Message(3,"This is an undecided path ....\n");
	      return CE_UNDECIDED_PATH;	      
	    }
	    changed = true;
	  }
	}
	simRel[*i].insert(succSim.begin(),succSim.end());
	if((*i) == static_cast<size_t>(concDag.loopBegin)) {
	  //check if simulation relation does not exist
	  if(ExprManager::ProveImplies(simRel[*i],set<Expr>())) {
	    Util::Message(3,"This is a spurious path ....\n");
	    concDag.loopCount = loopCount;
	    if(procManager.dagToLocs.count(concDag) == 0) {
	      //procManager.dagToLocs[dag] = procManager.branchLocs;
	      procManager.dagToLocs[concDag] = expLocs;
	    }
	    if(displayCE) concDag.Display(procManager);
	    return CE_INVALID_PATH;
	  }
	}
      }
      if(!changed) break;
    }
    for(list<size_t>::const_iterator i = concDag.handle.begin();i != concDag.handle.end();++i) {
      size_t succ = *(concDag.trans[*i].begin());
      set<Expr> succSim = simRel[succ];
      set<ContLoc*> dummySeeds;
      UpdatePrePost(dummySeeds,expLocs,succSim,*i,concDag,inlinedAssignMap,true,0);
      set<Expr> cons = static_cast<const CL1AbsState*>(concDag.stateLabel[*i])->GetCons();
      succSim.insert(cons.begin(),cons.end());
      simRel[*i].insert(succSim.begin(),succSim.end());
    }
    if(ExprManager::ProveImplies(simRel[0],set<Expr>())) {
      Util::Message(3,"This is a spurious path ....\n");
      if(procManager.dagToLocs.count(concDag) == 0) {
	//procManager.dagToLocs[dag] = procManager.branchLocs;
	procManager.dagToLocs[concDag] = expLocs;
      }
      if(displayCE) concDag.Display(procManager);
      return CE_INVALID_PATH;
    } else {
      concDag.Display(procManager);
      if (Database::EXPLAIN_CE)
	Explain(concDag,component).Run ();
      return CE_VALID;
    }
  } else if(Database::IMPL_TYPE == Database::IMPL_PACC) {
    concDag.Display(component);
    return CE_VALID;
  } else assert(false);
}

/*********************************************************************/
//the top-level routine when we are checking for deadlock
/*********************************************************************/
int CEDagVerifier::RunDeadlock()
{
  //if the implementation is a C program
  //initialise the concrete ce dag
  ConcCEDag concDag;
  if(!concDag.ComputeStateLabelDeadlock(component)) {
    assert(Database::IMPL_TYPE == Database::IMPL_C);
    return CE_INVALID_DAG;
  }
  //reintroduce skipped silent transitions into the counterexample
  ProcManager &procManager = static_cast<ProcManager&>(component);
  procManager.RestoreSilentTrans(concDag);
  //check out the concrete dag
  spuriousCE = false;
  if(Database::CHECK_CE_BRANCHING) VerifyConcreteDag(concDag);
  else {
    set<ConcCEDag> concPaths; concDag.DagToPaths(concPaths,component);
    for(set<ConcCEDag>::const_iterator i = concPaths.begin();i != concPaths.end();++i) {
      VerifyConcreteDag(*i);
    }
    if(!spuriousCE) VerifyConcreteDag(concDag);
  }
  if(!spuriousCE || displayCE) concDag.Display(component);

  return spuriousCE ? CE_INVALID_PATH : CE_VALID;
}

/*********************************************************************/
//end of CEDagVerifier.cpp
/*********************************************************************/
