/******************************** CPPFile *****************************

* FileName [Buchi.cpp]

* PackageName [main]

* Synopsis [Method definitions of Buchi class.]

* SeeAlso [Buchi.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <sys/types.h>
#include <unistd.h>
#include <string>
#include <list>
#include <set>
#include <map>
#include <vector>
using namespace std;

#include "Util.h"
#include "Timer.h"
#include "Statistics.h"
#include "Node.h"
#include "Action.h"
#include "Database.h"
#include "LtlFormula.h"
#include "LtlManager.h"
#include "Buchi.h"
using namespace magic;

/*********************************************************************/
//define static members
/*********************************************************************/
const int Buchi::LTL_TAUTOLOGY = 9000;
const int Buchi::LTL_UNSATISFIABLE = 9010;
const int Buchi::LTL_NON_TRIVIAL = 9020;

LtlFormula Buchi::formula;
size_t Buchi::stateNum;
set<size_t> Buchi::init;
map< size_t,set<size_t> > Buchi::trans;
map< size_t,set<Expr> > Buchi::posProp,Buchi::negProp;
map< size_t,set<Action> > Buchi::posAct,Buchi::negAct;
set< set<size_t> > Buchi::accept,Buchi::scc;
vector< set<size_t> > Buchi::acceptVector;

/*********************************************************************/
//initialize the automaton
/*********************************************************************/
int Buchi::Initialize(const LtlFormula &f)
{
  //start timer and cleanup
  Timer bacTimer; bacTimer.Start();
  Cleanup();
  //negate and normalize
  formula = LtlManager::NegateLtl(f);
  formula = LtlManager::NormalizeLtl(formula);
  //formula = LtlManager::AddActionConstraints(formula);
  //create the string representation suitable for Wring and also
  //create the mapping from propositions and actions to variables
  map<Expr,string> propToVar;
  map<string,Expr> varToProp;
  map<Action,string> actToVar;
  map<string,Action> varToAct;
  //create the buchi automaton
  char bfname[128];
  snprintf(bfname,128,"wring-%d.buchi",getpid());
  CreateBuchi(bfname,propToVar,varToProp,actToVar,varToAct);
  //parse the buchi automaton
  ParseBuchi(bfname,propToVar,varToProp,actToVar,varToAct);
  //remove the automata file
  remove(bfname);
  //compute the strongly connected components and simplify the
  //accepting states
  ComputeSCC();
  SimplifyAccepting();
  //stop timer and update statistics
  bacTimer.Stop(); Statistics::bacTimer.Forward(bacTimer);
  //check if the buchi automaton accepts no traces
  if(BuchiEmpty()) return LTL_TAUTOLOGY;
  return LTL_NON_TRIVIAL;
}

/*********************************************************************/
//cleanup allocated states
/*********************************************************************/
void Buchi::Cleanup()
{
  stateNum = 0;
  init.clear();
  trans.clear();
  posProp.clear();
  negProp.clear();
  posAct.clear();
  negAct.clear();
  accept.clear();
  acceptVector.clear();
  scc.clear();
}

/*********************************************************************/
//create the buchi automaton and store it in the file with name given
//by first argument. in the remaining arguments store mappings from
//propositions and actions to LTL variables and vice-versa.
/*********************************************************************/
void Buchi::CreateBuchi(const string &fname,map<Expr,string> &propToVar,
			map<string,Expr> &varToProp,map<Action,string> &actToVar,
			map<string,Action> &varToAct)
{
  char *envptr = getenv("MAGICROOT");
  if(envptr == NULL) {
    Util::Error("MAGICROOT not set ... Aborting ...\n");
  }
  string wringStr = LtlManager::GetWringString(formula,propToVar,varToProp,actToVar,varToAct);
#ifdef WIN32
  string command = "perl " + string(envptr) + "/Wring-1.1.0/ltl2aut.pl -f \'" + wringStr + "\' > " + fname;
#else
  string command = string(envptr) + "/Wring-1.1.0/ltl2aut.pl -f \'" + wringStr + "\' > " + fname;
#endif //WIN32
  system(command.c_str());
  remove("ltl2aut-buechi.dot");
  remove("ltl2aut-scc.dot");
  remove("ltl2aut-parse.dot");
}

/*********************************************************************/
//parse the buchi automaton stored in the file with name given by
//first argument. the remaining arguments contain mappings from
//propositions and actions to LTL variables and vice-versa.
/*********************************************************************/
void Buchi::ParseBuchi(const string &fname,map<Expr,string> &propToVar,
		       map<string,Expr> &varToProp,map<Action,string> &actToVar,
		       map<string,Action> &varToAct)
{
  //parse the automata file
  FILE *bfptr = fopen(fname.c_str(),"r");
  assert(bfptr != NULL);
  char buf[10000];
  //maps from state names to indices and vice-versa
  map<string,size_t> nameToId;
  //get to the states command
  while(true) {
    assert(fgets(buf,10000,bfptr) != NULL);
    if(strstr(buf,"States") != NULL) break;    
  }
  //get the states
  while(true) {
    assert(fscanf(bfptr,"%s",buf) != EOF);
    if(strstr(buf,"Arcs") != NULL) break;
    assert(buf[strlen(buf) - 1] == ':');
    buf[strlen(buf) - 1] = '\0';
    assert(nameToId.count(buf) == 0);
    size_t sid = nameToId.size();
    nameToId[buf] = sid;
    ++stateNum;
    assert(fscanf(bfptr,"%s",buf) != EOF);
    assert((buf[0] == '{') && (buf[strlen(buf) - 1] == '}'));
    assert(fscanf(bfptr,"%s",buf) != EOF);
    assert(!strcmp(buf,"label:"));
    assert(fscanf(bfptr,"%s",buf) != EOF);
    assert((buf[0] == '{') && (buf[strlen(buf) - 1] == '}'));
    size_t pos = 1;
    while(buf[pos] != '}') {
      //get the next variable and value
      string var;
      bool val;
      while(buf[pos] != '=') var.push_back(buf[pos++]);
      ++pos;
      if(buf[pos] == '0') val = false;
      else if(buf[pos] == '1') val = true;
      else assert(false);
      ++pos;
      //set the labels
      if(varToProp.count(var) != 0) {
	if(val) posProp[sid].insert(varToProp[var]);
	else negProp[sid].insert(varToProp[var]);
      } else if(varToAct.count(var) != 0) {
	if(val) posAct[sid].insert(varToAct[var]);
	else negAct[sid].insert(varToAct[var]);
      } else assert(false);
      if(buf[pos] == ',') ++pos;
    }
    assert(pos == (strlen(buf) - 1));
  }
  //get the arcs
  while(true) {
    assert(fscanf(bfptr,"%s",buf) != EOF);
    if(!strcmp(buf,"Fair")) {
      assert(fscanf(bfptr,"%s",buf) != EOF);
      assert(!strcmp(buf,"Sets"));
      break;
    } 
    bool isInit = false;
    if(!strcmp(buf,"->")) {
      isInit = true;
      assert(fscanf(bfptr,"%s",buf) != EOF);
    }
    assert(nameToId.count(buf) != 0);
    size_t sid = nameToId[buf];
    if(isInit) init.insert(sid);
    assert(fscanf(bfptr,"%s",buf) != EOF);
    assert(!strcmp(buf,"->"));
    assert(fscanf(bfptr,"%s",buf) != EOF);
    assert((buf[0] == '{') && (buf[strlen(buf) - 1] == '}'));
    size_t pos = 1;
    while(buf[pos] != '}') {
      string var;
      while((buf[pos] != ',') && (buf[pos] != '}')) var.push_back(buf[pos++]);
      assert(nameToId.count(var) != 0);
      size_t nid = nameToId[var];
      trans[sid].insert(nid);
      if(buf[pos] == ',') ++pos;
    }
    assert(pos == (strlen(buf) - 1));
  }
  //get the accepting states
  while(true) {
    assert(fscanf(bfptr,"%s",buf) != EOF);
    if(strstr(buf,"End") != NULL) break;
    assert((buf[0] == '{') && (buf[strlen(buf) - 1] == '}'));
    size_t pos = 1;
    set<size_t> fair;
    while(buf[pos] != '}') {
      string var;
      while((buf[pos] != ',') && (buf[pos] != '}')) var.push_back(buf[pos++]);
      assert(nameToId.count(var) != 0);
      size_t sid = nameToId[var];
      fair.insert(sid);
      if(buf[pos] == ',') ++pos;
    }
    assert(pos == (strlen(buf) - 1));
    accept.insert(fair);
  }
  //all done
  fclose(bfptr);
  //if there is no accepting set then every state is accepting
  if((stateNum != 0) && accept.empty()) {
    set<size_t> ele;
    for(size_t i = 0;i < stateNum;++i) ele.insert(i);
    accept.insert(ele);
  }
}

/*********************************************************************/
//display the buchi automaton
/*********************************************************************/
void Buchi::Display()
{
  Util::Message(2,"********* begin buchi automaton *********\n");
  Util::Message(2,"number of Buchi states: %d ...\n",stateNum);
  for(size_t i = 0;i < stateNum;++i) {
    Util::Message(2,"label %d: ",i);
    for(set<Expr>::const_iterator j = posProp[i].begin();j != posProp[i].end();++j) {
      Util::Message(2,"%s ",Util::TrimString(j->ToString()).c_str());
    }
    for(set<Expr>::const_iterator j = negProp[i].begin();j != negProp[i].end();++j) {
      Util::Message(2,"!(%s) ",Util::TrimString(j->ToString()).c_str());
    }
    for(set<Action>::const_iterator j = posAct[i].begin();j != posAct[i].end();++j) {
      Util::Message(2,"%s ",j->ToString().c_str());
    }
    for(set<Action>::const_iterator j = negAct[i].begin();j != negAct[i].end();++j) {
      Util::Message(2,"!(%s) ",j->ToString().c_str());
    }
    Util::Message(2,"\n");
  }
  if(!init.empty()) {
    Util::Message(2,"initial states: ");
    for(set<size_t>::const_iterator i = init.begin();i != init.end();++i) {
      Util::Message(2,"%d ",*i);
    }
    Util::Message(2,"\n");
  }
  int transNum = 0;
  for(map< size_t,set<size_t> >::const_iterator i = trans.begin();i != trans.end();++i) {
    for(set<size_t>::const_iterator j = i->second.begin();j != i->second.end();++j) {
      Util::Message(2,"%d -> %d\n",i->first,*j);
      ++transNum;
    }
  }
  Util::Message(2,"number of Buchi transitions: %d ...\n",transNum);
  for(vector< set<size_t> >::const_iterator i = acceptVector.begin();i != acceptVector.end();++i) {
    Util::Message(2,"accept: ");
    for(set<size_t>::const_iterator j = i->begin();j != i->end();++j) {
      Util::Message(2,"%d ",*j);
    }
    Util::Message(2,"\n");
  }
  for(set< set<size_t> >::const_iterator i = scc.begin();i != scc.end();++i) {
    Util::Message(2,"scc: ");
    for(set<size_t>::const_iterator j = i->begin();j != i->end();++j) {
      Util::Message(2,"%d ",*j);
    }
    Util::Message(2,"\n");
  }
  Util::Message(2,"********** end buchi automaton **********\n");
}

/*********************************************************************/
//return true if the language accepted by the buchi automaton is empty
//and false otherwise
/*********************************************************************/
bool Buchi::BuchiEmpty()
{
  //check if there are any states or accepting sets
  return ((stateNum == 0) || acceptVector.empty());
}

/*********************************************************************/
//compute the strongly connected components of the automaton
/*********************************************************************/
void Buchi::ComputeSCC()
{
  //the first DFS
  set<size_t> visited;
  map<size_t,size_t> finish;
  for(size_t i = 0;i < stateNum;++i) {
    if(visited.count(i) == 0) DFS1(i,visited,finish);
  }
  //reverse the transition relation
  map< size_t,set<size_t> > revTrans;
  for(map< size_t,set<size_t> >::const_iterator i = trans.begin();i != trans.end();++i) {
    for(set<size_t>::const_iterator j = i->second.begin();j != i->second.end();++j) {
      revTrans[*j].insert(i->first);
    }
  }
  //the second DFS
  visited.clear();
  size_t sccCount = 0;
  for(map<size_t,size_t>::reverse_iterator i = finish.rbegin();i != finish.rend();++i) {
    if(visited.count(i->second) == 0) {
      set<size_t> sccEle;
      DFS2(i->second,revTrans,visited,sccEle);
      scc.insert(sccEle);
      sccCount += sccEle.size();
    }
  }
  assert(sccCount == stateNum);
}

/*********************************************************************/
//the first DFS for computing SCC. the argument maps finish times to
//nodes.
/*********************************************************************/
void Buchi::DFS1(const size_t node,set<size_t> &visited,map<size_t,size_t> &finish)
{
  visited.insert(node);
  map< size_t,set<size_t> >::const_iterator i = trans.find(node);
  if(i != trans.end()) { 
    for(set<size_t>::const_iterator j = i->second.begin();j != i->second.end();++j) {
      if(visited.count(*j) == 0) DFS1(*j,visited,finish);
    }
  }
  size_t ft = finish.size();
  finish[ft] = node;
}

/*********************************************************************/
//the second DFS for computing SCC
/*********************************************************************/
void Buchi::DFS2(const size_t node,const map< size_t,set<size_t> > &revTrans,set<size_t> &visited,set<size_t> &sccEle)
{
  visited.insert(node);
  sccEle.insert(node);
  map< size_t,set<size_t> >::const_iterator i = revTrans.find(node);
  if(i == revTrans.end()) return;
  for(set<size_t>::const_iterator j = i->second.begin();j != i->second.end();++j) {
    if(visited.count(*j) == 0) DFS2(*j,revTrans,visited,sccEle);
  }
}

/*********************************************************************/
//simplify the accepting states on the basis of SCCs
/*********************************************************************/
void Buchi::SimplifyAccepting()
{
  //check if there are any states
  if(stateNum == 0) return;
  //compute the set of non-trivial sccs such that it contains at least
  //one element from each accepting state
  set< set<size_t> > ntscc;
  for(set< set<size_t> >::const_iterator i = scc.begin();i != scc.end();++i) {
    bool flag = false;    
    if(i->size() > 1) flag = true;
    else {
      const size_t &ele = *(i->begin());
      if(trans[ele].count(ele) != 0) flag = true;
    }
    if(!flag) continue;
    flag = true;
    for(set< set<size_t> >::const_iterator j = accept.begin();j != accept.end();++j) {
      set<size_t> a = *j;
      a.insert(i->begin(),i->end());
      if(a.size() == (i->size() + j->size())) {
	flag = false;
	break;
      }
    }
    if(flag) ntscc.insert(*i);
  }
  //update the set of accepting states by eliminating elements that do
  //not belong to some non-trivial scc or are supersets of already
  //existing elements. also convert it to a vector of sets of states
  //for easier lookup later on.
  for(set< set<size_t> >::const_iterator i = accept.begin();i != accept.end();++i) {
    set<size_t> newEle;
    for(set<size_t>::const_iterator j = i->begin();j != i->end();++j) {
      bool flag = false;
      for(set< set<size_t> >::const_iterator k = ntscc.begin();k != ntscc.end();++k) {
	if(k->count(*j) != 0) {
	  flag = true;
	  break;
	}
      }
      if(flag) newEle.insert(*j);
    }
    if(newEle.empty()) continue;
    bool flag = true;
    for(vector< set<size_t> >::const_iterator j = acceptVector.begin();j != acceptVector.end();++j) {
      set<size_t> x = *j;
      x.insert(newEle.begin(),newEle.end());
      if(x.size() == newEle.size()) {
	flag = false;
	break;
      }
    }
    if(flag) acceptVector.push_back(newEle);
  }
}

/*********************************************************************/
//return the initial states of this buchi automaton
/*********************************************************************/
void Buchi::GetInitStates(set<int> &res)
{
  for(set<size_t>::const_iterator i = init.begin();i != init.end();++i) {
    res.insert((*i) * acceptVector.size());
  }
}

/*********************************************************************/
//return the set of positive propositions labeling a state
/*********************************************************************/
set<Expr> Buchi::GetPosProps(int node)
{
  return posProp[node / acceptVector.size()];
}

/*********************************************************************/
//return the set of negative propositions labeling a state
/*********************************************************************/
set<Expr> Buchi::GetNegProps(int node)
{
  return negProp[node / acceptVector.size()];
}

/*********************************************************************/
//return the set of positive actions labeling a state
/*********************************************************************/
set<Action> Buchi::GetPosActs(int node)
{
  return posAct[node / acceptVector.size()];
}

/*********************************************************************/
//return the set of negative actions labeling a state
/*********************************************************************/
set<Action> Buchi::GetNegActs(int node)
{
  return negAct[node / acceptVector.size()];
}

/*********************************************************************/
//return the set of successors of the given node
/*********************************************************************/
set<int> Buchi::GetSuccs(int node)
{
  set<int> res;
  int count = node % acceptVector.size();
  int state = node / acceptVector.size();
  const set<size_t> &succs = trans[state];
  for(set<size_t>::const_iterator i = succs.begin();i != succs.end();++i) {
    int nextCount = (acceptVector[count].count(*i) == 0) ? count : ((count + 1) % acceptVector.size());
    res.insert((*i) * acceptVector.size() + nextCount);
  }
  return res;
}

/*********************************************************************/
//return true if the node is an accepting state
/*********************************************************************/
bool Buchi::IsAccept(const int node)
{
  if((node % acceptVector.size()) != 0) return false;
  return (acceptVector[0].count(node / acceptVector.size()) != 0);
}

/*********************************************************************/
//return true if the node is a sink node i.e. the only outgoing
//transition is a self-loop
/*********************************************************************/
bool Buchi::IsSink(const int node)
{
  if(!GetPosProps(node).empty()) return false;
  if(!GetNegProps(node).empty()) return false;
  if(!GetPosActs(node).empty()) return false;
  if(!GetNegActs(node).empty()) return false;
  set<int> s = GetSuccs(node);
  return ((s.size() == 1) && (s.count(node) != 0));
}

/*********************************************************************/
//end of Buchi.cpp
/*********************************************************************/
