/******************************** CPPFile *****************************

* FileName [BigInt.cpp]

* PackageName [main]

* Synopsis [Method definitions of BigInt class.]

* SeeAlso [BigInt.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cassert>
#include <gmp.h>
#include <string>
using namespace std;

#include "BigInt.h"
using namespace magic;

/*********************************************************************/
//constructors and destructors
/*********************************************************************/
BigInt::BigInt() { mpz_init(data); }

BigInt::BigInt(const signed long int i) 
{ 
  mpz_init_set_si(data,i); 
}

BigInt::BigInt(const string &str,const int base) 
{ 
  mpz_init_set_str(data,str.c_str(),base); 
}

BigInt::BigInt(const BigInt &rhs) 
{
  mpz_init_set(data,rhs.data);
}

BigInt::~BigInt() { mpz_clear(data); }

/*********************************************************************/
//operators
/*********************************************************************/
const BigInt &BigInt::operator = (const BigInt &rhs)
{
  mpz_set(data,rhs.data);
  return *this;
}

bool BigInt::operator == (const BigInt &rhs) const
{
  return (mpz_cmp(data,rhs.data) == 0);
}

bool BigInt::operator < (const BigInt &rhs) const
{
  return (mpz_cmp(data,rhs.data) < 0);
}

BigInt BigInt::operator + (const BigInt &rhs) const
{
  BigInt res;
  mpz_add(res.data,data,rhs.data);
  return res;
}

BigInt BigInt::operator - (const BigInt &rhs) const
{
  BigInt res;
  mpz_sub(res.data,data,rhs.data);
  return res;
}

BigInt BigInt::operator * (const BigInt &rhs) const
{
  BigInt res;
  mpz_mul(res.data,data,rhs.data);
  return res;
}

BigInt BigInt::operator / (const BigInt &rhs) const
{
  BigInt res;
  mpz_tdiv_q(res.data,data,rhs.data);
  return res;
}

/*********************************************************************/
//conversion to other formats
/*********************************************************************/
string BigInt::ToString() const
{
  char buf[128];
  assert(mpz_get_str(buf,10,data) == buf);
  return buf;
}

signed long int BigInt::ToSL() const
{
  return mpz_get_si(data);
}

/*********************************************************************/
//file input and output
/*********************************************************************/
void BigInt::Read(FILE *in)
{
  assert(mpz_inp_str(data,in,10) != 0);
}

void BigInt::Write(FILE *out) const
{
  assert(mpz_out_str(out,10,data) != 0);
}

/*********************************************************************/
//arithmetic operations
/*********************************************************************/
void BigInt::Add(BigInt &res,const BigInt &lhs,const BigInt &rhs)
{
  mpz_add(res.data,lhs.data,rhs.data);
}

void BigInt::Mul(BigInt &res,const BigInt &lhs,const BigInt &rhs)
{
  mpz_mul(res.data,lhs.data,rhs.data);
}

void BigInt::Div(BigInt &quo,BigInt &rem,const BigInt &lhs,const BigInt &rhs)
{
  mpz_tdiv_qr(quo.data,rem.data,lhs.data,rhs.data);
}

/*********************************************************************/
//end of BigInt.cpp
/*********************************************************************/
