/******************************** CPPFile *****************************

* FileName [AbsRefiner.cpp]

* PackageName [main]

* Synopsis [This file contains method definitions of the AbsRefiner
* class.]

* SeeAlso [AbsRefiner.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <gmp.h>
#include <string>
#include <list>
#include <map>
#include <vector>
#include <set>
#include <stack>
#include <typeinfo>
using namespace std;

#include "BigInt.h"
#include "Util.h"
#include "Timer.h"
#include "Statistics.h"
#include "Node.h"
#include "Action.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "GlobalCEDag.h"
#include "LocalAbsLts.h"
#include "ConcCEDag.h"
#include "Component.h"
#include "ProcManager.h"
#include "CEDagVerifier.h"
#include "PredAbsRefiner.h"
#include "LtsAbsRefiner.h"
#include "AbsRefiner.h"
using namespace magic;

/*********************************************************************/
//define static members of AbsRefiner
/*********************************************************************/
const int AbsRefiner::CE_VALID = 11500;
const int AbsRefiner::CE_INVALID_REFINED = 11510;
const int AbsRefiner::CE_INVALID_NOT_REFINED = 11520;

/*********************************************************************/
//do it all. if the counter example is invalid add new predicates and
//return false. otherwise return true.
/*********************************************************************/
int AbsRefiner::Run()
{
  //we cannot generate counterexample for simulation or error state
  //reachability unless we are using the horn sat solver
  if(((Database::CONF_TYPE == Database::CONF_SIMUL) || (Database::CONF_TYPE == Database::CONF_REACH)) && 
     (Database::SAT_SOLVER_USED != Database::SAT_SOLVER_HORN_AI)) {
    return CE_VALID;
  }

  //compute the global CE dag
  Statistics::cegTimer.Start();
  GlobalCEDag::Initialize();
  Statistics::cegTimer.Stop();
  Util::Message(2,"global counter example dag initialized ...\n");

  //if the implementation is a C program
  if(Database::IMPL_TYPE == Database::IMPL_C) return RunC();
#ifdef MAGIC_FULL
  //if the implementation is a PACC process
  else if(Database::IMPL_TYPE == Database::IMPL_PACC) return RunPacc();
#endif //MAGIC_FULL
  //impossible
  else assert(false);
}

/*********************************************************************/
//refine abstraction for a C program
/*********************************************************************/
int AbsRefiner::RunC()
{
  //flag to indicate if a valid CE was found
  bool validCE = false;
  //the index of the first component with an invalid dag
  int firstInvDag = -1;
  //the index of the first component with an invalid path
  int firstInvPath = -1;

  //clear all spurious concrete CEs
  for(size_t i = 0;i < Database::components.size();++i) {
    static_cast<ProcManager*>(Database::components[i])->dagToLocs.clear();
    Database::components[i]->ClearL2AbsRefineInfo();
  }

  //the number of spurious counterexamples found in each component
  vector<int> invDagCount,invPathCount;
  for(size_t i = 0;i < Database::components.size();++i) {
    invDagCount.push_back(0);
    invPathCount.push_back(0);
  }

  bool decidedPath = false;
  for(size_t j = 0;(!validCE) && (j < Database::MAX_GLOBAL_CE);++j) {
    //compute the global CE dag - continue only if this is a new dag
    Statistics::cegTimer.Start();
    bool gcd = GlobalCEDag::Compute();
    Statistics::cegTimer.Stop();
    if(!gcd) {
      assert(j > 0);
      break;
    }
    Util::Message(2,"global counter example dag computed ...\n");
    //otherwise do abstraction refinement on each component procedure
    validCE = true;
    for(size_t i = 0;i < Database::components.size();++i) {
      Statistics::cevTimer.Start();
      int x = CEDagVerifier(*(Database::components[i]),false).Run();
      Statistics::cevTimer.Stop();
      if(x == CEDagVerifier::CE_INVALID_DAG) {
	decidedPath = true;
	validCE = false;
	firstInvDag = (firstInvDag == -1) ? i : firstInvDag;
	++invDagCount[i];
      } else if(x == CEDagVerifier::CE_INVALID_PATH) {
	decidedPath = true;
	validCE = false;
	firstInvPath = (firstInvPath == -1) ? i : firstInvPath;
	++invPathCount[i];
      } else if(x == CEDagVerifier::CE_VALID) {
	decidedPath = true;
      } else {
	assert((x == CEDagVerifier::CE_UNDECIDED_DAG) ||
	       (x == CEDagVerifier::CE_UNDECIDED_PATH));
	if(Database::CONF_TYPE != Database::CONF_LTL) {
	  Util::Error("ERROR: undecided counterexample found for non LTL conformance ...\n");
	}
	validCE = false;
      }
    }
    Util::Message(2,"CE dag projections analysed ...\n");
  }
  //check if some path was decided
  if(!decidedPath) {
    Util::Error("ERROR: all counterexamples undecided ..\n"
		"HINT1: generate more counterexamples: use --ceDag %d\n"
		"HINT2: unwind loops further: use --predLoop %d\n",
		Database::MAX_GLOBAL_CE + 1,Database::MAX_PRED_INFER_LOOP + 1);
  }
  //if this counter example is valid
  if(validCE) return CE_VALID;
  //if this counter example is invalid because of lts abstraction
  else if(firstInvDag != -1) {
    int maxInd = 0,maxCount = invDagCount[0];
    for(size_t i = 1;i < Database::components.size();++i) {
      if(invDagCount[i] > maxCount) {
	maxInd = i;
	maxCount = invDagCount[i];
      }
    }
    Database::lastRefType = Database::REFINE_LTS;
    Database::lastRefComp = maxInd;
    Statistics::larTimer.Start();
    int res = LtsAbsRefiner(*(static_cast<ProcManager*>(Database::components[maxInd]))).Run();
    Statistics::larTimer.Stop();
    assert(res == LtsAbsRefiner::LTS_ABS_REFINED);
    return CE_INVALID_REFINED;
  }
  //if this counter example is invalid because of predicate
  //abstraction
  else {
    assert(firstInvPath != -1);
    int maxInd = 0,maxCount = invPathCount[0];
    for(size_t i = 1;i < Database::components.size();++i) {
      if(invPathCount[i] > maxCount) {
	maxInd = i;
	maxCount = invPathCount[i];
      }
    }
    Database::lastRefType = Database::REFINE_PRED;
    Database::lastRefComp = maxInd;
    Statistics::parTimer.Start();
    int res = PredAbsRefiner(*(static_cast<ProcManager*>(Database::components[maxInd]))).Run();
    Statistics::parTimer.Stop();
    if(res == PredAbsRefiner::REFINED_NEW_PREDS) {
      return CE_INVALID_REFINED;
    } else {
      assert(res == PredAbsRefiner::REFINED_NO_NEW_PREDS);
      return CE_INVALID_NOT_REFINED;
    }
  }
}

/*********************************************************************/
//refine abstraction for a PACC process
/*********************************************************************/
#ifdef MAGIC_FULL
int AbsRefiner::RunPacc()
{
  return CE_VALID;
}
#endif //MAGIC_FULL

/*********************************************************************/
//end of AbsRefiner.cpp
/*********************************************************************/
