/************************************************************************ 
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * This file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 **********************************************************************/

/* Port Transmit state machine : 17.27 */
  
#include "global.h"
#include "Transmit.h"
#include "Port.h"
#include "STPM.h"
#include "Log.h"
#include "RapidSpanningTree.h"
#include "Switch.h"

#include <netinet/in.h>

#define BPDU_LEN8023_OFF    12

#define STATES {        \
  CHOOSE(TRANSMIT_INIT),    \
  CHOOSE(TRANSMIT_PERIODIC),    \
  CHOOSE(IDLE),         \
  CHOOSE(TRANSMIT_CONFIG),  \
  CHOOSE(TRANSMIT_TCN),     \
  CHOOSE(TRANSMIT_RSTP),    \
}

#define GET_STATE_NAME Transmit::GetStateName
#include "choose.h"

#define MIN_FRAME_LENGTH    64


typedef struct tx_tcn_bpdu_t {
  MAC_HEADER_T  mac;
  ETH_HEADER_T  eth;
  BPDU_HEADER_T hdr;
} TCN_BPDU_T;

typedef struct tx_stp_bpdu_t {
  MAC_HEADER_T  mac;
  ETH_HEADER_T  eth;
  BPDU_HEADER_T hdr;
  BPDU_BODY_T   body;
} CONFIG_BPDU_T;

typedef struct tx_rstp_bpdu_t {
  MAC_HEADER_T  mac;
  ETH_HEADER_T  eth;
  BPDU_HEADER_T hdr;
  BPDU_BODY_T   body;
  unsigned char ver_1_length[2];
} RSTP_BPDU_T;


static RSTP_BPDU_T bpdu_packet  = {
  {/* MAC_HEADER_T */
    {0x01, 0x80, 0xc2, 0x00, 0x00, 0x00},   /* dst_mac */
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00}    /* src_mac */
  },
  { /* ETH_HEADER_T */
    {0x00, 0x00},               /* len8023 */
    BPDU_L_SAP, BPDU_L_SAP, LLC_UI      /* dsap, ssap, llc */
  },
  {/* BPDU_HEADER_T */
    {0x00, 0x00},               /* protocol */
    BPDU_VERSION_ID, 0x00           /* version, bpdu_type */
  },
  {
    0x00,                   /*  flags; */
    {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},  /*  root_id[8]; */
    {0x00,0x00,0x00,0x00},          /*  root_path_cost[4]; */
    {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},  /*  bridge_id[8]; */
    {0x00,0x00},                /*  port_id[2]; */
    {0x00,0x00},                /*  message_age[2]; */
    {0x00,0x00},                /*  max_age[2]; */
    {0x00,0x00},                /*  hello_time[2]; */
    {0x00,0x00},                /*  forward_delay[2]; */
  },
   {0x00,0x00},                 /*  ver_1_length[2]; */
};

Transmit::Transmit(Port *p) : StateMachine("Transmit"), owner(p)
{
  // empty 
}

size_t Transmit::build_bpdu_header (int port_index, unsigned char bpdu_type,
                                    unsigned short pkt_len)
{
  unsigned short len8023;

  owner->owner->GetRapidSpanningTree()->GetSwitch()->GetPortMac(port_index, 
                                                                bpdu_packet.mac.src_mac);

  bpdu_packet.hdr.bpdu_type = bpdu_type;
  bpdu_packet.hdr.version = (BPDU_RSTP == bpdu_type) ?
                            BPDU_VERSION_RAPID_ID    :
                            BPDU_VERSION_ID;

  /* NOTE: I suppose, that sizeof(unsigned short)=2 ! */
  len8023 = htons ((unsigned short) (pkt_len + 3));
  memcpy (&bpdu_packet.eth.len8023, &len8023, 2); 

  if (pkt_len < MIN_FRAME_LENGTH) pkt_len = MIN_FRAME_LENGTH;
  return pkt_len;
}

int Transmit::txTcn()
{ /* 17.19.17 (page 68) & 9.3.2 (page 25) */
  size_t       pkt_len;
  int          port_index, vlan_id;
  Port *port = owner;

  if (port->skip_tx > 0) {
    if (1 == port->skip_tx)
      stp_trace ("port %s stop tx skipping", port->port_name);
    port->skip_tx--;
    return STP_Nothing_To_Do;
  }

  if (port->admin_non_stp) {
    return 1;
  }
  port_index = port->port_index;
  vlan_id = port->owner->GetVlanID();

  pkt_len = build_bpdu_header (port_index,
                               BPDU_TOPO_CHANGE_TYPE,
                               sizeof (BPDU_HEADER_T));

  if (debug) {
    stp_trace ("txTcn port %s", port->port_name);
  }

  return port->owner->GetRapidSpanningTree()->GetSwitch()->Send(
                                                                vlan_id,
                                                                port_index, 
                                                                (char *) &bpdu_packet,
                                                                pkt_len);
}

void Transmit::build_config_bpdu(bool set_topo_ack_flag)
{
  Port *port = owner;

  bpdu_packet.body.flags = 0;
  if (port->tcWhile) {
    if (port->topoch->GetDebug())
      stp_trace ("tcWhile=%d =>tx TOLPLOGY_CHANGE_BIT to port %s",
                 (int) port->tcWhile, port->port_name);
    bpdu_packet.body.flags |= TOLPLOGY_CHANGE_BIT;
  }

  if (set_topo_ack_flag && port->tcAck) {
    bpdu_packet.body.flags |= TOLPLOGY_CHANGE_ACK_BIT;
  }

  STP_VECT_set_vector (&port->portPrio, &bpdu_packet.body);

  STP_VECT_print("build_config_bpdu about to transmit vector", &port->portPrio);

  STP_set_times (&port->portTimes, &bpdu_packet.body);
}

int Transmit::txConfig()
{/* 17.19.15 (page 67) & 9.3.1 (page 23) */
  size_t   pkt_len;
  Port *port = owner;
  int      port_index, vlan_id;

  if (port->skip_tx > 0) {
    if (1 == port->skip_tx) {
      stp_trace ("port %s stop tx skipping", port->port_name);
    }
    port->skip_tx--;
    return STP_Nothing_To_Do;
  }

  if (port->admin_non_stp) {
    return 1;
  }
  port_index = port->port_index;
  vlan_id = port->owner->GetVlanID();
  
  pkt_len = build_bpdu_header (port->port_index,
                               BPDU_CONFIG_TYPE,
                               sizeof (BPDU_HEADER_T) + sizeof (BPDU_BODY_T));
  build_config_bpdu (true);
  stp_trace("txConfig port %s", port->port_name);
 
  if (debug) {
    stp_trace ("port %s txConfig flags=0X%lx",
        port->port_name, (unsigned long) bpdu_packet.body.flags);
  }

  return port->owner->GetRapidSpanningTree()->GetSwitch()->Send(
                                                                vlan_id,
                                                                port_index, 
                                                                (char *) &bpdu_packet,
                                                                pkt_len);
}

int Transmit::txRstp()
{/* 17.19.16 (page 68) & 9.3.3 (page 25) */
  size_t       pkt_len;
  Port *port = owner;
  int          port_index, vlan_id;
  unsigned char         role;

  if (port->skip_tx > 0) {
    if (1 == port->skip_tx) {
      stp_trace ("port %s stop tx skipping", port->port_name);
    } else {
      stp_trace ("port %s skip tx %d", port->port_name, port->skip_tx);
    }
    port->skip_tx--;
    return STP_Nothing_To_Do;
  }

  if (port->admin_non_stp) {
    return 1;
  }
  port_index = port->port_index;
  vlan_id = port->owner->GetVlanID();

  pkt_len = build_bpdu_header (port->port_index,
                               BPDU_RSTP,
                               sizeof (BPDU_HEADER_T) + sizeof (BPDU_BODY_T) + 2);
  build_config_bpdu (false);
  stp_trace("txRstp to port %s", port->port_name);

  switch (port->selectedRole) {
    default:
    case DisabledPort:
      role = RSTP_PORT_ROLE_UNKN;
      break;
    case AlternatePort:
      role = RSTP_PORT_ROLE_ALTBACK;
      break;
    case BackupPort:
      role = RSTP_PORT_ROLE_ALTBACK;
      break;
    case RootPort:
      role = RSTP_PORT_ROLE_ROOT;
      break;
    case DesignatedPort:
      role = RSTP_PORT_ROLE_DESGN;
      break;
  }

  bpdu_packet.body.flags |= (role << PORT_ROLE_OFFS);

  if (port->synced) {
    if (port->roletrns->GetDebug()) {
      stp_trace ("tx AGREEMENT_BIT to port %s", port->port_name);
    }
    bpdu_packet.body.flags |= AGREEMENT_BIT;
  }

  if (port->proposing) {
    if (port->roletrns->GetDebug()) {
      stp_trace ("tx PROPOSAL_BIT to port %s", port->port_name);
    }
    bpdu_packet.body.flags |= PROPOSAL_BIT;
  }

  if (debug) {
    stp_trace ("port %s txRstp flags=0X%lx",
               port->port_name,
               (unsigned long) bpdu_packet.body.flags);
  }

  return port->owner->GetRapidSpanningTree()->GetSwitch()->Send(
                                                                vlan_id,
                                                                port_index, 
                                                                (char *) &bpdu_packet,
                                                                pkt_len);   
}

void Transmit::EnterState()
{
  Port *port = owner;

  switch (State) {
    case BEGIN:
    case TRANSMIT_INIT:
      port->newInfo = false;
      port->helloWhen = 0;
      port->txCount = 0;
      break;
    case TRANSMIT_PERIODIC:
      port->newInfo = port->newInfo ||
                            ((port->role == DesignatedPort) ||
                             ((port->role == RootPort) && port->tcWhile));
      port->helloWhen = port->owner->GetRootTimes().HelloTime;
      break;
    case IDLE:
      break;
    case TRANSMIT_CONFIG:
      port->newInfo = false;
      txConfig();
      port->txCount++;
      port->tcAck = false;
      break;
    case TRANSMIT_TCN:
      port->newInfo = false;
      txTcn();
      port->txCount++;
      break;
    case TRANSMIT_RSTP:
      port->newInfo = false;
      txRstp();
      port->txCount++;
      port->tcAck = false;
      break;
  };
}
  
bool Transmit::CheckCondition()
{
  Port *port = owner;

  if (BEGIN == State) {
    return Hop2State(TRANSMIT_INIT, port->adminEnable);
  }

  switch (State) {
    case TRANSMIT_INIT:
      return Hop2State(IDLE, port->adminEnable);
    case TRANSMIT_PERIODIC:
      return Hop2State(IDLE, port->adminEnable);
    case IDLE:
      if (!port->helloWhen) {
        return Hop2State(TRANSMIT_PERIODIC, port->adminEnable);
      }
      if (!port->sendRSTP && port->newInfo &&
          (port->txCount < TxHoldCount) &&
          (port->role == DesignatedPort) &&
          port->helloWhen) {
        return Hop2State(TRANSMIT_CONFIG, port->adminEnable);
      }
      if (!port->sendRSTP && port->newInfo &&
          (port->txCount < TxHoldCount) &&
          (port->role == RootPort) &&
          port->helloWhen) {
        return Hop2State(TRANSMIT_TCN, port->adminEnable);
      }
      if (port->sendRSTP && port->newInfo &&
          (port->txCount < TxHoldCount) &&
          ((port->role == RootPort) ||
           (port->role == DesignatedPort))) {
        return Hop2State(TRANSMIT_RSTP, port->adminEnable);
      }
      break;
    case TRANSMIT_CONFIG:
      return Hop2State(IDLE, port->adminEnable);
    case TRANSMIT_TCN:
      return Hop2State(IDLE, port->adminEnable);
    case TRANSMIT_RSTP:
      return Hop2State(IDLE, port->adminEnable);
  };
  return false;
}

