/************************************************************************ 
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * This file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 **********************************************************************/

/* Topolgy Change state machine : 17.25 */
  
#include "global.h"
#include "TopologyChange.h"
#include "Port.h"
#include "STPM.h"
#include "Log.h"
#include "stp_to.h" /* for STP_OUT_flush_lt */
  
#define STATES { \
  CHOOSE(INIT),             \
  CHOOSE(INACTIVE),         \
  CHOOSE(TCACTIVE),         \
  CHOOSE(DETECTED),         \
  CHOOSE(NOTIFIED_TC),          \
  CHOOSE(PROPAGATING),          \
  CHOOSE(ACKNOWLEDGED),         \
  CHOOSE(NOTIFIED_TCN),         \
}

#define GET_STATE_NAME TopologyChange::GetStateName
#include "choose.h"

TopologyChange::TopologyChange(Port *p) :
  StateMachine("TopologyChange"), owner(p)
{
  // empty
}

// #ifndef STRONGLY_SPEC_802_1W
/* 
 * In many kinds of hardware the function
 * STP_OUT_flush_lt is a) is very hard and b) cannot
 * delete learning emtries per port. The alternate
 * method may be used: we don't care operEdge flag here,
 * but clean learning table once for TopologyChange
 * for all ports, except the received port. I am ready to discuss :(
 * See below word STRONGLY_SPEC_802_1W
 */
// #else // STRONGLY_SPEC_802_1W
bool TopologyChange::flush(char* reason) /* 17.19.9 */
{
  Port *port = owner;

  if (port->operEdge) {
    return true;
  }
  if (debug) {
    stp_trace("%s (%s, %s, %s, '%s')",
              "flush", port->port_name, port->owner->GetName(),
              "this port", reason);
  }
  return STP_OUT_flush_lt (port->port_index, port->owner->GetVlanID(),
                           LT_FLASH_ONLY_THE_PORT, reason);
}
// #endif // STRONGLY_SPEC_802_1W

void TopologyChange::setTcPropBridge(char* reason) /* 17.19.14 */
{
  Port *port = owner;
  port->owner->setTcProp(port);

#ifndef STRONGLY_SPEC_802_1W
  if (debug) {
    stp_trace("%s (%s, %s, %s, '%s')",
              "clearFDB", port->port_name, port->owner->GetName(),
              "other ports", reason);
  }
  STP_OUT_flush_lt (port->port_index, port->owner->GetVlanID(),
                    LT_FLASH_ALL_PORTS_EXCLUDE_THIS, reason);
#endif
}

unsigned int TopologyChange::newTcWhile() /* 17.19.7 */
{
  Port *port = owner;

  if (port->sendRSTP && port->operPointToPointMac) {
    return 2 * port->owner->GetRootTimes().HelloTime;
  }
  return port->owner->GetRootTimes().MaxAge;
}

void TopologyChange::EnterState()
{
  Port *port = owner;
  
  switch (State) {
  case BEGIN:
  case INIT:
#ifdef STRONGLY_SPEC_802_1W
    flush("topoch INIT");
#endif
    port->tcWhile = 0;
    port->tc =
      port->tcProp =
      port->tcAck = false;
    break;
  case INACTIVE:
    port->rcvdTc =
      port->rcvdTcn =
      port->rcvdTcAck = port->tc = port->tcProp = false;
    break;
  case TCACTIVE:
    break;
  case DETECTED:
    port->tcWhile = newTcWhile();
    if (debug) {
      stp_trace("DETECTED: tcWhile=%d on port %s", 
                port->tcWhile, port->port_name);
    }
    setTcPropBridge("DETECTED");
    port->tc = false;  
    break;
  case NOTIFIED_TC:
    port->rcvdTcn = port->rcvdTc = false;
    if (port->role == DesignatedPort) {
      port->tcAck = true;
    }
    setTcPropBridge("NOTIFIED_TC");
    break;
  case PROPAGATING:
    port->tcWhile = newTcWhile();
    if (this->debug) {
      stp_trace("PROPAGATING: tcWhile=%d on port %s", 
                port->tcWhile, port->port_name);
    }
#ifdef STRONGLY_SPEC_802_1W
    flush("topoch PROPAGATING");
#endif
    port->tcProp = false;
    break;
  case ACKNOWLEDGED:
    port->tcWhile = 0;
    if (debug) {
      stp_trace("ACKNOWLEDGED: tcWhile=%d on port %s", 
                port->tcWhile, port->port_name);
    }
    port->rcvdTcAck = false;
    break;
  case NOTIFIED_TCN:
    port->tcWhile = newTcWhile();
    if (debug) {
      stp_trace("NOTIFIED_TCN: tcWhile=%d on port %s", 
                port->tcWhile, port->port_name);
    }
    break;
  };
}

bool TopologyChange::CheckCondition()
{
  Port *port = owner;

  if (BEGIN == State) {
    return Hop2State(INIT);
  }

  switch (State) {
    case INIT:
      return Hop2State(INACTIVE);
    case INACTIVE:
      if (port->role == RootPort || port->role == DesignatedPort)
        return Hop2State(TCACTIVE);
      if (port->rcvdTc || port->rcvdTcn || port->rcvdTcAck ||
          port->tc || port->tcProp)
        return Hop2State(INACTIVE);
      break;
    case TCACTIVE:
      if (port->role != RootPort && (port->role != DesignatedPort))
        return Hop2State(INIT);
      if (port->tc)
        return Hop2State(DETECTED);
      if (port->rcvdTcn)
        return Hop2State(NOTIFIED_TCN);
      if (port->rcvdTc)
        return Hop2State(NOTIFIED_TC);
      if (port->tcProp && !port->operEdge)
        return Hop2State(PROPAGATING);
      if (port->rcvdTcAck)
        return Hop2State(ACKNOWLEDGED);
      break;
    case DETECTED:
      return Hop2State(TCACTIVE);
    case NOTIFIED_TC:
      return Hop2State(TCACTIVE);
    case PROPAGATING:
      return Hop2State(TCACTIVE);
    case ACKNOWLEDGED:
      return Hop2State(TCACTIVE);
    case NOTIFIED_TCN:
      return Hop2State(NOTIFIED_TC);
  };
  return false;
}

