#include "global.h"
#include "Switch.h"
#include "uid_stp.h"
#include "bitmap.h"
#include "Simulator.h"
#include "Log.h"
#include <iostream>
#include <string.h>
#include <sstream>

Switch::Switch(unsigned long switchID, unsigned numPorts, double offset, 
               Simulator *sim)
  : NumPorts(numPorts), SwitchID(switchID), simulator(sim), RSTP(this), 
    OST(offset, this, sim)
{
  // we'll insert NumPorts+1 ports.  port 0 is really just a
  // placeholder because RSTP doesn't like port id's of 0.  so we add
  // the unused port 0 and then we don't have to constantly add and
  // subtract 1 when trying to find things in the Interfaces vector
  for (unsigned i = 0; i <= NumPorts; i++) {
    Interface x;
    x.enabled = false;
    x.used = false;
    Interfaces.push_back(x);
  }

  ostringstream ost;
  ost << "br" << SwitchID;
  SwitchName = strdup(ost.str().c_str());
}

Switch::~Switch()
{
  free(SwitchName);
  SwitchName = "nutt'n'honey";
}

void Switch::Init()
{
  LogContext lc(SwitchName);
  RSTP.STP_IN_init(NumPorts);

  BITMAP_T ports;
  UID_STP_CFG_T uid_cfg;

  BitmapClear(&ports);
  for (unsigned i = 1; i <= NumPorts; i++) {
    BitmapSetBit(&ports, i - 1);
  }
  uid_cfg.field_mask = BR_CFG_STATE;
  uid_cfg.stp_enabled = STP_ENABLED;
  snprintf (uid_cfg.vlan_name, NAME_LEN - 1, "B%ld", (long)SwitchID);
  // note: we're just arbitrarily using a vlan id of 0
  unsigned rc = RSTP.STP_IN_stpm_set_cfg (0, &ports, &uid_cfg);
  if (rc != STP_OK) {
    cerr << "can't enable spanning tree: " 
         << RSTP.STP_IN_get_error_explanation(rc) << endl;
    exit(1);
  }
}

void Switch::GetPortMac(unsigned port_index, unsigned char *mac)
{
  assert(port_index > 0);
  assert(port_index <= NumPorts);

  /* starting with number 11 means that hex printouts will start with
     0xb, and b is for bridge */
  static unsigned char mac_beg[] = {11, '\0', '\0', '\0', '\0', '\0'};

  memcpy (mac_beg + 1, &SwitchID, 4);
  memcpy (mac, mac_beg, 5);
  mac[5] = port_index;
}

bool Switch::GetPortStatus(unsigned port_index)
{
  assert(port_index > 0);
  assert(port_index <= NumPorts);

  return Interfaces[port_index].enabled;
}

unsigned Switch::GetUnusedPort(unsigned suggestedPortNum)
{
  LogContext lc(SwitchName);
  unsigned i;
  unsigned toUse = 0;

  if (suggestedPortNum != 0 &&
      suggestedPortNum < Interfaces.size() &&
      !Interfaces[suggestedPortNum].used) {
    toUse = suggestedPortNum;
  } else {
    // suggestion didn't work out.  just try each port in order.

    for (i = 1; i < Interfaces.size(); i++) {
      // start i at 1 because interface 0 doesn't exist!
      if (!Interfaces[i].used) {
        toUse = i;
        break;
      }
    }
  }

  if (toUse != 0) {
    Interfaces[toUse].used = true;
    return toUse;
  }

  cerr << "Switch " << SwitchID << " has no free ports but is being asked for one."
       << endl;
  exit(1);
}

void Switch::EnablePort(unsigned port)
{
  LogContext lc(SwitchName);

  assert(port > 0 && port <= NumPorts);
  assert(Interfaces[port].enabled == false);

  Interfaces[port].enabled = true;
  RSTP.STP_IN_enable_port(port, true);
}

void Switch::DisablePort(unsigned port)
{
  LogContext lc(SwitchName);

  assert(port > 0 && port <= NumPorts);
  assert(Interfaces[port].enabled);

  Interfaces[port].enabled = false;
  RSTP.STP_IN_enable_port(port, false);
}

int Switch::Send(int vlan_id, unsigned port_index, char *orig_pkt, size_t pktlen)
{
  char *pkt = new char[pktlen];
  memcpy(pkt, orig_pkt, pktlen);

  assert(pkt);
  PortUID puid(this, port_index);
  stp_trace("send switch id %d port %d pkt len %d", SwitchID, port_index, pktlen);

  stp_trace_bpdu(pkt, pktlen, true, port_index);

  simulator->ForwardPacket(puid, pkt, pktlen);

  return 0;
}

void Switch::Recv(int vlan_id, unsigned port_index, char *pkt, size_t pktlen)
{
  LogContext lc(SwitchName);
  assert(pkt);
  stp_trace("recv switch id %d port %d pkt len %d", SwitchID, port_index, pktlen);
  stp_trace_bpdu(pkt, pktlen, false, port_index);
  RSTP.STP_IN_rx_bpdu(vlan_id, port_index, 
                      (BPDU_T *)(pkt + sizeof(MAC_HEADER_T)), 
                      pktlen - sizeof(MAC_HEADER_T));
  delete[] pkt;
}

void Switch::OneSecondCallback()
{
  LogContext lc(SwitchName);
  RSTP.STP_IN_one_second();
}

void OneSecondTimer::Callback()
{
  S->OneSecondCallback();
}
