#ifndef SIMULATOR_H_INCLUDED
#define SIMULATOR_H_INCLUDED

#include <ext/hash_map>
#include <queue>

#include "Event.h"

class Switch;
class Simulator;

struct PortUID
{
  PortUID(Switch *s, unsigned p) : sw(s), port(p) { }
  PortUID() { sw = NULL; port = 0; }

  Switch *sw;  // i would have called sw 'switch' but that's a keyword
  unsigned port;
};

struct Link {
  Link(PortUID d, double del) : dest(d), delay(del) { }
  Link() { delay = -1; }

  PortUID dest;
  double delay;
};

struct hashPortUID {
  unsigned operator()(const PortUID& p) const
  {
    return (unsigned)(p.sw) ^ p.port;
  }
};

struct eqPortUID {
  unsigned operator()(const PortUID& p, const PortUID& q) const
  {
    return (p.sw == q.sw) && (p.port == q.port);
  }
};

class EndSimulationEvent : public Event
{
 public:
  EndSimulationEvent(double when);
  void Callback();
};

class PortDownEvent : public Event
{
 public:
  PortDownEvent(double when, PortUID p1, Simulator *sim);
  ~PortDownEvent() { }
  void Callback();
 protected:
  PortUID P1;
  Simulator *simulator;
};

class LinkDownEvent : public Event
{
 public:
  LinkDownEvent(double when, PortUID p1, PortUID p2, Simulator *sim);
  ~LinkDownEvent() { }
  void Callback();
 protected:
  PortUID P1;
  PortUID P2;
  Simulator *simulator;
};

class PacketForwardEvent : public Event
{
 public:
  PacketForwardEvent(PortUID recv, char *pkt, int pktlen, double linkDelay);
  ~PacketForwardEvent();
  void Callback();

 protected:
  PortUID Receiver;
  char *Data;
  int Length;
};

class Simulator 
{
 public:
  Simulator();
  virtual ~Simulator();
  
  // interface for setting up the topology
  void AddLinkDownEvent(double t, int nodeA, int nodeB, 
                        double delayA, double delayB);
  void AddSimulationEndEvent(double t);
  void AddLink(int nodeA, int nodeB, double delay);
  void AddNode(int nodeID, double offset);

  // enqueue an event (e.g. for a timer)
  void AddEvent(Event *ev);

  // forward a packet.  if port is disconnected, will just delete packet.
  void ForwardPacket(PortUID sender, char *buf, int len);

  /* 
     to take down a link between PortUID p1 and PortUID p2...  first
     call DisableLink(p1, p2).  later call DisablePort(p1) and
     DisablePort(p2)
  */

  // inform the RSTP running on the port that the port has been
  // disabled
  void DisablePort(PortUID p1);

  // remove a link from the interconnect
  void DisableLink(PortUID p1, PortUID p2);

  // to start the simulation
  void Run();

 protected:
  typedef hash_map<PortUID, Link, hashPortUID, eqPortUID> LinkMap;
  LinkMap Interconnect;

  typedef hash_map<int, Switch *> NodeMap;
  NodeMap Switches;

  typedef priority_queue<Event *, vector<Event *>, gtevent> EventQueue;
  EventQueue Events;
};

#endif // SIMULATOR_H_INCLUDED
