#include "global.h"
#include "Simulator.h"
#include "Switch.h"
#include "bitmap.h"  // for NUMBER_OF_PORTS
#include "Log.h"

#include <iostream>

static double CurrentTime = 0.0;

// to stop the sim, EndSimulationEvent sets this bool to false
static bool KeepGoing = true; 

double GetCurrentTime() {
  return CurrentTime;
}

Simulator::Simulator() 
{
  // empty
}

Simulator::~Simulator()
{
  NodeMap::iterator i;
  for (i = Switches.begin(); i != Switches.end(); i++) {
    delete i->second;
  }
  Switches.clear();
  
  while (!Events.empty()) {
    Event *e = Events.top();
    Events.pop();
    delete e;
  }
}

void Simulator::AddLinkDownEvent(double when, int nodeA, int nodeB,
                                 double delayA, double delayB)
{
  cerr << "AddLinkDownEvent " << when << " " << nodeA << " " 
       << nodeB << endl;
  Switch *n1 = Switches[nodeA];
  if (!n1) {
    cerr << "AddLinkDownEvent: node " << nodeA << " does not exist." << endl;
    exit(1);
  }
  Switch *n2 = Switches[nodeB];
  if (!n2) {
    cerr << "AddLinkDownEvent: node " << nodeB << " does not exist." << endl;
    exit(1);
  }
  unsigned p1, p2 = 0;
  for (p1 = 1; p1 < NUMBER_OF_PORTS; p1++) {
    PortUID uid(n1, p1);
    LinkMap::iterator lmi = Interconnect.find(uid);
    if (lmi != Interconnect.end() && lmi->second.dest.sw == n2) {
      // we've found it!
      p2 = lmi->second.dest.port;
      PortUID uid2(n2, p2);
      Event *ev1 = new PortDownEvent(when + delayA, uid, this);
      AddEvent(ev1);
      Event *ev2 = new PortDownEvent(when + delayB, uid2, this);
      AddEvent(ev2);
      Event *ev3 = new LinkDownEvent(when, uid, uid2, this);
      AddEvent(ev3);

      return;
    }
  }
  cerr << "AddLinkDownEvent: could not find connect between nodes "
       << nodeA << " and " << nodeB << endl;
  exit(1);
}

void Simulator::AddSimulationEndEvent(double when)
{
  cerr << "AddSimulationEndEvent " << when << endl;
  Event *ev = new EndSimulationEvent(when);
  AddEvent(ev);
}

// takes 2 node id's and the link characteristic
void Simulator::AddLink(int nodeA, int nodeB, double delay)
{
  cerr << "AddLink " << nodeA << " " << nodeB << " "
       << delay << endl;

  Switch *a = Switches[nodeA];
  if (!a) {
    cerr << "AddLink: could not find node with id " << nodeA << endl;
    exit(1);
  }
  Switch *b = Switches[nodeB];
  if (!b) {
    cerr << "AddLink: could not find node with id " << nodeB << endl;
    exit(1);
  }
  unsigned portA, portB;

  // we'll try to make switch A's connection to switch B be via A's
  // port number B, and vice versa.  that's why we're feeding nodeB to
  // a->GetUnusedPort
  portA = a->GetUnusedPort(nodeB);
  assert(portA != 0);
  portB = b->GetUnusedPort(nodeA);
  assert(portB != 0);

  PortUID pa(a, portA), pb(b, portB);

  assert(Interconnect.find(pa) == Interconnect.end());
  assert(Interconnect.find(pb) == Interconnect.end());

  Link LinkAtoB(pb, delay);  
  Link LinkBtoA(pa, delay);
  Interconnect[pa] = LinkAtoB;
  Interconnect[pb] = LinkBtoA;

  LogContext lc("topology");
  stp_trace("bridge %d port %d is connected to bridge %d port %d",
            nodeA, portA, nodeB, portB);

  // now that the connections are made, enable the ports
  a->EnablePort(portA);
  b->EnablePort(portB);
}

void Simulator::AddNode(int nodeID, double offset)
{
  cerr << "AddNode " << nodeID << " with offset " << offset << endl;

  if (Switches.find(nodeID) != Switches.end()) {
    cerr << "AddNode: already have a node with id " << nodeID << endl;
    exit(1);
  }
  Switch *s = new Switch(nodeID, NUMBER_OF_PORTS, offset, this);
  Switches[nodeID] = s;
  s->Init();
}

void Simulator::AddEvent(Event *ev)
{
  assert(ev);
  Events.push(ev);
}

void Simulator::ForwardPacket(PortUID sender, char *buf, int len)
{
  LinkMap::iterator i = Interconnect.find(sender);
  if (i == Interconnect.end()) {
    stp_trace("nothing connected to switch id %d port %d",
              sender.sw->GetSwitchID(), sender.port);
    delete[] buf;
  } else {
    Link& link = i->second;
    // vlan_id is always 0
    assert(link.dest.sw);

    stp_trace("fwd to switch id %d port %d",
              link.dest.sw->GetSwitchID(), link.dest.port);
    Event *ev = new PacketForwardEvent(link.dest, buf, len, link.delay);
    AddEvent(ev);
  }
}

void Simulator::DisableLink(PortUID p1, PortUID p2)
{
  stp_trace("DisableLink sw1 %d port1 %d sw2 %d port2 %d",
            p1.sw->GetSwitchID(), p1.port, p2.sw->GetSwitchID(), p2.port);
  Interconnect.erase(p1);
  Interconnect.erase(p2);
}

void Simulator::DisablePort(PortUID p1)
{
  stp_trace("DisablePort sw %d port %d", p1.sw->GetSwitchID(), p1.port);
  p1.sw->DisablePort(p1.port);
}

void Simulator::Run()
{
  cout << "Starting to run events..." << endl;
  while (!Events.empty() && KeepGoing) {
    Event *e = Events.top();
    Events.pop();
    if (!e->IsCancelled()) {
      CurrentTime = e->GetWhen();
      e->Callback();
    }
    // delete the event whether or not it's been cancelled
    delete e;
  }
  cout << "Stopping event loop" << endl;
}

EndSimulationEvent::EndSimulationEvent(double when)
  : Event("EndSimulationEvent", when)
{
  // empty
}

void EndSimulationEvent::Callback()
{
  cout << "Callback to stop sim" << endl;
  KeepGoing = false;
}


PacketForwardEvent::PacketForwardEvent(PortUID recv, char *pkt, int pktlen, 
                                       double linkDelay) 
  : Event("PacketForwardEvent", GetCurrentTime() + linkDelay),
    Receiver(recv), Data(pkt), Length(pktlen)
{
  // empty
}

PacketForwardEvent::~PacketForwardEvent()
{
  if (Data) {
    delete[] Data;
    Data = NULL;
  }
}

void PacketForwardEvent::Callback()
{
  if (Receiver.sw->GetPortStatus(Receiver.port)) {
    stp_trace("delivering a packet to switch %d port %d length %d",
              Receiver.sw->GetSwitchID(), Receiver.port, Length);
    // vlan_id is always 0
    Receiver.sw->Recv(0, Receiver.port, Data, Length);
    Data = NULL;  // Recv will delete[] it, so we drop our pointer to it
  } else {
    stp_trace("would have delivered a packet to switch %d but port %d is disabled",
              Receiver.sw->GetSwitchID(), Receiver.port);
  }
}

PortDownEvent::PortDownEvent(double when, PortUID p1, Simulator *sim)
  : Event("PortDownEvent", when), P1(p1), simulator(sim)
{
  // empty
}

void PortDownEvent::Callback()
{
  simulator->DisablePort(P1);
}


LinkDownEvent::LinkDownEvent(double when, PortUID p1, PortUID p2, Simulator *sim)
  : Event("LinkDownEvent", when), P1(p1), P2(p2), simulator(sim)
{
  // empty
}

void LinkDownEvent::Callback()
{
  simulator->DisableLink(P1, P2);
}
