/************************************************************************ 
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * This file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 **********************************************************************/

/* STP machine instance : bridge per VLAN: 17.17 */
/* The Clause 17.13 points: "NOTE:The operation of the Bridge as a whole can 
 * be represented by the interaction between Bridge Ports specified,
 * and by parameters of the Bridge stored in �ort 0� This removes the
 * need for any �er Bridge�specification elements, and helps ensure
 * the minimum dependencies between Bridge Ports. This in turn supports
 * the development of implementations that scale well with increasing
 * numbers of Bridge Ports. This shift of focus to �er Port operation� * for the RSTP is supported by underlying technical changes from the
 * Spanning Tree Algorithm and Protocol (Clause 8):"
 * Newetheless, it seems to me, the behaviour of of the bridge, its variables
 * and functions are so distinct from Port, that I decided to design Bridge
 * instance. I called this object 'stpm' from STP machine. I'd like to see
 * another procedural model, more corresponding to this note on Clause 17.13 */
  
#ifndef _STP_MACHINE_H__
#define _STP_MACHINE_H__

#include "vector.h"
#include "times.h"
#include "uid_stp.h"
#include "StateMachine.h"

#include <vector>
#include <set>

extern unsigned TxHoldCount;  /* 17.16.6, 17.28.2(Table 17-5) */

typedef enum {/* 17.12, 17.16.1 */
  FORCE_STP_COMPAT = 0,
  NORMAL_RSTP = 2
} PROTOCOL_VERSION_T;

class Port;
class RapidSpanningTree;
class RoleSelection;

class STPM 
{
 public:
  STPM(int vlan_id, char *name, RapidSpanningTree *owner);
  
  ~STPM();

  RapidSpanningTree *GetRapidSpanningTree() { return Owner; }

  UID_STP_MODE_T get_admin_state(void) { return admin_state; }

  void STP_stpm_one_second();

  int STP_stpm_enable (UID_STP_MODE_T admin_state);

  int STP_stpm_start ();

  void STP_stpm_stop ();

  int STP_stpm_update ();

  BRIDGE_ID *STP_compute_bridge_id ();

  void STP_stpm_update_after_bridge_management ();

  const char* STP_stpm_get_port_name_by_id (PORT_ID port_id);
  
  PROTOCOL_VERSION_T GetForceVersion(void) { return ForceVersion; }
  char *GetName(void) { return name; }

  // do all ports (excluding Port *except) have their synced variable
  // set true?
  bool compute_all_synced(Port *except);

  bool compute_re_rooted(Port *except);

  // set sync flag to true for all ports
  void setSyncBridge();

  // set reRoot flag to true for all ports
  void setReRootBridge();

  TIMEVALUES_T GetRootTimes() { return rootTimes; }

  int GetVlanID() { return vlan_id; }

  // set tcProp to true in all ports with the exception of Port *except
  void setTcProp(Port *except);

 protected:
  RapidSpanningTree *Owner;

  int _stp_stpm_iterate_machines(CallbackType ct, bool exit_on_non_zero_ret);
  int _individualCallback(StateMachine *sm, CallbackType ct);
  void _stp_stpm_init_data();
  unsigned char _check_topoch();

  vector<Port *> Ports;

  /* The only "per bridge" state machine */
  StateMachine *rolesel;   /* the Port Role Selection State machione: 17.22 */
  set<StateMachine *, ltstatemachine> machines;

  /* variables */
  PROTOCOL_VERSION_T    ForceVersion;   /* 17.12, 17.16.1 */
  BRIDGE_ID             BrId;           /* 17.17.2 */
  TIMEVALUES_T          BrTimes;        /* 17.17.4 */
  PORT_ID               rootPortId;     /* 17.17.5 */
  PRIO_VECTOR_T         rootPrio;       /* 17.17.6 */
  TIMEVALUES_T          rootTimes;      /* 17.17.7 */
  
  int                   vlan_id;        /* let's say: tag */
  char*                 name;           /* name of the VLAN, maily for debugging */
  UID_STP_MODE_T        admin_state;    /* STP_DISABLED or STP_ENABLED; type see in UiD */

  unsigned long         timeSince_Topo_Change; /* 14.8.1.1.3.b */
  unsigned long         Topo_Change_Count;     /* 14.8.1.1.3.c */
  unsigned char         Topo_Change;           /* 14.8.1.1.3.d */

  friend class RapidSpanningTree;
  friend class Port;
  friend class RoleSelection;
};


#endif // _STP_MACHINE_H__

