/************************************************************************ 
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * This file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 **********************************************************************/

 /* STP machine instance : bridge per VLAN: 17.17 */

#include "global.h"
#include "STPM.h"
#include "Port.h"
#include "Log.h"
#include "RapidSpanningTree.h"
#include "RoleSelection.h"
#include "Switch.h"
#include <string.h>
#include <iostream>
#include "stp_to.h" /* for STP_OUT_flush_lt */

unsigned TxHoldCount = 3;

int STPM::_individualCallback(StateMachine *sm, CallbackType ct)
{
  int iret = 0;
  switch (ct) {
  case CB_init_machine:
    sm->State = BEGIN;
    sm->EnterState();
    /* we don't update iret in this case */
    break;
  case CB_change_state:
    iret = sm->ChangeState();
    break;
  case CB_check_condition:
    iret = sm->CheckCondition();
    break;
  default:
    assert(0);
  }
  return iret;
}

int STPM::_stp_stpm_iterate_machines (CallbackType ct, bool exit_on_non_zero_ret)
{
  int                    mret = 0;

  /* state machines per bridge */
  set<StateMachine *>::iterator smi;
  for (smi = machines.begin(); smi != machines.end(); smi++) {
    int iret = _individualCallback(*smi, ct);
    if (exit_on_non_zero_ret && iret)
      return iret;
    else
      mret += iret;
  }

  /* state machines per port */
  for (unsigned i = 1; i < Ports.size(); i++) {
    LogContext lc(Ports[i]->port_name);
    for (smi = Ports[i]->machines.begin(); smi != Ports[i]->machines.end(); smi++) {
      int iret = _individualCallback(*smi, ct);
      if (exit_on_non_zero_ret && iret)
        return iret;
      else
        mret += iret;
    }
  }
  
  return mret;
}

void STPM::_stp_stpm_init_data()
{
  STP_VECT_create (&this->rootPrio,
                   &this->BrId,
                   0,
                   &this->BrId,
                   0, 0);

  this->BrTimes.MessageAge = 0;

  STP_copy_times (&this->rootTimes, &this->BrTimes);
}

unsigned char STPM::_check_topoch()
{
  for (unsigned i =1; i < Ports.size(); i++) {
    if (Ports[i]->tcWhile) {
      return 1;
    }
  }
  return 0;
}

void STPM::STP_stpm_one_second ()
{
  if (STP_ENABLED != this->admin_state) {
    return;
  }
  stp_trace("stpm_one_second");

  for (unsigned p = 1; p < Ports.size(); p++) {
    if (Ports[p]->txCount >= TxHoldCount) {
      stp_trace("port %d txhold in effect: txCount is %d (about to be decremented)",
                p, Ports[p]->txCount);
    }
    for (int iii = 0; iii < TIMERS_NUMBER; iii++) {
      if (*(Ports[p]->timers[iii]) > 0) {
        (*Ports[p]->timers[iii])--;
      }
    }
    Ports[p]->uptime++;
  }

  STP_stpm_update();
  this->Topo_Change = _check_topoch();
  if (this->Topo_Change) {
    this->Topo_Change_Count++;
    this->timeSince_Topo_Change = 0;
  } else {
    this->Topo_Change_Count = 0;
    this->timeSince_Topo_Change++;
  }
}

STPM::STPM(int vlan_id, char* cname, RapidSpanningTree *owner)
  : Owner(owner)
{
  this->admin_state = STP_DISABLED;
  
  this->vlan_id = vlan_id;
  if (cname) {
    this->name = strdup(cname);
  }

  rolesel = new RoleSelection(this);
  machines.insert(rolesel);
}

int STPM::STP_stpm_enable (UID_STP_MODE_T admin_state)
{
  int rc = 0;

  if (admin_state == this->admin_state) {
    /* nothing to do :) */
    return 0;
  }

  if (STP_ENABLED == admin_state) {
    rc = STP_stpm_start();
    this->admin_state = admin_state;
  } else {
    this->admin_state = admin_state;
    STP_stpm_stop();
  }
  
  return rc;
}

STPM::~STPM()
{
  STP_stpm_enable(STP_DISABLED);
  
  set<StateMachine *>::iterator i;
  for (i = machines.begin(); i != machines.end(); i++) {
    delete *i;
  }
  machines.clear();

  for (unsigned i = 1; i < Ports.size(); i++) {
    delete Ports[i];
    Ports[i] = NULL;
  }
  Ports.clear();

  free(name);
  name = "nada";
}

int STPM::STP_stpm_start()
{
  if (Ports.size() < 2) {
    // a size() of 1 means that there's just port 0 there, and that's
    // not really a port at all
    return STP_There_Are_No_Ports;
  }

  if (! STP_compute_bridge_id()) {/* can't compute bridge id ? :( */
    return STP_Cannot_Compute_Bridge_Prio;
  }

  /* check, that the stpm has unique bridge Id */
  if (0 != Owner->STP_check_bridge_priority (this)) {
    /* there is an enabled bridge with same ID :( */
    return STP_Invalid_Bridge_Priority;
  }

  _stp_stpm_init_data ();

  for (unsigned i = 1; i < Ports.size(); i++) {
    Ports[i]->STP_port_init (this, true);
  }

#ifndef STRONGLY_SPEC_802_1W
  /* A. see comment near STRONGLY_SPEC_802_1W in topoch.c */
  /* B. port=0 here means: delete for all ports */
  stp_trace("%s (all, start stpm)", "clearFDB");

  STP_OUT_flush_lt (0, this->vlan_id, LT_FLASH_ONLY_THE_PORT, "start stpm");
#endif

  _stp_stpm_iterate_machines (CB_init_machine, false);
  STP_stpm_update ();

  return 0;
}

void STPM::STP_stpm_stop ()
{
}

int STPM::STP_stpm_update () /* returns number of loops */
{
  bool     need_state_change;
  int      number_of_loops = 0;

  stp_trace("STP_stpm_update");

  need_state_change = false; 
  
  for (;;) {/* loop until not need changes */
    need_state_change = _stp_stpm_iterate_machines (CB_check_condition, true);
    if (! need_state_change) return number_of_loops;

    number_of_loops++;
    /* here we know, that at least one stater must be
       updated (it has changed state) */
    number_of_loops += _stp_stpm_iterate_machines (CB_change_state, false);
  }
  return number_of_loops;
}

BRIDGE_ID *STPM::STP_compute_bridge_id()
{
  Port *min_num_port = NULL;
  int              port_index = 0;

  for (unsigned i = 1; i < Ports.size(); i++) {
    if (! port_index || Ports[i]->port_index < port_index) {
      min_num_port = Ports[i];
      port_index = Ports[i]->port_index;
    }
  }

  if (! min_num_port) return NULL; /* IMHO, it may not be */

  Owner->GetSwitch()->GetPortMac(min_num_port->port_index,
                                 BrId.addr);
  return &BrId;
}

void STPM::STP_stpm_update_after_bridge_management ()
{
  for (unsigned i = 1; i < Ports.size(); i++) {
    Ports[i]->reselect = true;
    Ports[i]->selected = false;
  }
}

const char* STPM::STP_stpm_get_port_name_by_id(PORT_ID port_id)
{
  for (unsigned i = 1; i < Ports.size(); i++) {
    if (port_id == Ports[i]->port_id) {
        return Ports[i]->port_name;
    }
  }
  return "Undef?";
}

bool STPM::compute_all_synced(Port *except)
{
    for (unsigned i = 1; i < Ports.size(); i++) {
      if (Ports[i] == except) {
        continue;
      }
      if (!Ports[i]->synced) {
        return false;
      }
    }
    return true;
}

bool STPM::compute_re_rooted(Port *except)
{
  for (unsigned i = 1; i < Ports.size(); i++) {
    if (Ports[i] == except) {
      continue;
    }
    if (!Ports[i]->rrWhile) {
      return false;
    }
  }
  return true;
}

void STPM::setSyncBridge()
{
  for (unsigned i = 1; i < Ports.size(); i++) {
    Ports[i]->sync = true;  /* in ROOT_PROPOSED (setSyncBridge) */
  }
}

void STPM::setReRootBridge()
{
  for (unsigned i = 1; i < Ports.size(); i++) {
    Ports[i]->reRoot = true;  /* In setReRootBridge */
  }
}

void STPM::setTcProp(Port *except)
{
  for (unsigned i = 1; i < Ports.size(); i++) {
    if (Ports[i] != except) {
      Ports[i]->tcProp = true;
    }
  }
}
