/************************************************************************ 
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * This file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 **********************************************************************/

/* Port Role Transitions state machine : 17.24 */
 
#include "global.h"
#include "RoleTransitions.h"
#include "STPM.h"
#include "Port.h"

#define STATES { \
   CHOOSE(INIT_PORT),       \
   CHOOSE(BLOCK_PORT),      \
   CHOOSE(BLOCKED_PORT),    \
   CHOOSE(BACKUP_PORT),     \
   CHOOSE(ROOT_PROPOSED),   \
   CHOOSE(ROOT_AGREED),     \
   CHOOSE(REROOT),      \
   CHOOSE(ROOT_PORT),       \
   CHOOSE(REROOTED),        \
   CHOOSE(ROOT_LEARN),      \
   CHOOSE(ROOT_FORWARD),    \
   CHOOSE(DESIGNATED_PROPOSE),  \
   CHOOSE(DESIGNATED_SYNCED),   \
   CHOOSE(DESIGNATED_RETIRED),  \
   CHOOSE(DESIGNATED_PORT), \
   CHOOSE(DESIGNATED_LISTEN),   \
   CHOOSE(DESIGNATED_LEARN),    \
   CHOOSE(DESIGNATED_FORWARD),  \
}

#define GET_STATE_NAME RoleTransitions::GetStateName
#include "choose.h"

RoleTransitions::RoleTransitions(Port *p)
  : StateMachine("RoleTransitions"), owner(p)
{
  // empty
}

void RoleTransitions::setSyncBridge()
{
  owner->owner->setSyncBridge();
}

void RoleTransitions::setReRootBridge()
{
  owner->owner->setReRootBridge();
}

bool RoleTransitions::compute_all_synced()
{
  // owner's owner is an STPM
  return owner->owner->compute_all_synced(owner);
}

bool RoleTransitions::compute_re_rooted()
{
  return owner->owner->compute_re_rooted(owner);
}

void RoleTransitions::EnterState()
{
  Port *port = owner;
  STPM *stpm = owner->owner;

  switch (State) {
    case BEGIN:
    case INIT_PORT:
#if 0 /* due 802.1y Z.4 */
      port->role = DisabledPort;
#else
      port->role = port->selectedRole = DisabledPort;
      port->reselect = true;
#endif
      port->synced = false; /* in INIT */
      port->sync = true; /* in INIT */
      port->reRoot = true; /* in INIT_PORT */
      port->rrWhile = stpm->GetRootTimes().ForwardDelay;
      port->fdWhile = stpm->GetRootTimes().ForwardDelay;
      port->rbWhile = 0; 
      port->learn = false;  // acm - valgrind tells me learn wasn't initialized
      if (debug) {
        port->STP_port_trace_flags ("after init");
      }
      break;
    case BLOCK_PORT:
      port->role = port->selectedRole;
      port->learn =
      port->forward = false;
      break;
    case BLOCKED_PORT:
      port->fdWhile = stpm->GetRootTimes().ForwardDelay;
      port->synced = true; /* In BLOCKED_PORT */
      port->rrWhile = 0;
      port->sync = port->reRoot = false; /* BLOCKED_PORT */
      break;
    case BACKUP_PORT:
      port->rbWhile = 2 * stpm->GetRootTimes().HelloTime;
      break;

    /* 17.23.2 */
    case ROOT_PROPOSED:
      setSyncBridge();
      port->proposed = false;
      if (debug) {
        port->STP_port_trace_flags ("ROOT_PROPOSED");
      }
      break;
    case ROOT_AGREED:
      port->proposed = port->sync = false; /* in ROOT_AGREED */
      port->synced = true; /* In ROOT_AGREED */
      port->newInfo = true;
      if (debug) {
        port->STP_port_trace_flags ("ROOT_AGREED");
      }
      break;
    case REROOT:
      setReRootBridge();
      if (debug) {
        port->STP_port_trace_flags ("REROOT");
      }
      break;
    case ROOT_PORT:
      port->role = RootPort;
      port->rrWhile = stpm->GetRootTimes().ForwardDelay;
      if (debug) {
        port->STP_port_trace_flags ("ROOT_PORT");
      }
      break;
    case REROOTED:
      port->reRoot = false; /* In REROOTED */
      if (debug) {
        port->STP_port_trace_flags ("REROOTED");
      }
      break;
    case ROOT_LEARN:
      port->fdWhile = stpm->GetRootTimes().ForwardDelay;
      port->learn = true;
      if (debug) {
        port->STP_port_trace_flags ("ROOT_LEARN");
      }
      break;
    case ROOT_FORWARD:
      port->fdWhile = 0;
      port->forward = true;
      if (debug) {
        port->STP_port_trace_flags ("ROOT_FORWARD");
      }
      break;

    /* 17.23.3 */
    case DESIGNATED_PROPOSE:
      port->proposing = true; /* in DESIGNATED_PROPOSE */
      port->newInfo = true;
      if (debug) {
        port->STP_port_trace_flags ("DESIGNATED_PROPOSE");
      }
      break;
    case DESIGNATED_SYNCED:
      port->rrWhile = 0;
      port->synced = true; /* DESIGNATED_SYNCED */
      port->sync = false; /* DESIGNATED_SYNCED */
      if (debug) {
        port->STP_port_trace_flags ("DESIGNATED_SYNCED");
      }
      break;
    case DESIGNATED_RETIRED:
      port->reRoot = false; /* DESIGNATED_RETIRED */
      if (debug) {
        port->STP_port_trace_flags ("DESIGNATED_RETIRED");
      }
      break;
    case DESIGNATED_PORT:
      port->role = DesignatedPort;
      if (debug) {
        port->STP_port_trace_flags ("DESIGNATED_PORT");
      }
      break;
    case DESIGNATED_LISTEN:
      port->learn = port->forward = false;
      port->fdWhile = stpm->GetRootTimes().ForwardDelay;
      if (debug) {
        port->STP_port_trace_flags ("DESIGNATED_LISTEN");
      }
      break;
    case DESIGNATED_LEARN:
      port->learn = true;
      port->fdWhile = stpm->GetRootTimes().ForwardDelay;
      if (debug) {
        port->STP_port_trace_flags ("DESIGNATED_LEARN");
      }
      break;
    case DESIGNATED_FORWARD:
      port->forward = true;
      port->fdWhile = 0;
      if (debug) {
        port->STP_port_trace_flags ("DESIGNATED_FORWARD");
      }
      break;
  };
}
    
bool RoleTransitions::CheckCondition()
{
  Port *port = owner;
  STPM *stpm = owner->owner;
  bool                      allSynced;
  bool                      allReRooted;

  if (BEGIN == State) {
    return Hop2State(INIT_PORT);
  }

  if (port->role != port->selectedRole &&
      port->selected &&
      ! port->updtInfo) {
    switch (port->selectedRole) {
      case DisabledPort:
      case AlternatePort:
      case BackupPort:
#if 0 /* def STP_DBG */
        if (this->debug) {
          stp_trace ("hop to BLOCK_PORT role=%d selectedRole=%d",
                                (int) port->role, (int) port->selectedRole);
        }
#endif
        return Hop2State(BLOCK_PORT);
      case RootPort:
        return Hop2State(ROOT_PORT);
      case DesignatedPort:
        return Hop2State(DESIGNATED_PORT);
      default:
        return false;
    }
  }

  switch (State) {
    /* 17.23.1 */
    case INIT_PORT:
      return Hop2State(BLOCK_PORT);
    case BLOCK_PORT:
      if (!port->selected || port->updtInfo) {
        break;
      }
      if (!port->learning && !port->forwarding) {
        return Hop2State(BLOCKED_PORT);
      }
      break;
    case BLOCKED_PORT:
      if (!port->selected || port->updtInfo) {
        break;
      }
      if (port->fdWhile != stpm->GetRootTimes().ForwardDelay ||
          port->sync                ||
          port->reRoot              ||
          !port->synced) {
        return Hop2State(BLOCKED_PORT);
      }
      if (port->rbWhile != (unsigned)2 * stpm->GetRootTimes().HelloTime &&
          port->role == BackupPort) {
        return Hop2State(BACKUP_PORT);
      }
      break;
    case BACKUP_PORT:
      return Hop2State(BLOCKED_PORT);

    /* 17.23.2 */
    case ROOT_PROPOSED:
      return Hop2State(ROOT_PORT);
    case ROOT_AGREED:
      return Hop2State(ROOT_PORT);
    case REROOT:
      return Hop2State(ROOT_PORT);
    case ROOT_PORT:
      if (!port->selected || port->updtInfo) {
        break;
      }
      if (!port->forward && !port->reRoot) {
        return Hop2State(REROOT);
      }
      allSynced = compute_all_synced();
      if ((port->proposed && allSynced) ||
          (!port->synced && allSynced)) {
        return Hop2State(ROOT_AGREED);
      }
      if (port->proposed && !port->synced) {
        return Hop2State(ROOT_PROPOSED);
      }

      allReRooted = compute_re_rooted();
      if ((!port->fdWhile || 
           ((allReRooted && !port->rbWhile) && stpm->GetForceVersion() >=2)) &&
          port->learn && !port->forward) {
        return Hop2State(ROOT_FORWARD);
      }
      if ((!port->fdWhile || 
           ((allReRooted && !port->rbWhile) && stpm->GetForceVersion() >=2)) &&
          !port->learn) {
        return Hop2State(ROOT_LEARN);
      }

      if (port->reRoot && port->forward) {
        return Hop2State(REROOTED);
      }
      if (port->rrWhile != stpm->GetRootTimes().ForwardDelay) {
        return Hop2State(ROOT_PORT);
      }
      break;
    case REROOTED:
      return Hop2State(ROOT_PORT);
    case ROOT_LEARN:
      return Hop2State(ROOT_PORT);
    case ROOT_FORWARD:
      return Hop2State(ROOT_PORT);

    /* 17.23.3 */
    case DESIGNATED_PROPOSE:
      return Hop2State(DESIGNATED_PORT);
    case DESIGNATED_SYNCED:
      return Hop2State(DESIGNATED_PORT);
    case DESIGNATED_RETIRED:
      return Hop2State(DESIGNATED_PORT);
    case DESIGNATED_PORT:
      if (!port->selected || port->updtInfo) {
        break;
      }
      if (!port->forward && !port->agreed && !port->proposing && !port->operEdge) {
        return Hop2State(DESIGNATED_PROPOSE);
      }

      if (!port->rrWhile && port->reRoot) {
        return Hop2State(DESIGNATED_RETIRED);
      }
      
      if (!port->learning && !port->forwarding && !port->synced) {
        return Hop2State(DESIGNATED_SYNCED);
      }

      if (port->agreed && !port->synced) {
        return Hop2State(DESIGNATED_SYNCED);
      }

      if (port->operEdge && !port->synced) {
        return Hop2State(DESIGNATED_SYNCED);
      }

      if (port->sync && port->synced) {
        return Hop2State(DESIGNATED_SYNCED);
      }

      if ((!port->fdWhile || port->agreed || port->operEdge) &&
          (!port->rrWhile  || !port->reRoot) &&
          !port->sync &&
          (port->learn && !port->forward)) {
        return Hop2State(DESIGNATED_FORWARD);
      }
      if ((!port->fdWhile || port->agreed || port->operEdge) &&
          (!port->rrWhile  || !port->reRoot) &&
          !port->sync && !port->learn) {
        return Hop2State(DESIGNATED_LEARN);
      }
      if (((port->sync && !port->synced) ||
           (port->reRoot && port->rrWhile)) &&
          !port->operEdge && (port->learn || port->forward)) {
        return Hop2State(DESIGNATED_LISTEN);
      }
      break;
    case DESIGNATED_LISTEN:
      return Hop2State(DESIGNATED_PORT);
    case DESIGNATED_LEARN:
      return Hop2State(DESIGNATED_PORT);
    case DESIGNATED_FORWARD:
      return Hop2State(DESIGNATED_PORT);
  };

  return false;
}


