/************************************************************************ 
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * This file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 **********************************************************************/

/* Port Role Selection state machine : 17.22 */

#include "global.h"
#include "RoleSelection.h"
#include "STPM.h"
#include "Log.h"
#include "RapidSpanningTree.h"
#include "Switch.h"

#include <sstream>

#define STATES { \
  CHOOSE(INIT_BRIDGE),      \
  CHOOSE(ROLE_SELECTION),   \
}

#define GET_STATE_NAME RoleSelection::GetStateName
#include "choose.h"

RoleSelection::RoleSelection(STPM *cowner) : 
  StateMachine("RoleSelection"), owner(cowner)
{
  // empty
}

bool RoleSelection::_is_backup_port (Port *port, STPM *stpm)
{
  if (!STP_VECT_compare_bridge_id
      (&port->portPrio.design_bridge, &stpm->BrId)) {
#if 0 /* def STP_DBG */
    if (port->info->debug) {
      STP_VECT_br_id_print ("portPrio.design_bridge",
                            &port->portPrio.design_bridge, True);
      STP_VECT_br_id_print ("            this->BrId",
                            &stpm->BrId, True);
    }
#endif
    return true;
  } else {
    return false;
  }
}

void RoleSelection::setRoleSelected (char *reason, STPM *stpm, Port *port, 
                                     PORT_ROLE_T newRole)
{
  char* new_role_name;

  port->selectedRole = newRole;

  if (newRole == port->role)
    return;

  switch (newRole) {
    case DisabledPort:
      new_role_name = "Disabled";
      break;
    case AlternatePort:
      new_role_name = "Alternate";
      break;
    case BackupPort:
      new_role_name = "Backup";
      break;
    case RootPort:
      new_role_name = "Root";
      break;
    case DesignatedPort:
      new_role_name = "Designated";
      break;
    case NonStpPort:
      new_role_name = "NonStp";
      port->role = newRole;
      break;
    default:
      stp_trace ("%s-%s:port %s => Unknown (%d ?)",
                 reason, stpm->name, port->port_name, (int) newRole);
      return;
  }

  if (port->roletrns->GetDebug()) {
    stp_trace ("%s(%s-%s) => %s",
               reason, stpm->name, port->port_name, new_role_name);
  }
}

void RoleSelection::updtRoleDisableBridge (STPM *stpm)
{               /* 17.10.20 */
  for (unsigned i = 1; i < stpm->Ports.size(); i++) {
    stpm->Ports[i]->selectedRole = DisabledPort;
  }
}

void RoleSelection::clearReselectBridge (STPM *stpm)
{               /* 17.19.1 */
  for (unsigned i = 1; i < stpm->Ports.size(); i++) {
    stpm->Ports[i]->reselect = false;
  }
}

void RoleSelection::updtRootPrio()
{
  PRIO_VECTOR_T rootPathPrio;   /* 17.4.2.2 */
  STPM *stpm;
  unsigned int dm;

  stpm = this->owner;

  for (unsigned i = 1; i < stpm->Ports.size(); i++) {
    Port *port = stpm->Ports[i];
    if (port->admin_non_stp) {
      continue;
    }
    if (Disabled == port->infoIs) {
      continue;
    }
    if (Aged == port->infoIs) {
      stp_trace("updtRootPrio skipping aged port %s", port->port_name);
      continue;
    }
    if (Mine == port->infoIs) {
      stp_trace("updtRootPrio skipping 'info is mine' port %s", port->port_name);
      continue;
    }

    STP_VECT_copy (&rootPathPrio, &port->portPrio);
    rootPathPrio.root_path_cost += port->operPCost;

    stp_trace("updtRootPrio compairing port %s with best known priority", 
	      port->port_name);
    if (STP_VECT_compare_vector (&rootPathPrio, &stpm->rootPrio) < 0) {
      STP_VECT_copy (&stpm->rootPrio, &rootPathPrio);
      STP_copy_times (&stpm->rootTimes, &port->portTimes);
      dm = (8 +  stpm->rootTimes.MaxAge) / 16;
      if (!dm)
        dm = 1;
      stpm->rootTimes.MessageAge += dm;
      if (port->roletrns->GetDebug()) {
          stp_trace ("updtRootPrio: dm=%d rootTimes.MessageAge=%d on port %s",
                 (int) dm, (int) stpm->rootTimes.MessageAge,
                 port->port_name);
      }
    }
  }
}

void RoleSelection::updtRolesBridge()
{               /* 17.19.21 */
  STPM *stpm;
  PORT_ID old_root_port; /* for tracing of root port changing */

  stpm = this->owner;
  old_root_port = stpm->rootPortId;
  PRIO_VECTOR_T oldRootPrio = stpm->rootPrio;

  STP_VECT_create (&stpm->rootPrio, &stpm->BrId, 0, &stpm->BrId, 0, 0);
  STP_copy_times (&stpm->rootTimes, &stpm->BrTimes);
  stpm->rootPortId = 0;

  updtRootPrio();
  for (unsigned i = 1; i < stpm->Ports.size(); i++) {
    Port *port = stpm->Ports[i];
    if (port->admin_non_stp) {
      continue;
    }
    STP_VECT_create (&port->designPrio,
             &stpm->rootPrio.root_bridge,
             stpm->rootPrio.root_path_cost,
             &stpm->BrId, port->port_id, port->port_id);
    STP_copy_times (&port->designTimes, &stpm->rootTimes);

#if 0
    if (port->roletrns->debug) {
      STP_VECT_br_id_print ("ch:designPrio.design_bridge",
                            &port->designPrio.design_bridge, True);
    }
#endif
  }

  stpm->rootPortId = stpm->rootPrio.bridge_port;

  if (old_root_port != stpm->rootPortId) {
    if (! stpm->rootPortId) {
      stp_trace ("IMP I AM root", stpm->name);
    } else {
      ostringstream br;
      br << stpm->rootPrio.root_bridge;
      string sbr = br.str();
      stp_trace ("IMP new root port: %s bridge %s cost %u",
                 stpm->STP_stpm_get_port_name_by_id (stpm->rootPortId),
                 sbr.c_str(),
                 stpm->rootPrio.root_path_cost);
    }
  } else if (oldRootPrio.root_path_cost != stpm->rootPrio.root_path_cost ||
                oldRootPrio.root_bridge != stpm->rootPrio.root_bridge) {
    // if we haven't changed root ports, but our priority vector has
    // changed, let's log it
    ostringstream ost;
    ost << "IMP updated prio: " << stpm->rootPrio.root_bridge << " cost " 
        << stpm->rootPrio.root_path_cost;
    string b = ost.str();
    stp_trace(b.c_str());
  }

  for (unsigned i = 1; i < stpm->Ports.size(); i++) {
    Port *port = stpm->Ports[i];
    if (port->admin_non_stp) {
      setRoleSelected ("Non", stpm, port, NonStpPort);
      port->forward = port->learn = true;
      continue;
    }

    switch (port->infoIs) {
      case Disabled:
        setRoleSelected ("Dis", stpm, port, DisabledPort);
        break;
      case Aged:
        setRoleSelected ("Age", stpm, port, DesignatedPort);
        port->updtInfo = true;
        break;
      case Mine:
        setRoleSelected ("Mine", stpm, port, DesignatedPort);
        if (0 != STP_VECT_compare_vector (&port->portPrio,
                      &port->designPrio) ||
            0 != STP_compare_times (&port->portTimes,
                  &port->designTimes)) {
            port->updtInfo = true;
        }
        break;
      case Received:
        if (stpm->rootPortId == port->port_id) {
          setRoleSelected ("Rec", stpm, port, RootPort);
        } else if (STP_VECT_compare_vector (&port->designPrio, &port->portPrio) < 0) {
          /* Note: this important piece has been inserted after
           * discussion with Mick Sieman and reading 802.1y Z1 */
          setRoleSelected ("Rec", stpm, port, DesignatedPort);
          port->updtInfo = true;
          break;
        } else {
          if (_is_backup_port (port, stpm)) {
            setRoleSelected ("rec", stpm, port, BackupPort);
          } else {
            setRoleSelected ("rec", stpm, port, AlternatePort);
          }
        }
        port->updtInfo = false;
        break;
      default:
        stp_trace ("undef infoIs=%d", (int) port->infoIs);
        break;
    }
  }

}


bool RoleSelection::setSelectedBridge()
{
  for (unsigned i = 1; i < owner->Ports.size(); i++) {
    Port *port = owner->Ports[i];
    if (port->reselect) {
      if (debug) {
        stp_trace ("setSelectedBridge: TRUE=reselect on port %s", port->port_name);
      }
      return false;
    }
  }
  for (unsigned i = 1; i < owner->Ports.size(); i++) {
    owner->Ports[i]->selected = true;
  }
  return true;
}

void RoleSelection::EnterState()
{
  switch (State) {
    case BEGIN:
    case INIT_BRIDGE:
      owner->rootPortId = 0;  // acm - added this to prevent an unitialized mem read
      updtRoleDisableBridge (owner);
      break;
    case ROLE_SELECTION:
      clearReselectBridge (owner);
      updtRolesBridge();
      setSelectedBridge();
      break;
  }
}

bool RoleSelection::CheckCondition()
{
  if (BEGIN == State) {
    Hop2State(INIT_BRIDGE);
  }

  switch (State) {
    case BEGIN:
      return Hop2State (INIT_BRIDGE);
    case INIT_BRIDGE:
      return Hop2State (ROLE_SELECTION);
    case ROLE_SELECTION:
      for (unsigned i = 1; i < owner->Ports.size(); i++) {
        if (owner->Ports[i]->reselect) {
          stp_trace ("reselect on port %s", owner->Ports[i]->port_name); 
          return Hop2State (ROLE_SELECTION);
        }
      }
      break;
  }
  return false;
}

void RoleSelection::STP_rolesel_update_stpm()
{
  PRIO_VECTOR_T rootPathPrio;   /* 17.4.2.2 */

  stp_trace ("%s", "??? STP_rolesel_update_stpm ???");
  STP_VECT_create (&rootPathPrio, &owner->BrId, 0, &owner->BrId, 0, 0);

  if (!owner->rootPortId ||
      STP_VECT_compare_vector (&rootPathPrio, &owner->rootPrio) < 0) {
    STP_VECT_copy (&owner->rootPrio, &rootPathPrio);
  }

  for (unsigned i = 1; i < owner->Ports.size(); i++) {
    Port *port = owner->Ports[i];
    STP_VECT_create (&port->designPrio,
             &owner->rootPrio.root_bridge,
             owner->rootPrio.root_path_cost,
             &owner->BrId, port->port_id, port->port_id);
    if (Received != port->infoIs || owner->rootPortId == port->port_id) {
      STP_VECT_copy (&port->portPrio, &port->designPrio);
    }
    port->reselect = true;
    port->selected = false;
  }
}

