/************************************************************************ 
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * This file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 **********************************************************************/

#include "global.h"
#include "RapidSpanningTree.h"
#include "Port.h"
#include "Log.h"
#include "stp_to.h"

#include <netinet/in.h> // for ntohs

// XXXacm - remove this!
unsigned max_port = 1024;

RapidSpanningTree::RapidSpanningTree(Switch *owner)
  : Owner(owner)
{
  // empty
}

RapidSpanningTree::~RapidSpanningTree()
{
  STPMHash::iterator i;
  for (i = STPMs.begin(); i != STPMs.end(); i++) {
    delete i->second;
  }
  STPMs.clear();
}

STPM *RapidSpanningTree::stp_in_stpm_create (int vlan_id, 
                                             char* name,
                                             BITMAP_T* port_bmp, 
                                             int* err_code)
{
  unsigned port_index;
  STPM *stpm;

  /* stp_trace ("stp_in_stpm_create(%s)", name); */
  stpm = stpapi_stpm_find (vlan_id);
  if (stpm) { /* it had just been created :( */
    *err_code = STP_Nothing_To_Do;
    return stpm;
  }

  // stpm = STP_stpm_create (vlan_id, name);
  stpm = new STPM(vlan_id, name, this);
  assert(STPMs.find(vlan_id) == STPMs.end());
  STPMs[vlan_id] = stpm;

  if (!stpm) { /* can't create stpm :( */
    *err_code = STP_Cannot_Create_Instance_For_Vlan;
    return NULL;
  }

  stpm->Ports.push_back(NULL); // there is no port 0
  for (port_index = 1; port_index <= max_port; port_index++) {
    if (BitmapGetBit(port_bmp, (port_index - 1))) {
      Port *port = new Port(stpm, port_index);
      assert(port);
      stpm->Ports.push_back(port);
      assert(stpm->Ports[port_index] == port);
    }
  }

  *err_code = STP_OK;
  return stpm;
}

int RapidSpanningTree::_stp_in_stpm_enable (int vlan_id, char* name,
                                            BITMAP_T* port_bmp,
                                            UID_STP_MODE_T admin_state)
{
  STPM *stpm;
  bool created_here = false;
  int rc, err_code;

  /* stp_trace ("_stp_in_stpm_enable(%s)", name); */
  stpm = stpapi_stpm_find (vlan_id);
  
  if (STP_DISABLED != admin_state) {
    if (! vlan_id) { /* STP_IN_stop_all (); */
      STPMHash::iterator i;
      for (i = STPMs.begin(); i != STPMs.end(); i++) {
        STPM *stpm2 = i->second;
        if (STP_DISABLED != stpm2->admin_state) {
            STP_OUT_set_hardware_mode (stpm2->vlan_id, STP_DISABLED);
            stpm2->STP_stpm_enable (STP_DISABLED);
          }
        }
    }
  }

  if (!stpm) { /* it had not yet been created */
    if (STP_ENABLED == admin_state) {/* try to create it */
      stp_trace ("implicit create to vlan '%s'", name);
      stpm = stp_in_stpm_create (vlan_id, name, port_bmp, &err_code);
      if (!stpm) {
        stp_trace ("implicit create to vlan '%s' failed", name);
        return STP_Imlicite_Instance_Create_Failed;
      }
      created_here = true;
    } else {/* try to disable nothing ? */
      return 0;
    }
  }

  if (stpm->get_admin_state() == admin_state) { /* nothing to do :) */
    return 0;
  }

  rc = stpm->STP_stpm_enable (admin_state);
  if (! rc) {
    STP_OUT_set_hardware_mode (vlan_id, admin_state);
  }

  if (rc && created_here) {
    STPMs.erase(vlan_id);
    delete stpm;
    stpm = NULL;
  }
    
  return rc;
}


STPM *RapidSpanningTree::stpapi_stpm_find (int vlan_id)
{
  STPMHash::iterator i = STPMs.find(vlan_id);
  if (i == STPMs.end()) {
    return NULL;
  }
  return i->second;
}

Port *RapidSpanningTree::_stpapi_port_find (STPM *t, unsigned port_index)
{
  assert(t);
  if (port_index > t->Ports.size()) {
    return NULL;
  }
  return t->Ports[port_index];
}


void RapidSpanningTree::_conv_br_id_2_uid (IN BRIDGE_ID* f, 
					   OUT UID_BRIDGE_ID_T* t)
{
  memcpy (t, f, sizeof (UID_BRIDGE_ID_T));
}

int RapidSpanningTree::_check_stpm_config (IN UID_STP_CFG_T* uid_cfg)
{
  if (uid_cfg->bridge_priority < MIN_BR_PRIO) {
    stp_trace ("%d bridge_priority small", (int) uid_cfg->bridge_priority);
    return STP_Small_Bridge_Priority;
  }

  if (uid_cfg->bridge_priority > MAX_BR_PRIO) {
    stp_trace ("%d bridge_priority large", (int) uid_cfg->bridge_priority);
    return STP_Large_Bridge_Priority;
  }

  if (uid_cfg->hello_time < MIN_BR_HELLOT) {
    stp_trace ("%d hello_time small", (int) uid_cfg->hello_time);
    return STP_Small_Hello_Time;
  }

  if (uid_cfg->hello_time > MAX_BR_HELLOT) {
    stp_trace ("%d hello_time large", (int) uid_cfg->hello_time);
    return STP_Large_Hello_Time;
  }

  if (uid_cfg->max_age < MIN_BR_MAXAGE) {
    stp_trace ("%d max_age small", (int) uid_cfg->max_age);
    return STP_Small_Max_Age;
  }

  if (uid_cfg->max_age > MAX_BR_MAXAGE) {
    stp_trace ("%d max_age large", (int) uid_cfg->max_age);
    return STP_Large_Max_Age;
  }

  if (uid_cfg->forward_delay < MIN_BR_FWDELAY) {
    stp_trace ("%d forward_delay small", (int) uid_cfg->forward_delay);
    return STP_Small_Forward_Delay;
  }

  if (uid_cfg->forward_delay > MAX_BR_FWDELAY) {
    stp_trace ("%d forward_delay large", (int) uid_cfg->forward_delay);
    return STP_Large_Forward_Delay;
  }

  if (2 * (uid_cfg->forward_delay - 1) < uid_cfg->max_age) {
    return STP_Forward_Delay_And_Max_Age_Are_Inconsistent;
  }

  if (uid_cfg->max_age < 2 * (uid_cfg->hello_time + 1)) {
    return STP_Hello_Time_And_Max_Age_Are_Inconsistent;
  }

  return 0;
}

void RapidSpanningTree::_stp_in_enable_port_on_stpm (STPM* stpm, 
                                                     unsigned port_index, 
                                                     bool enable)
{
  Port *port;

  port = _stpapi_port_find (stpm, port_index);
  if (! port) return; 
  if (port->portEnabled == enable) {/* nothing to do :) */
    return;
  }

  port->uptime = 0;
  if (enable) { /* clear port statistics */ 
    port->rx_cfg_bpdu_cnt =
    port->rx_rstp_bpdu_cnt =
    port->rx_tcn_bpdu_cnt = 0;
  }  

  if (port->edge->debug) {
    stp_trace ("Port %s became '%s' adminEdge=%c",
        port->port_name, enable ? "enable" : "disable",
        port->adminEdge ? 'Y' : 'N');
  }

  port->adminEnable = enable;
  port->STP_port_init(stpm, false);

  port->reselect = true;
  port->selected = false;  
}

void RapidSpanningTree::STP_IN_init (unsigned max_port_index)
{
  max_port = max_port_index;
  RSTP_INIT_CRITICAL_PATH_PROTECTIO;
}

int RapidSpanningTree::STP_IN_stpm_get_cfg (IN int vlan_id, OUT UID_STP_CFG_T* uid_cfg)
{
  STPM* stpm;

  uid_cfg->field_mask = 0;
  
  RSTP_CRITICAL_PATH_START;  
  stpm = stpapi_stpm_find (vlan_id);

  if (!stpm) { /* it had not yet been created :( */
    RSTP_CRITICAL_PATH_END;
    return STP_Vlan_Had_Not_Yet_Been_Created;
  }

  if (stpm->admin_state != STP_DISABLED) {
    uid_cfg->field_mask |= BR_CFG_STATE;
  }
  uid_cfg->stp_enabled = stpm->admin_state;

  if (stpm->ForceVersion != 2) {
    uid_cfg->field_mask |= BR_CFG_FORCE_VER;
  }
  uid_cfg->force_version = stpm->ForceVersion;

  if (stpm->BrId.prio != DEF_BR_PRIO) {
    uid_cfg->field_mask |= BR_CFG_PRIO;
  }
  uid_cfg->bridge_priority = stpm->BrId.prio;

  if (stpm->BrTimes.MaxAge != DEF_BR_MAXAGE) {
    uid_cfg->field_mask |= BR_CFG_AGE;
  }
  uid_cfg->max_age = stpm->BrTimes.MaxAge;

  if (stpm->BrTimes.HelloTime != DEF_BR_HELLOT) {
    uid_cfg->field_mask |= BR_CFG_HELLO;
  }
  uid_cfg->hello_time = stpm->BrTimes.HelloTime;

  if (stpm->BrTimes.ForwardDelay != DEF_BR_FWDELAY) {
    uid_cfg->field_mask |= BR_CFG_DELAY;
  }
  uid_cfg->forward_delay = stpm->BrTimes.ForwardDelay;
  
  uid_cfg->hold_time = TxHoldCount;

  RSTP_CRITICAL_PATH_END;
  return 0;
}
    
int RapidSpanningTree::STP_IN_port_get_cfg (int vlan_id, unsigned port_index, 
                                            UID_STP_PORT_CFG_T* uid_cfg)
{
  STPM *stpm;
  Port *port;
  
  RSTP_CRITICAL_PATH_START;
  stpm = stpapi_stpm_find (vlan_id);
    
  if (!stpm) { /* it had not yet been created :( */
    RSTP_CRITICAL_PATH_END;
    return STP_Vlan_Had_Not_Yet_Been_Created;
  }

  port = _stpapi_port_find (stpm, port_index);
  if (! port) {/* port is absent in the stpm :( */
    RSTP_CRITICAL_PATH_END;
    return STP_Port_Is_Absent_In_The_Vlan;
  }

  uid_cfg->field_mask = 0;

  uid_cfg->port_priority = port->port_id >> 8;
  if (uid_cfg->port_priority != DEF_PORT_PRIO)
    uid_cfg->field_mask |= PT_CFG_PRIO;

  uid_cfg->admin_port_path_cost = port->adminPCost;
  if (uid_cfg->admin_port_path_cost != ADMIN_PORT_PATH_COST_AUTO)
    uid_cfg->field_mask |= PT_CFG_COST;

  uid_cfg->admin_point2point = port->adminPointToPointMac;
  if (uid_cfg->admin_point2point != DEF_P2P)
    uid_cfg->field_mask |= PT_CFG_P2P;

  uid_cfg->admin_edge = port->adminEdge;
  if (uid_cfg->admin_edge != DEF_ADMIN_EDGE)
    uid_cfg->field_mask |= PT_CFG_EDGE;
    
  RSTP_CRITICAL_PATH_END;
  return 0;
}

int RapidSpanningTree::STP_IN_port_get_state (IN int vlan_id, INOUT 
                                              UID_STP_PORT_STATE_T* entry)
{
  STPM *stpm;
  Port *port;

  RSTP_CRITICAL_PATH_START;
  stpm = stpapi_stpm_find (vlan_id);

  if (!stpm) { /* it had not yet been created :( */
    RSTP_CRITICAL_PATH_END;
    return STP_Vlan_Had_Not_Yet_Been_Created;
  }

  port = _stpapi_port_find (stpm, entry->port_no);
  if (! port) {/* port is absent in the stpm :( */
    RSTP_CRITICAL_PATH_END;
    return STP_Port_Is_Absent_In_The_Vlan;
  }

  entry->port_id = port->port_id;
  if (DisabledPort == port->role) {
    entry->state = UID_PORT_DISABLED;
  } else if (! port->forward && ! port->learn) {
    entry->state = UID_PORT_DISCARDING;
  } else if (! port->forward && port->learn) {
    entry->state = UID_PORT_LEARNING;
  } else {
    entry->state = UID_PORT_FORWARDING;
  }

  entry->uptime = port->uptime;
  entry->path_cost = port->operPCost;
  _conv_br_id_2_uid (&port->portPrio.root_bridge, &entry->designated_root);
  entry->designated_cost = port->portPrio.root_path_cost;
  _conv_br_id_2_uid (&port->portPrio.design_bridge, &entry->designated_bridge);
  entry->designated_port = port->portPrio.design_port;

  switch (port->role) {
    case DisabledPort:   entry->role = ' '; break;
    case AlternatePort:  entry->role = 'A'; break;
    case BackupPort:     entry->role = 'B'; break;
    case RootPort:       entry->role = 'R'; break;
    case DesignatedPort: entry->role = 'D'; break;
    case NonStpPort:     entry->role = '-'; break;
    default:             entry->role = '?'; break;
  }

  if (DisabledPort == port->role || NonStpPort == port->role) {
    memset (&entry->designated_root, 0, sizeof (UID_BRIDGE_ID_T));
    memset (&entry->designated_bridge, 0, sizeof (UID_BRIDGE_ID_T));
    entry->designated_cost = 0;
    entry->designated_port = port->port_id;
  }

  if (DisabledPort == port->role) {
    entry->oper_point2point = (P2P_FORCE_FALSE == port->adminPointToPointMac) ? 0 : 1;
    entry->oper_edge = port->adminEdge;
    entry->oper_stp_neigb = 0;
  } else {
    entry->oper_point2point = port->operPointToPointMac ? 1 : 0;
    entry->oper_edge = port->operEdge                   ? 1 : 0;
    entry->oper_stp_neigb = port->sendRSTP              ? 0 : 1;
  }
  entry->oper_port_path_cost = port->operPCost;

  entry->rx_cfg_bpdu_cnt = port->rx_cfg_bpdu_cnt;
  entry->rx_rstp_bpdu_cnt = port->rx_rstp_bpdu_cnt;
  entry->rx_tcn_bpdu_cnt = port->rx_tcn_bpdu_cnt;

  entry->fdWhile =       port->fdWhile;      /* 17.15.1 */
  entry->helloWhen =     port->helloWhen;    /* 17.15.2 */
  entry->mdelayWhile =   port->mdelayWhile;  /* 17.15.3 */
  entry->rbWhile =       port->rbWhile;      /* 17.15.4 */
  entry->rcvdInfoWhile = port->rcvdInfoWhile;/* 17.15.5 */
  entry->rrWhile =       port->rrWhile;      /* 17.15.6 */
  entry->tcWhile =       port->tcWhile;      /* 17.15.7 */
  entry->txCount =       port->txCount;      /* 17.18.40 */
  entry->lnkWhile =      port->lnkWhile;

  entry->rcvdInfoWhile = port->rcvdInfoWhile;
  entry->top_change_ack = port->tcAck;
  entry->tc = port->tc;
  
  RSTP_CRITICAL_PATH_END;
  return 0; 
}

int RapidSpanningTree::STP_IN_stpm_get_state (IN int vlan_id, OUT 
                                              UID_STP_STATE_T* entry)
{
  STPM *stpm;

  RSTP_CRITICAL_PATH_START;
  stpm = stpapi_stpm_find (vlan_id);

  if (!stpm) { /* it had not yet been created :( */
    RSTP_CRITICAL_PATH_END;
    return STP_Vlan_Had_Not_Yet_Been_Created;
  }

  strncpy (entry->vlan_name, stpm->name, NAME_LEN);
  entry->vlan_id = stpm->vlan_id;
  _conv_br_id_2_uid (&stpm->rootPrio.root_bridge, &entry->designated_root);
  entry->root_path_cost = stpm->rootPrio.root_path_cost;
  entry->root_port = stpm->rootPortId;
  entry->max_age =       stpm->rootTimes.MaxAge;
  entry->forward_delay = stpm->rootTimes.ForwardDelay;
  entry->hello_time =    stpm->rootTimes.HelloTime;

  _conv_br_id_2_uid (&stpm->BrId, &entry->bridge_id);

  entry->stp_enabled = stpm->admin_state;

  entry->timeSince_Topo_Change = stpm->timeSince_Topo_Change;
  entry->Topo_Change_Count = stpm->Topo_Change_Count;
  entry->Topo_Change = stpm->Topo_Change;
  
  RSTP_CRITICAL_PATH_END;
  return 0;
}

int RapidSpanningTree::STP_IN_stpm_get_name_by_vlan_id (int vlan_id, 
                                                        char* name, 
                                                        size_t buffsize)
{
  STPM *stpm;
  int iret = -1;

  RSTP_CRITICAL_PATH_START;
  STPMHash::iterator shi = STPMs.find(vlan_id);
  if (shi != STPMs.end()) {
    stpm = shi->second;
    if (stpm->name) {
      strncpy (name, stpm->name, buffsize);
    } else {
      memset (name, 0, buffsize);
    }
    iret = 0;
  }
  RSTP_CRITICAL_PATH_END;
  return iret;
}

int RapidSpanningTree::STP_IN_enable_port (unsigned port_index, bool enable)
{
  STPM *stpm;

  RSTP_CRITICAL_PATH_START;
  if (! enable) {
    stp_trace("%s (p%02u, all, %s, '%s')",
        "clearFDB", port_index, "this port", "disable port");
    STP_OUT_flush_lt (port_index, 0, LT_FLASH_ONLY_THE_PORT, "disable port");
  }

  STPMHash::iterator shi;
  for (shi = STPMs.begin(); shi != STPMs.end(); shi++) {
    stpm = shi->second;
    if (STP_ENABLED != stpm->admin_state) continue;
    
    _stp_in_enable_port_on_stpm (stpm, port_index, enable);
    /* acm - this next line used to be commented out.  wonder why */
    stpm->STP_stpm_update();
  }

  RSTP_CRITICAL_PATH_END;
  return 0;
}

int RapidSpanningTree::STP_IN_changed_port_speed (unsigned port_index, 
                                                  long speed)
{
  STPM *stpm;
  Port *port;

  RSTP_CRITICAL_PATH_START;
  STPMHash::iterator shi;
  for (shi = STPMs.begin(); shi != STPMs.end(); shi++) {
    stpm = shi->second;
    if (STP_ENABLED != stpm->admin_state) continue;
    
    port = _stpapi_port_find (stpm, port_index);
    if (! port) continue; 
    port->operSpeed = speed;

    if (port->pcost->debug) {
      stp_trace ("changed operSpeed=%lu", port->operSpeed);
    }

    port->reselect = true;
    port->selected = false;
  }
  RSTP_CRITICAL_PATH_END;
  return 0;
}

int RapidSpanningTree::STP_IN_changed_port_duplex (unsigned port_index)
{
  STPM *stpm;
  Port *port;

  RSTP_CRITICAL_PATH_START;
  STPMHash::iterator shi;
  for (shi = STPMs.begin(); shi != STPMs.end(); shi++) {
    stpm = shi->second;
    if (STP_ENABLED != stpm->admin_state) continue;
    
    port = _stpapi_port_find (stpm, port_index);
    if (! port) {
      continue; 
    }
    if (port->p2p->debug) {
      stp_trace ("STP_IN_changed_port_duplex(%s)", port->port_name);
    }

    port->p2p_recompute = true;
    port->reselect = true;
    port->selected = false;
  }
  RSTP_CRITICAL_PATH_END;
  return 0;
}

int RapidSpanningTree::STP_IN_check_bpdu_header (BPDU_T* bpdu, size_t len)
{
  unsigned short len8023;

  len8023 = ntohs (*(unsigned short *)bpdu->eth.len8023);
  if (len8023 > 1500) {/* big len8023 format :( */
    return STP_Big_len8023_Format;
  }

  if (len8023 < MIN_BPDU) { /* small len8023 format :( */
    return STP_Small_len8023_Format;
  }

  if (len8023 + 14 > (int)len) { /* len8023 format gt len :( */
    return STP_len8023_Format_Gt_Len;
  }

  if (bpdu->eth.dsap != BPDU_L_SAP                 ||
      bpdu->eth.ssap != BPDU_L_SAP                 ||
      bpdu->eth.llc != LLC_UI) {
    /* this is not a proper 802.3 pkt! :( */
    return STP_Not_Proper_802_3_Packet;
  }

  if (bpdu->hdr.protocol[0] || bpdu->hdr.protocol[1]) {
    return STP_Invalid_Protocol;
  }

#if 0
  if (bpdu->hdr.version != BPDU_VERSION_ID) {
    return STP_Invalid_Version;  
  }
#endif
  /* see also 9.3.4: think & TBD :( */
  return 0;
}

int dbg_rstp_deny = 0;

int RapidSpanningTree::STP_IN_rx_bpdu (int vlan_id, unsigned port_index, 
                                       BPDU_T* bpdu, size_t len)
{
  STPM *stpm;
  Port *port;
  int              iret;

  if (1 == dbg_rstp_deny) {
    return 0;
  }

  RSTP_CRITICAL_PATH_START;
  stpm = stpapi_stpm_find (vlan_id);
  if (!stpm) { /*  the stpm had not yet been created :( */
    RSTP_CRITICAL_PATH_END;
    return STP_Vlan_Had_Not_Yet_Been_Created;
  }

  if (STP_DISABLED == stpm->admin_state) {/* the stpm had not yet been enabled :( */
    RSTP_CRITICAL_PATH_END;
    return STP_Had_Not_Yet_Been_Enabled_On_The_Vlan;
  }

  port = _stpapi_port_find (stpm, port_index);
  if (! port) {/* port is absent in the stpm :( */
    stp_trace ("RX bpdu vlan_id=%d port=%u port is absent in the stpm :(", 
               vlan_id, port_index);
    RSTP_CRITICAL_PATH_END;
    return STP_Port_Is_Absent_In_The_Vlan;
  }

  if (port->skip_rx > 0) {
    if (1 == port->skip_rx)
      stp_trace ("port %s stop rx skipping",
                 port->port_name);
    else
      stp_trace ("port %s skip rx %d",
                 port->port_name, port->skip_rx);
    port->skip_rx--;
    RSTP_CRITICAL_PATH_END;
    return STP_Nothing_To_Do;
  }

  if (port->operEdge && ! port->lnkWhile && port->portEnabled) {
    if (port->topoch->debug) {
      stp_trace ("port %s tc=TRUE by operEdge", port->port_name);
    }
    port->tc = true; /* IEEE 802.1y, 17.30 */
  }

  if (! port->portEnabled) {/* port link change indication will come later :( */
    _stp_in_enable_port_on_stpm (stpm, port->port_index, true);
  }
  
  if (port->edge->debug && port->operEdge) {
    stp_trace ("port %s not operEdge !", port->port_name);
  }

  port->operEdge = false;
  port->wasInitBpdu = true;
  
  iret = port->STP_port_rx_bpdu(bpdu, len);
  stpm->STP_stpm_update();
  RSTP_CRITICAL_PATH_END;

  return iret;
}

int RapidSpanningTree::STP_IN_one_second (void)
{
  STPM *stpm;
  int     dbg_cnt = 0;

  RSTP_CRITICAL_PATH_START;
  STPMHash::iterator shi;
  for (shi = STPMs.begin(); shi != STPMs.end(); shi++) {
    stpm = shi->second;
    if (STP_ENABLED == stpm->admin_state) {
      stp_trace ("STP_IN_one_second vlan_id=%d", (int) stpm->vlan_id);
      stpm->STP_stpm_one_second();
      dbg_cnt++;
    }
  }
  
  RSTP_CRITICAL_PATH_END;

  return dbg_cnt;
}

int RapidSpanningTree::STP_IN_stpm_set_cfg (IN int vlan_id,
                                            IN BITMAP_T* port_bmp,
                                            IN UID_STP_CFG_T* uid_cfg)
{
  int rc = 0, prev_prio, err_code;
  bool created_here, enabled_here;
  STPM* stpm;
  UID_STP_CFG_T old;
           
  /* stp_trace ("STP_IN_stpm_set_cfg"); */
  if (0 != STP_IN_stpm_get_cfg (vlan_id, &old)) {
    STP_OUT_get_init_stpm_cfg (vlan_id, &old);
  }
  
  RSTP_CRITICAL_PATH_START;
  if (BR_CFG_PRIO & uid_cfg->field_mask) {
    old.bridge_priority = uid_cfg->bridge_priority;
  }

  if (BR_CFG_AGE & uid_cfg->field_mask) {
    old.max_age = uid_cfg->max_age;
  }

  if (BR_CFG_HELLO & uid_cfg->field_mask) {
    old.hello_time = uid_cfg->hello_time;
  }

  if (BR_CFG_DELAY & uid_cfg->field_mask) {
    old.forward_delay = uid_cfg->forward_delay;
  }

  if (BR_CFG_FORCE_VER & uid_cfg->field_mask) {
    old.force_version = uid_cfg->force_version;
  }

  rc = _check_stpm_config (&old);
  if (0 != rc) {
    stp_trace ("_check_stpm_config failed %d", (int) rc);
    RSTP_CRITICAL_PATH_END;
    return rc;
  }

  if ((BR_CFG_STATE & uid_cfg->field_mask) &&
      (STP_DISABLED == uid_cfg->stp_enabled)) {
    rc = _stp_in_stpm_enable (vlan_id, uid_cfg->vlan_name, port_bmp, STP_DISABLED);
    if (0 != rc) {
      stp_trace ("can't disable rc=%d", (int) rc);
      RSTP_CRITICAL_PATH_END;
      return rc;
    }
    uid_cfg->field_mask &= ! BR_CFG_STATE;
    if (! uid_cfg->field_mask)  {
      RSTP_CRITICAL_PATH_END;
      return 0;
    }
  }

  /* get current state */
  stpm = stpapi_stpm_find (vlan_id);
  created_here = false;
  enabled_here = false;
  if (!stpm) { /* it had not yet been created */
    stpm = stp_in_stpm_create (vlan_id, uid_cfg->vlan_name, port_bmp, &err_code);/*STP_IN_stpm_set_cfg*/
    if (!stpm) {
      RSTP_CRITICAL_PATH_END;
      return err_code;
    }
  }

  prev_prio = stpm->BrId.prio;
  stpm->BrId.prio = old.bridge_priority;
  if (STP_ENABLED == stpm->admin_state) {
    if (0 != STP_check_bridge_priority(stpm)) {
      stpm->BrId.prio = prev_prio;
      stp_trace ("%s", "STP_stpm_check_bridge_priority failed");
      RSTP_CRITICAL_PATH_END;
      return STP_Invalid_Bridge_Priority;
    }
  }

  stpm->BrTimes.MaxAge = old.max_age;
  stpm->BrTimes.HelloTime = old.hello_time;
  stpm->BrTimes.ForwardDelay = old.forward_delay;
  stpm->ForceVersion = (PROTOCOL_VERSION_T) old.force_version;

  if ((BR_CFG_STATE & uid_cfg->field_mask) &&
      STP_DISABLED != uid_cfg->stp_enabled &&
      STP_DISABLED == stpm->admin_state) {
    rc = _stp_in_stpm_enable (vlan_id, uid_cfg->vlan_name, port_bmp, uid_cfg->stp_enabled);
    if (0 != rc) {
      stp_trace ("%s", "cannot enable");
      if (created_here) {
        STPMs.erase(vlan_id);
        delete stpm;
        stpm = NULL;
      }
      RSTP_CRITICAL_PATH_END;
      return rc;
    }
    enabled_here = true;
  }

  if (! enabled_here && STP_DISABLED != stpm->admin_state) {
    stpm->STP_stpm_update_after_bridge_management();
  }
  RSTP_CRITICAL_PATH_END;
  return 0;
}

int RapidSpanningTree::STP_IN_set_port_cfg (IN int vlan_id, 
                                            IN UID_STP_PORT_CFG_T* uid_cfg)
{
  STPM* stpm;
  Port* port;
  unsigned port_no;

  RSTP_CRITICAL_PATH_START;
  stpm = stpapi_stpm_find (vlan_id);
  if (!stpm) { /* it had not yet been created :( */
    RSTP_CRITICAL_PATH_END;
    stp_trace("RSTP instance with tag %d hasn't been created\n", (int) vlan_id);
    return STP_Vlan_Had_Not_Yet_Been_Created;
  }

  for (port_no = 1; port_no <= max_port; port_no++) {
    if (! BitmapGetBit(&uid_cfg->port_bmp, port_no - 1)) continue;
  
    port = _stpapi_port_find (stpm, port_no);
    if (! port) {/* port is absent in the stpm :( */
      continue;
    }

    if (PT_CFG_MCHECK & uid_cfg->field_mask) {
      if (stpm->ForceVersion >= NORMAL_RSTP)
        port->mcheck = true;
    }

    if (PT_CFG_COST & uid_cfg->field_mask) {
      port->adminPCost = uid_cfg->admin_port_path_cost;
    }
  
    if (PT_CFG_PRIO & uid_cfg->field_mask) {
      port->port_id = (uid_cfg->port_priority << 8) + port_no;
    }
  
    if (PT_CFG_P2P & uid_cfg->field_mask) {
      port->adminPointToPointMac = uid_cfg->admin_point2point;
      port->p2p_recompute = true;
    }
  
    if (PT_CFG_EDGE & uid_cfg->field_mask) {
      port->adminEdge = uid_cfg->admin_edge;
      port->operEdge = port->adminEdge;
      if (port->edge->debug) {
        stp_trace ("port %s is operEdge=%c in STP_IN_set_port_cfg",
            port->port_name,
            port->operEdge ? 'Y' : 'n');
      }
    }

    if (PT_CFG_NON_STP & uid_cfg->field_mask) {
      if (port->roletrns->debug && port->admin_non_stp != uid_cfg->admin_non_stp) {
        stp_trace ("port %s is adminNonStp=%c in STP_IN_set_port_cfg",
            port->port_name,
            uid_cfg->admin_non_stp ? 'Y' : 'n');
      }
      port->admin_non_stp = uid_cfg->admin_non_stp;
    }

    if (PT_CFG_DBG_SKIP_RX & uid_cfg->field_mask) {
      port->skip_rx = uid_cfg->skip_rx;
    }

    if (PT_CFG_DBG_SKIP_TX & uid_cfg->field_mask) {
      port->skip_tx = uid_cfg->skip_tx;
    }

    port->reselect = true;
    port->selected = false;
  }
  
  stpm->STP_stpm_update();
  
  RSTP_CRITICAL_PATH_END;

  return 0;
}

int RapidSpanningTree::STP_IN_dbg_set_port_trace (char* mach_name, int enadis,
                                                  int vlan_id, BITMAP_T* ports,
                                                  int is_print_err)
{
  STPM *stpm;
  Port *port;
  unsigned port_no;

  RSTP_CRITICAL_PATH_START;
  stpm = stpapi_stpm_find (vlan_id);
  if (!stpm) { /* it had not yet been created :( */
    RSTP_CRITICAL_PATH_END;
    if (is_print_err) {
      stp_trace("RSTP instance with tag %d hasn't been created\n", 
                (int) vlan_id);
    }
    return STP_Vlan_Had_Not_Yet_Been_Created;
  }

  for (port_no = 1; port_no <= max_port; port_no++) {
    if (! BitmapGetBit(ports, port_no - 1)) continue;
  
    port = _stpapi_port_find (stpm, port_no);
    if (! port) {/* port is absent in the stpm :( */
      continue;
    }
    port->STP_port_trace_state_machine (mach_name, enadis, vlan_id);
  }
  
  RSTP_CRITICAL_PATH_END;

  return 0;
}

const char* RapidSpanningTree::STP_IN_get_error_explanation (int rstp_err_no)
{
#define CHOOSE(a) #a
static char* rstp_error_names[] = RSTP_ERRORS;
#undef CHOOSE
  if (rstp_err_no < STP_OK) {
    return "Too small error code :(";
  }
  if (rstp_err_no >= STP_LAST_DUMMY) {
    return "Too big error code :(";
  }
  
  return rstp_error_names[rstp_err_no];
}

int RapidSpanningTree::STP_check_bridge_priority(STPM *stpm)
{
  STPMHash::iterator i;
  for (i = STPMs.begin(); i != STPMs.end(); i++) {
    STPM *oth = i->second;
    if (STP_ENABLED == oth->admin_state && oth != stpm &&
        ! STP_VECT_compare_bridge_id (&stpm->BrId, &oth->BrId)) {
      return STP_Invalid_Bridge_Priority;
    }        
  }
  return 0;
}
