/************************************************************************ 
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * This file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 **********************************************************************/

/* Port State Transition state machine : 17.24 */
    
#include "global.h"
#include "PortStateTransitions.h"
#include "STPM.h"
#include "stp_to.h"
#include "Port.h"

#define STATES { \
  CHOOSE(DISCARDING),   \
  CHOOSE(LEARNING), \
  CHOOSE(FORWARDING),   \
}

#define GET_STATE_NAME PortStateTransitions::GetStateName
#include "choose.h"

PortStateTransitions::PortStateTransitions(Port *p)
  : StateMachine("PortStateTransitions"), owner(p)
{
  // empty
}

bool PortStateTransitions::disableLearning()
{
  Port *port = owner;

  return STP_OUT_set_learning (port->port_index, port->owner->GetVlanID(), false);
}

bool PortStateTransitions::enableLearning()
{
  Port *port = owner;

  return STP_OUT_set_learning (port->port_index, port->owner->GetVlanID(), true);
}

bool PortStateTransitions::disableForwarding()
{
  Port *port = owner;

  return STP_OUT_set_forwarding (port->port_index, port->owner->GetVlanID(), false);
}

bool PortStateTransitions::enableForwarding()
{
  Port *port = owner;

  return STP_OUT_set_forwarding (port->port_index, port->owner->GetVlanID(), true);
}

void PortStateTransitions::EnterState()
{
  Port *port = owner;

  switch (State) {
    case BEGIN:
    case DISCARDING:
      port->learning = false;
      port->forwarding = false;
#ifdef STRONGLY_SPEC_802_1W
      disableLearning();
      disableForwarding();
#else
      STP_OUT_set_port_state (port->port_index, port->owner->GetVlanID(), UID_PORT_DISCARDING);
#endif
      break;
    case LEARNING:
      port->learning = true;
#ifdef STRONGLY_SPEC_802_1W
      enableLearning();
#else
      STP_OUT_set_port_state (port->port_index, port->owner->GetVlanID(), UID_PORT_LEARNING);
#endif
      break;
    case FORWARDING:
      port->tc = !port->operEdge;
      port->forwarding = true;
#ifdef STRONGLY_SPEC_802_1W
      enableForwarding();
#else
      STP_OUT_set_port_state (port->port_index, port->owner->GetVlanID(), UID_PORT_FORWARDING);
#endif
      break;
  }
}

bool PortStateTransitions::CheckCondition()
{
  Port *port = owner;

  if (BEGIN == State) {
    return Hop2State(DISCARDING);
  }

  switch (this->State) {
    case DISCARDING:
      if (port->learn) { 
        return Hop2State(LEARNING);
      }
      break;
    case LEARNING:
      if (port->forward) {
        return Hop2State(FORWARDING);
      }
      if (!port->learn) {
        return Hop2State(DISCARDING);
      }
      break;
    case FORWARDING:
      if (!port->forward) {
        return Hop2State(DISCARDING);
      }
      break;
  }

  return false;
}

