/************************************************************************ 
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * S file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 **********************************************************************/

#include "global.h"
#include "STPM.h"
#include "Port.h"
#include "PortInfo.h"
#include "Log.h"

/* The Port Information State Machine : 17.21 */

#define STATES { \
  CHOOSE(DISABLED), \
  CHOOSE(ENABLED),  \
  CHOOSE(AGED),     \
  CHOOSE(UPDATE),   \
  CHOOSE(CURRENT),  \
  CHOOSE(RECEIVE),  \
  CHOOSE(SUPERIOR), \
  CHOOSE(REPEAT),   \
  CHOOSE(AGREEMENT),    \
}

#define GET_STATE_NAME PortInfo::GetStateName
#include "choose.h"

#if 0 /* for debug */
void
_stp_dump (char* title, unsigned char* buff, int len)
{
  register int iii;

  stp_trace("%s:", title);
  for (iii = 0; iii < len; iii++) {
    if (! (iii % 24)) { 
      if (iii != 0) {
	stp_trace_nolf("\n");
      }
      stp_trace_nolf ("%6d:", iii);
    }
    if (! (iii % 8)) stp_trace_nolf(" ");
    stp_trace_nolf("%02lx", (unsigned long) buff[iii]);
  }
  stp_trace_nolf("\n");
}
#endif


PortInfo::PortInfo(Port *p) : StateMachine("PortInfo"), owner(p)
{
  // empty
}

RCVD_MSG_T PortInfo::rcvBpdu ()
{/* 17.19.8 */
  int   bridcmp;
  Port *port = owner;

  if (port->msgBpduType == BPDU_TOPO_CHANGE_TYPE) {
    if (debug) {
        stp_trace ("OtherMsg:BPDU_TOPO_CHANGE_TYPE");
    }
    return OtherMsg;
  }

  port->msgPortRole = RSTP_PORT_ROLE_UNKN;

  if (BPDU_RSTP == port->msgBpduType) {
    port->msgPortRole = (port->msgFlags & PORT_ROLE_MASK) >> PORT_ROLE_OFFS;
  }

  if (RSTP_PORT_ROLE_DESGN == port->msgPortRole ||
      BPDU_CONFIG_TYPE == port->msgBpduType) {
    bridcmp = STP_VECT_compare_vector (&port->msgPrio, &port->portPrio);

    if (bridcmp < 0 ||
        (! STP_VECT_compare_bridge_id (&port->msgPrio.design_bridge,
                                       &port->portPrio.design_bridge) &&
         port->msgPrio.design_port == port->portPrio.design_port      &&
         STP_compare_times (&port->msgTimes, &port->portTimes))) {
         if (debug) {
           stp_trace ("SuperiorDesignateMsg:bridcmp=%d", (int) bridcmp);
         }
      return SuperiorDesignateMsg;
    }
  }

  if (BPDU_CONFIG_TYPE == port->msgBpduType ||
      RSTP_PORT_ROLE_DESGN == port->msgPortRole) {
    if (! STP_VECT_compare_vector (&port->msgPrio,
                                   &port->portPrio) &&
        ! STP_compare_times (&port->msgTimes, &port->portTimes)) {
        if (debug) {
          stp_trace ("%s", "RepeatedDesignateMsg");
        }
        return RepeatedDesignateMsg;
    }
  }

  if (RSTP_PORT_ROLE_ROOT == port->msgBpduType &&
      port->operPointToPointMac &&
      ! STP_VECT_compare_bridge_id (&port->msgPrio.design_bridge,
                                    &port->portPrio.design_bridge) &&
      AGREEMENT_BIT & port->msgFlags) {
    if (debug) {
      stp_trace ("%s", "ConfirmedRootMsg");
    }
    return ConfirmedRootMsg;
  }
  
    if (debug) {
      stp_trace ("%s", "OtherMsg");
    }
  return OtherMsg;
}

bool PortInfo::recordProposed(char* reason)
{/* 17.19.9 */
  Port *port = owner;

  if (RSTP_PORT_ROLE_DESGN == port->msgPortRole &&
      (PROPOSAL_BIT & port->msgFlags) &&
      port->operPointToPointMac) {
    return true;
  }
  return false;
}

bool PortInfo::setTcFlags()
{/* 17.19.13 */
  Port *port = owner;

  if (BPDU_TOPO_CHANGE_TYPE == port->msgBpduType) {
      if (debug) {
        stp_trace ("port %s rx rcvdTcn", port->port_name);
      }
    port->rcvdTcn = true;
  } else {
    if (TOLPLOGY_CHANGE_BIT & port->msgFlags) {
      if (debug) {
        stp_trace ("(%s-%s) rx rcvdTc 0X%lx",
                   port->owner->GetName(), port->port_name,
                   (unsigned long) port->msgFlags);
      }
      port->rcvdTc = true;
    }
    if (TOLPLOGY_CHANGE_ACK_BIT & port->msgFlags) {
      if (debug) {
        stp_trace ("port %s rx rcvdTcAck 0X%lx",
            port->port_name,
            (unsigned long) port->msgFlags);
      }
      port->rcvdTcAck = true;
    }
  }
  return true;
}

bool PortInfo::updtBPDUVersion()
{/* 17.19.18 */
  Port *port = owner;

  if (BPDU_TOPO_CHANGE_TYPE == port->msgBpduType) {
    port->rcvdSTP = true;
  }

  if (port->msgBpduVersion < 2) {
    port->rcvdSTP = true;
  }
  
  if (BPDU_RSTP == port->msgBpduType) {
    /* port->port->owner->ForceVersion >= NORMAL_RSTP
       we have checked in STP_info_rx_bpdu */
    port->rcvdRSTP = true;
  }

  return true;
}

bool PortInfo::updtRcvdInfoWhile()
{/* 17.19.19 */
  int eff_age, dm, dt;
  int hello3;
  Port *port = owner;
  
  eff_age = ( + port->portTimes.MaxAge) / 16;
  if (eff_age < 1) {
    eff_age = 1;
  }
  eff_age += port->portTimes.MessageAge;

  if (eff_age <= port->portTimes.MaxAge) {
    hello3 = 3 *  port->portTimes.HelloTime;
    dm = port->portTimes.MaxAge - eff_age;
    if (dm > hello3) {
      dt = hello3;
    } else {
      dt = dm;
    }
    port->rcvdInfoWhile = dt;

    stp_trace ("ma=%d eff_age=%d dm=%d dt=%d p=%s",
               (int) port->portTimes.MessageAge,
               (int) eff_age, (int) dm, (int) dt, port->port_name);
  } else {
    port->rcvdInfoWhile = 0;
  }

  if (debug)
    {
      stp_trace ("port %s: MaxAge=%d MessageAge=%d HelloTime=%d rcvdInfoWhile=%d",
                 port->port_name,
                 (int) port->portTimes.MaxAge,
                 (int) port->portTimes.MessageAge,
                 (int) port->portTimes.HelloTime,
                 (int) port->rcvdInfoWhile);
    }
  return true;
}


void PortInfo::STP_info_rx_bpdu (Port* port, struct stp_bpdu_t* bpdu, size_t len)
{  
#if 0
  _stp_dump ("all BPDU", ((unsigned char*) bpdu) - 12, len + 12);
  _stp_dump ("ETH_HEADER", (unsigned char*) &bpdu->eth, 5);
  _stp_dump ("BPDU_HEADER", (unsigned char*) &bpdu->hdr, 4);
  stp_trace_nolf("protocol=%02x%02x version=%02x bpdu_type=%02x\n",
     bpdu->hdr.protocol[0], bpdu->hdr.protocol[1],
     bpdu->hdr.version, bpdu->hdr.bpdu_type);

  _stp_dump ("BPDU_BODY", (unsigned char*) &bpdu->body, sizeof (BPDU_BODY_T) + 2);
  stp_trace_nolf("flags=%02x\n", bpdu->body.flags);
  _stp_dump ("root_id", bpdu->body.root_id, 8);
  _stp_dump ("root_path_cost", bpdu->body.root_path_cost, 4);
  _stp_dump ("bridge_id", bpdu->body.bridge_id, 8);
  _stp_dump ("port_id", bpdu->body.port_id, 2);
  _stp_dump ("message_age", bpdu->body.message_age, 2);
  _stp_dump ("max_age", bpdu->body.max_age, 2);
  _stp_dump ("hello_time", bpdu->body.hello_time, 2);
  _stp_dump ("forward_delay", bpdu->body.forward_delay, 2);
  _stp_dump ("ver_1_len", bpdu->ver_1_len, 2);
#endif
  
  /* check bpdu type */
  switch (bpdu->hdr.bpdu_type) {
    case BPDU_CONFIG_TYPE:
      port->rx_cfg_bpdu_cnt++;
      if (port->info->debug) 
        stp_trace ("RECV CfgBpdu on port %s", port->port_name);
      if (port->admin_non_stp) return;
      port->rcvdBpdu = true;
      break;
    case BPDU_TOPO_CHANGE_TYPE:
      port->rx_tcn_bpdu_cnt++;
      if (port->info->debug)
        stp_trace ("RECV TcnBpdu on port %s", port->port_name);
      if (port->admin_non_stp) return;
      port->rcvdBpdu = true;
      port->msgBpduVersion = bpdu->hdr.version;
      port->msgBpduType = bpdu->hdr.bpdu_type;
      return;
    default:
      stp_trace ("RECV RX undef bpdu type=%d", (int) bpdu->hdr.bpdu_type);
      return;
    case BPDU_RSTP:
      port->rx_rstp_bpdu_cnt++;
      if (port->admin_non_stp) return;
      if (port->owner->GetForceVersion() >= NORMAL_RSTP) {
        port->rcvdBpdu = true;
      } else {          
        return;
      }
      if (port->info->debug)
        stp_trace ("RECV BPDU_RSTP on port %s", port->port_name);
      break;
  }

  port->msgBpduVersion = bpdu->hdr.version;
  port->msgBpduType =    bpdu->hdr.bpdu_type;
  port->msgFlags =       bpdu->body.flags;

  /* 17.18.11 */
  STP_VECT_get_vector (&bpdu->body, &port->msgPrio);
  port->msgPrio.bridge_port = port->port_id;

  /* 17.18.12 */
  STP_get_times (&bpdu->body, &port->msgTimes);

  /* 17.18.25, 17.18.26 : see setTcFlags() */
}

void PortInfo::EnterState()
{
  Port *port = owner;
  
  switch (State) {
  case BEGIN:
    port->rcvdMsg = OtherMsg;
    // port->msgBpduType = -1;  XXXacm var is unsigned char
    port->msgBpduType = 0xff;
    port->msgPortRole = RSTP_PORT_ROLE_UNKN;
    port->msgFlags = 0;
    
    /* clear port statistics */
    port->rx_cfg_bpdu_cnt =
      port->rx_rstp_bpdu_cnt =
      port->rx_tcn_bpdu_cnt = 0;
    
  case DISABLED:
    port->rcvdBpdu = port->rcvdRSTP = port->rcvdSTP = false;
    port->updtInfo = port->proposing = false; /* In DISABLED */
    port->agreed = port->proposed = false;
    port->rcvdInfoWhile = 0;
    port->infoIs = Disabled;
    port->reselect = true;
    port->selected = false;
    break;
  case ENABLED: /* IEEE 802.1y, 17.21, Z.14 */
    STP_VECT_copy (&port->portPrio, &port->designPrio);
    STP_copy_times (&port->portTimes, &port->designTimes);
    break;
  case AGED:
    port->infoIs = Aged;
    port->reselect = true;
    port->selected = false;
    break;
  case UPDATE:
    STP_VECT_copy (&port->portPrio, &port->designPrio);
    STP_copy_times (&port->portTimes, &port->designTimes);
    port->updtInfo = false;
    port->agreed = port->synced = false; /* In UPDATE */
    port->proposed = port->proposing = false; /* in UPDATE */
    port->infoIs = Mine;
    port->newInfo = true;
    if (debug) {
      stp_trace_begin();
      STP_VECT_br_id_print ("updated: portPrio.design_bridge",
                            &port->portPrio.design_bridge);
      stp_trace_end();
    }
    break;
  case CURRENT:
    break;
  case RECEIVE:
    port->rcvdMsg = rcvBpdu();
    updtBPDUVersion();
    setTcFlags();
    port->rcvdBpdu = false;
    break;
  case SUPERIOR:
    STP_VECT_copy(&port->portPrio, &port->msgPrio);
    STP_copy_times(&port->portTimes, &port->msgTimes);
    updtRcvdInfoWhile();
#if 1 /* due 802.1y, Z.7 */
    port->agreed = false; /* deleted due 802.y in SUPERIOR */
    port->synced = false; /* due 802.y deleted in SUPERIOR */
#endif
    port->proposing = false; /* in SUPERIOR */
    port->proposed = recordProposed("SUPERIOR");
    port->infoIs = Received;
    port->reselect = true;
    port->selected = false;
    if (debug) {
      stp_trace_begin();
      STP_VECT_br_id_print ("stored: portPrio.design_bridge",
                            &port->portPrio.design_bridge);
      stp_trace_more(" proposed=%d on port %s",
                     (int) port->proposed, port->port_name);
      stp_trace_end();
    }
    break;
  case REPEAT:
    port->proposed = recordProposed("REPEAT");
    updtRcvdInfoWhile();
    break;
  case AGREEMENT:
    if (port->roletrns->GetDebug()) {
      stp_trace ("(%s-%s) rx AGREEMENT flag !",
                 port->owner->GetName(), port->port_name);
    }
    port->agreed = true;
    port->proposing = false; /* In AGREEMENT */
    break;
  }
}

bool PortInfo::CheckCondition()
{
  Port *port = owner;

  if ((! port->portEnabled && port->infoIs != Disabled) || BEGIN == State) {
    return Hop2State(DISABLED);
  }

  switch (State) {
    case DISABLED:
      if (port->updtInfo) {
        return Hop2State(DISABLED);
      }
      if (port->portEnabled && port->selected) {
        return Hop2State(ENABLED);
      }
      if (port->rcvdBpdu) {
        return Hop2State(DISABLED);
      }
      break; 
    case ENABLED: /* IEEE 802.1y, 17.21, Z.14 */
      return Hop2State(AGED);
      break; 
    case AGED:
      if (port->selected && port->updtInfo) {
        return Hop2State(UPDATE);
      }
      break;
    case UPDATE:
      return Hop2State(CURRENT);
      break;
    case CURRENT:
      if (port->selected && port->updtInfo) {
        return Hop2State(UPDATE);
      }

      if (Received == port->infoIs       &&
          ! port->rcvdInfoWhile &&
          ! port->updtInfo               &&
          ! port->rcvdBpdu) {
        return Hop2State(AGED);
      }
      if (port->rcvdBpdu && !port->updtInfo) {
        return Hop2State(RECEIVE);
      }
      break;
    case RECEIVE:
      switch (port->rcvdMsg) {
        case SuperiorDesignateMsg:
          return Hop2State(SUPERIOR);
        case RepeatedDesignateMsg:
          return Hop2State(REPEAT);
        case ConfirmedRootMsg:
          return Hop2State(AGREEMENT);
        default:
          return Hop2State(CURRENT);
      }
      break;
    case SUPERIOR:
      return Hop2State(CURRENT);
      break;
    case REPEAT:
      return Hop2State(CURRENT);
      break;
    case AGREEMENT:
      return Hop2State(CURRENT);
      break;
  }

  return false;
}


