/************************************************************************ 
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * This file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 **********************************************************************/

/* STP PORT instance : 17.18, 17.15 */

#include "global.h"
#include "Port.h"
#include "RapidSpanningTree.h"
#include "Switch.h"
#include "Log.h"

#include "stp_to.h" /* for STP_OUT_get_port_name & STP_OUT_get_port_link_status */

Port::Port(STPM *stpm, int port_index_arg) 
{
  port_name = strdup (STP_OUT_get_port_name (port_index_arg));

  LogContext lc(port_name);

  UID_STP_PORT_CFG_T port_cfg;
  register int   iii;
  unsigned short port_prio;

  owner = stpm;
  port_index = port_index_arg;
  uptime = 0;

  STP_OUT_get_init_port_cfg (stpm->vlan_id, port_index, &port_cfg);
  port_prio =                  port_cfg.port_priority;
  admin_non_stp =        port_cfg.admin_non_stp;
  adminEdge =            port_cfg.admin_edge;
  adminPCost =           port_cfg.admin_port_path_cost;
  adminPointToPointMac = port_cfg.admin_point2point;
  
  LinkDelay = DEF_LINK_DELAY;
  port_id = (port_prio << 8) + port_index;

  iii = 0;
  timers[iii++] = &this->fdWhile;
  timers[iii++] = &this->helloWhen;
  timers[iii++] = &this->mdelayWhile;
  timers[iii++] = &this->rbWhile;
  timers[iii++] = &this->rcvdInfoWhile;
  timers[iii++] = &this->rrWhile;
  timers[iii++] = &this->tcWhile;
  timers[iii++] = &this->txCount;
  timers[iii++] = &this->lnkWhile;

  /* create and bind port state machines */
  topoch = new TopologyChange(this);
  machines.insert(topoch);
  migrate = new ProtocolMigration(this);
  machines.insert(migrate);
  p2p = new P2P(this);
  machines.insert(p2p);
  edge = new Edge(this);
  machines.insert(edge);
  pcost = new PathCost(this);
  machines.insert(pcost);
  info = new PortInfo(this);
  machines.insert(info);
  roletrns = new RoleTransitions(this);
  machines.insert(roletrns);
  sttrans = new PortStateTransitions(this);
  machines.insert(sttrans);
  transmit = new Transmit(this);
  machines.insert(transmit);
                  
#if 0
  this->roletrns->ignoreHop2State = 14; /* DESIGNATED_PORT; */
  this->info->ignoreHop2State =      3; /* CURRENT */
  this->transmit->ignoreHop2State =  3; /* IDLE */
  this->edge->ignoreHop2State =      0; /* DISABLED; */
#endif

#if 0
  this->info->debug = 1;
  this->pcost->debug = 1;
  this->p2p->debug = 1;
  this->edge->debug = 1;
  this->migrate->debug = 1;
  this->sttrans->debug = 1;
  this->topoch->debug = 1;
  this->roletrns->debug = 1;
  this->sttrans->debug = 1;
#endif
}

void Port::STP_port_init (STPM* stpm, bool check_link)
{
  LogContext lc(port_name);

  if (check_link) {
    adminEnable = 
      stpm->GetRapidSpanningTree()->GetSwitch()->GetPortStatus(port_index);
    STP_VECT_create (&this->designPrio,
                   &stpm->BrId,
                   0,
                   &stpm->BrId,
                   this->port_id,
                   this->port_id);
    STP_copy_times (&this->designTimes, &stpm->rootTimes);
  }

  skip_tx = skip_rx = 0;

  /* reset timers */
  this->fdWhile =
  this->helloWhen =
  this->mdelayWhile =
  this->rbWhile =
  this->rcvdInfoWhile =
  this->rrWhile =
  this->tcWhile =
  this->txCount = 0;

  this->msgPortRole = RSTP_PORT_ROLE_UNKN;
  this->selectedRole = DisabledPort;
  this->sendRSTP = true;
  this->operSpeed = STP_OUT_get_port_oper_speed (this->port_index);
  this->p2p_recompute = true;
}

Port::~Port()
{
  LogContext lc(port_name);
  
  set<StateMachine *>::iterator i;
  for (i = machines.begin(); i != machines.end(); i++) {
    delete *i;
  }
  machines.clear();
  free(port_name);
}

int Port::STP_port_rx_bpdu (BPDU_T* bpdu, size_t len)
{
  LogContext lc(port_name);

  info->STP_info_rx_bpdu (this, bpdu, len);
  return 0;
}

int Port::STP_port_trace_state_machine (char* mach_name, int enadis, 
                                        int vlan_id)
{
  LogContext lc(port_name);

  set<StateMachine *>::iterator i;
  for (i = machines.begin(); i != machines.end(); i++) {
    StateMachine *stater = *i;
    if (! strcmp (mach_name, "all") || ! strcmp (mach_name, stater->GetName())) {
      /* if (stater->debug != enadis) */
      {
        stp_trace ("port %s on %s trace %-8s (was %s) now %s",
                   port_name, owner->name,
                   stater->GetName(),
                   stater->GetDebug() ? " enabled" :"disabled",
                   enadis        ? " enabled" :"disabled");
      }
      stater->SetDebug(enadis);
    }
  }
  return 0;
}

void Port::STP_port_trace_flags (char* title)
{
  LogContext lc(port_name);

#if 0 /* it may be opened for more deep debugging */
    unsigned long flag = 0L;
    
    if (this->reRoot)   flag |= 0x000001L;
    if (this->sync)     flag |= 0x000002L;
    if (this->synced)   flag |= 0x000004L;

    if (this->proposed)  flag |= 0x000010L;
    if (this->proposing) flag |= 0x000020L;
    if (this->agreed)    flag |= 0x000040L;
    if (this->updtInfo)  flag |= 0x000080L;

    if (this->operEdge)   flag |= 0x000100L;
    stp_trace ("         %-12s: flags=0X%04lx port=%s", title, flag, this->port_name);
#endif
}

