/************************************************************************ 
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * S file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 **********************************************************************/

/* Path Cost monitoring state machine */
 
#include "global.h"
#include "STPM.h"
#include "PathCost.h"
#include "Port.h"
#include "stp_to.h" /* for STP_OUT_get_port_oper_speed */
#include "Log.h"

#define STATES {        \
  CHOOSE(AUTO),         \
  CHOOSE(FORSE),        \
  CHOOSE(STABLE),       \
}

#define GET_STATE_NAME PathCost::GetStateName
#include "choose.h"

PathCost::PathCost(Port *p) : StateMachine("PathCost"), owner(p)
{
  // empty
}

long PathCost::computeAutoPCost()
{
  long lret;
  Port *port = owner;

  if (port->GetUsedSpeed()        < 10L) {         /* < 10Mb/s */
    lret = 20000000;
  } else if (port->GetUsedSpeed() <= 10L) {        /* 10 Mb/s  */
    lret = 2000000;        
  } else if (port->GetUsedSpeed() <= 100L) {       /* 100 Mb/s */
    lret = 200000;     
  } else if (port->GetUsedSpeed() <= 1000L) {      /* 1 Gb/s */
    lret = 20000;      
  } else if (port->GetUsedSpeed() <= 10000L) {     /* 10 Gb/s */
    lret = 2000;       
  } else if (port->GetUsedSpeed() <= 100000L) {    /* 100 Gb/s */
    lret = 200;        
  } else if (port->GetUsedSpeed() <= 1000000L) {   /* 1 GTb/s */
    lret = 20;     
  } else if (port->GetUsedSpeed() <= 10000000L) {  /* 10 Tb/s */
    lret = 2;      
  } else   /* ??? */                        { /* > Tb/s */
    lret = 1;       
  }

  if (port->pcost->debug) {
    stp_trace ("usedSpeed=%lu lret=%ld", port->GetUsedSpeed(), lret);
  }
  return lret;
}

void PathCost::EnterState()
{
  Port *port = owner;

  switch (State) {
    case BEGIN:
      break;
    case AUTO:
      port->operSpeed = STP_OUT_get_port_oper_speed (port->port_index);

      if (port->pcost->debug) {
        stp_trace ("AUTO:operSpeed=%lu", port->operSpeed);
      }
      port->usedSpeed = port->operSpeed;
      port->operPCost = computeAutoPCost();
      break;
    case FORSE:
      port->operPCost = port->adminPCost;
      // port->usedSpeed = -1; // acm: this is how it was, but -1 in
                                               // an unsigned is bad news
      port->usedSpeed = 0xffffffff;
      break;
    case STABLE:
      // acm: this is a noop.  used to be a call to an empty function, updPortPathCost
      break;
  }
}

bool PathCost::CheckCondition()
{
  Port *port = owner;

  switch (State) {
    case BEGIN:
      return Hop2State(AUTO);
    case AUTO:
      return Hop2State(STABLE);
    case FORSE:
      return Hop2State(STABLE);
    case STABLE:
      if (ADMIN_PORT_PATH_COST_AUTO == port->adminPCost && 
          port->operSpeed != port->GetUsedSpeed()) {
        return Hop2State(AUTO);
      }
      if (ADMIN_PORT_PATH_COST_AUTO != port->adminPCost &&
          port->operPCost != port->adminPCost) {
          return Hop2State(FORSE);
      }
      break;
  }
  return false;
}

