#include "global.h"
#include "Parser.h"
#include "Simulator.h"
#include "ArgParser.h"

#include <iostream>
#include <istream>
#include <fstream>
#include <sstream>

void Parser::Parse(char *filename) {
  ifstream fstream(filename, ios_base::in);
  string line;

  if (!fstream.is_open()) {
    cerr << "Could not open file " << filename << endl;
    exit(1);
  }
  
  enum Section { SecBogus, SecNodes, SecLinks, SecEvents };
  Section status = SecBogus;
  unsigned lineNum = 0;

  /* read lines from the configuration file and process it */
  while(getline(fstream, line)) {
    lineNum++;
    line = trim(line);
    if (line.empty()) {
      continue;
    }

  if (line.find('[') != string::npos) // find configuration title
    {
      // it's a section header line
      if (line.find("nodes") != string::npos)	{
        cerr << "found a [nodes] section" << endl;
        status = SecNodes;
      } else if (line.find("links") != string::npos) {
        cerr << "found a [links] section" << endl;
        status = SecLinks;
      } else if (line.find("events") !=string::npos) {
        cerr << "found an [events] section" << endl;
        status = SecEvents;
      } else {
        cerr << "on line " << lineNum << " could not parse section header:"
             << endl << line << endl;
        exit(1);
      }
    } else {
      // it's a section body line
      switch (status) {
      case SecNodes:
        ParseNodeLine(line, lineNum);
        break;
      case SecLinks:
        ParseLinkLine(line, lineNum);
        break;
      case SecEvents:
        ParseEventLine(line, lineNum);
        break;
      default:
        assert(0);
      }
    }
  } // while getline

  fstream.close();
  cout<<"Parser done reading file " << filename << endl;
}

//  trim: get rid of extra blank both at the beginning and end of the line
string Parser::trim (string s)
{
  if (s.empty())
    return s;
  
  int pos_t = s.find('\r');
  if (pos_t!=-1)
    s.erase(pos_t,1);
  
  int pos_b = s.find_first_not_of(" ");
  if (pos_b!=0)
    s.erase(0, pos_b);
  
  unsigned int pos_e = s.find_last_not_of(" ");
  if (pos_e!=s.size()-1)
    s.erase(pos_e+1, s.length());
  return s;
}

// line format: nodeID
// XXXacm - allow nodes to have arbitrary strings for names
void Parser::ParseNodeLine(string& line, unsigned lineNum)
{
  istringstream str(line);
  unsigned nodeid = 0;

  str >> nodeid;
  if (nodeid == 0) {
    cerr << "could not parse node id in line " << lineNum << " of input:"
         << endl << line << endl;
    exit(1);
  }

  ArgParser ap;
  DoubleArg offset("offset", 0, ap);

  ap.Parse(str, lineNum);

  if (offset.val < 0 || offset.val >= 1.0) {
    cerr << "offset must be greater than or equal to 0 or less than 1.0 on input line "
         << lineNum << endl;
    exit(1);
  }

  simulator->AddNode(nodeid, offset.val);  
}

// line format: nodeid1 nodeid2 delayInSeconds
void Parser::ParseLinkLine(string& line, unsigned lineNum)
{
  istringstream str(line);
  unsigned nodeA = 0, nodeB = 0;

  str >> nodeA >> nodeB;

  ArgParser ap;
  DoubleArg delay("delay", -1.0, ap);
  ap.Parse(str, lineNum);

  if (nodeA == 0 || nodeB == 0 || delay.val < 0) {
    cerr << "could not parse link info on line " << lineNum << " of input:"
         << endl << line << endl;
    exit(1);
  }
  simulator->AddLink(nodeA, nodeB, delay.val);
}

// line format: timeInSeconds EventType [type-specitic args]
void Parser::ParseEventLine(string& line, unsigned lineNum)
{
  istringstream str(line);
  double when = -1;
  string evType = "bogus";

  str >> when >> evType;

  if (when == -1 || evType == "bogus") {
    cerr << "could not parse event on line " << lineNum << " of input:"
         << endl << line << endl;
    exit(1);
  }

  if (evType == "linkdown") {
    ArgParser ap;
    UnsignedArg nodeA("nodeA", 0, ap);
    UnsignedArg nodeB("nodeB", 0, ap);
    DoubleArg delayA("delayA", 0, ap);
    DoubleArg delayB("delayB", 0, ap);
    ap.Parse(str, lineNum);

    if (delayA.val < 0 || delayB.val < 0) {
      cerr << "on line " << lineNum << " delayA and delayB must be non-negative"
           << endl;
      exit(1);
    }

    simulator->AddLinkDownEvent(when, nodeA.val, nodeB.val, delayA.val, delayB.val);
  } else if (evType == "simend") {
    simulator->AddSimulationEndEvent(when);
  } else {
    cerr << "could not parse event type " << evType << " on line " << lineNum 
         << " of input:" << endl << line << endl;
    exit(1);
  }
}

