#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <assert.h>
#include <sys/time.h>
#include <stdint.h>
#include <netinet/in.h>
#include <iostream>
#include <vector>

#include "global.h"
#include "Log.h"
#include "stp_bpdu.h"

static FILE *LOGFILE = NULL;

static vector<LogContext *> ContextVector;

unsigned pushLogContext(LogContext* a)
{
  ContextVector.push_back(a);
  return ContextVector.size();  /* vector index of context */
}

void popLogContext(unsigned uid)
{
  assert(uid == ContextVector.size());
  ContextVector.pop_back();
}

void log_init(const char *name)
{
  LOGFILE = fopen(name, "w");
  if (LOGFILE == NULL) {
    fprintf(stderr, "COULD NOT OPEN LOG %s\n", name);
    exit(1);
  }
  stp_trace("BridgeSim starting up!");
  stp_trace("Logging to %s", name);
}

void log_close(void)
{
  assert(LOGFILE);
  if (LOGFILE) {
    fclose(LOGFILE);
  }
}

static void print_header(FILE *out)
{
  const char *context = "sim";
  if (ContextVector.size() > 0) {
    context = ContextVector[0]->name;
  }

  fprintf(out, "%.6f %s", GetCurrentTime(), context);
  for (unsigned i = 1; i < ContextVector.size(); i++) {
    fprintf(out, ":%s", ContextVector[i]->name);
  }
  fprintf(out, " ");
}

static void print_footer(FILE *out)
{
  fprintf(out, "\r\n");
  fflush(out);
}

unsigned long stp_trace(const char *format, ...) 
{
  FILE *out = LOGFILE ? LOGFILE : stdout;

  print_header(out);

  va_list  args;
  va_start(args, format);
  vfprintf (out, format, args);
  va_end(args);

  print_footer(out);

  return 0;
}

unsigned long stp_trace_begin(void)
{
  FILE *out = LOGFILE ? LOGFILE : stdout;
  print_header(out);
  return 0;
}

unsigned long stp_trace_end(void)
{
  FILE *out = LOGFILE ? LOGFILE : stdout;
  print_footer(out);
  return 0;
}

unsigned long stp_trace_more(const char *format, ...) 
{
  FILE *out = LOGFILE ? LOGFILE : stdout;
  va_list  args;
  va_start(args, format);

  vfprintf (out, format, args);
  va_end(args);
  fflush(out);

  return 0;
}

#define PACK __attribute__ ((packed));

struct mac_head {
  unsigned char dst_mac[6];
  unsigned char src_mac[6];
} PACK;

struct eth_head {
  unsigned char len8023[2];
  unsigned char dsap;
  unsigned char ssap;
  unsigned char llc;
} PACK;

struct bpdu_head {
  uint16_t protocol;
  uint8_t version;
  uint8_t bpdu_type;
} PACK;

struct bpdu_bod {
  uint8_t flags;
  uint64_t root_id;
  uint32_t root_path_cost;
  uint64_t bridge_id;
  uint16_t port_id;
  uint16_t message_age;
  uint16_t  max_age;
  uint16_t hello_time;
  uint16_t forward_delay;
} PACK;

struct stp_bpdu {
  struct eth_head eth;
  struct bpdu_head hdr;
  struct bpdu_bod body;
  unsigned char ver_1_len[2];
} PACK;

struct tx_tcn_bpdu {
  struct mac_head  mac;
  struct eth_head eth;
  struct bpdu_head hdr;
} PACK;

struct tx_stp_bpdu {  /* aka a CONFIG bpdu */
  struct mac_head  mac;
  struct eth_head eth;
  struct bpdu_head hdr;
  struct bpdu_bod body;
} PACK;

struct tx_rstp_bpdu {
  struct mac_head  mac;
  struct eth_head eth;
  struct bpdu_head hdr;
  struct bpdu_bod body;
  unsigned char ver_1_length[2];
} PACK;

/* convert a 64 bit int from network byte order */
/* from http://www.codeproject.com/cpp/endianness.asp */
#define ntohll(x) (((uint64_t)(ntohl((int)((x << 32) >> 32))) << 32) | (unsigned int)ntohl(((int)(x >> 32))))
#define htonll(x) ntohll(x)

static void dump_body(const struct bpdu_bod *body)
{
  /* note: all struct fields are in network byte order */
  uint8_t role;

  uint64_t mask = ((uint64_t)0xffff << 32) | 0xffffffff;
  uint64_t bid = ntohll(body->root_id);
  bid &= mask;
  uint64_t sid = ntohll(body->bridge_id);
  sid &= mask;

  stp_trace_more(" root_bridge 0x%llx root_path_cost %u\n", 
		 bid, ntohl(body->root_path_cost));
  stp_trace_more("\tsending_bridge 0x%llx port %u", 
		 sid, ntohs(body->port_id));
  /* flags */
  stp_trace_more(" flags:");
  if (body->flags & 0x1) {
    stp_trace_more(" (topology_change)");
  } 
  if (body->flags>>1 & 0x1) {
    stp_trace_more(" (proposal)");
  } 
  /* bits 3 and 4 are port role */

  if (body->flags>>4 & 0x1) {
    stp_trace_more(" (learning)");
  } 
  if (body->flags>>5 & 0x1) {
    stp_trace_more(" (forwarding)");
  } 
  if (body->flags>>6 & 0x1) {
    stp_trace_more(" (agreement)");
  }
  if (body->flags>>7 & 0x1) {
    stp_trace_more(" (topology_change_ack)");
  }

  /* port roles: disabled, root, designated, alternate */
  role = body->flags>>2 & 0x3;
  switch (role) {
  case 0:
    stp_trace_more(" port_role: disabled");
    break;
  case 1:
    stp_trace_more(" port_role: root");
    break;
  case 2:
    stp_trace_more(" port_role: designated");
    break;
  case 3:
    stp_trace_more(" port_role: alternate");
    break;
  default:
    stp_trace_more(" port_role: weird: %d", role);
    assert(0);
  }
  stp_trace_more("\n");
  stp_trace_more("\tmessage_age %u max_age %u hello_time %u fwd_delay %u",
		 ntohs(body->message_age)/256, ntohs(body->max_age)/256,
		 ntohs(body->hello_time)/256, ntohs(body->forward_delay)/256);

}

void stp_trace_bpdu(const char *bpdu, const unsigned length, 
		    const bool xmitOrRecv,  const unsigned portNum)
{
  struct tx_tcn_bpdu *tcn = (struct tx_tcn_bpdu *)bpdu;
  assert(length >= sizeof(struct tx_tcn_bpdu));
  char *bt;

  stp_trace_begin();
  if (xmitOrRecv) {
    stp_trace_more("XMIT BPDU port %d", portNum);
  } else {
    stp_trace_more("RECV BPDU port %d", portNum);
  }

  switch (tcn->hdr.bpdu_type) {
  case BPDU_TOPO_CHANGE_TYPE:
    bt = "TOPOLOGY CHANGE";
    break;
  case BPDU_CONFIG_TYPE:
    bt = "CONFIG";
    assert(length >= sizeof(struct tx_stp_bpdu));
    break;
  case BPDU_RSTP:
    bt = "RSTP";
    assert(length >= sizeof(struct tx_rstp_bpdu));
    break;
  default:
    stp_trace_more("unknown bpdu type %d\n", tcn->hdr.bpdu_type);
    assert(0);
  }
  stp_trace_more(" %s (len %d)", bt, length);

  if (tcn->hdr.bpdu_type != BPDU_TOPO_CHANGE_TYPE) {
    /* casting to an rstp bpdu type, but format of topo change bpdu is
       the same at this level */
    dump_body(&(((struct tx_rstp_bpdu *)bpdu)->body));
  }
  stp_trace_end();
}
